
/* **************************************************************************************************** */
/* DISCOS Project																						*/
/* This code is under GNU General Public Licence (GPL).                                                 */
/*                                                                                                      */
/* Who                                 when           What                                              */
/* Andrea Orlati(aorlati@ira.inaf.it)  10/01/2017     Creation                                          */

#ifndef SCHEDULEREPORT_H_
#define SCHEDULEREPORT_H_

#include <IRA>
#include <list>

/*
 * This class is in charge of creating and managing the schedule report.
 * The report is a text file that contains some information regarding the execution of schedule:
 * 1) the schedule file name
 * 2) the corresponding generated log file
 * 3) the sequence of folders (corresponding to the created scans) generated by the schedule
 * 4) TBD, AOB
 * The class also creates a lock files that can be used by external application to understand if the system
 * is acquiring data or not.
 * This class is not thread safe.
 */

class CScheduleReport {
public:
	/*
	 * Object constructor.
	 * @param logPath path to the log files folder
	 * @param reportPath this is the path in the local filessystem where the schedule report will be saved. If
	 *        empty the schedule report is not created.
	 * @param backupPath this is the path in the local file system where the schedule files are stored for
	 *        archiving purposes
	 * @param lock name of the file to be created as lock file. If empty string the lock file is not handled.
	 */
	CScheduleReport(const IRA::CString logPath,const IRA::CString& reportPath,const IRA::CString& backupPath,
			const IRA::CString& lock);
	CScheduleReport();
	~CScheduleReport();
	const IRA::CString& getLastError() const { return m_lastError; }
	void addScheduleName(const IRA::CString& fullName);
	void addAuxScheduleFile(const IRA::CString& fullName);
	IRA::CString getLogFileName() const;
	IRA::CString getReportFileName() const;
	IRA::CString getBackupFolder() const;
	IRA::CString getPostFix() const { return m_currentPostfix; }
	IRA::CString getSchedule() const;
	bool isActive() const { return m_active; }
	
	/*
	 * Duplicate paths are excluded
	 */
	void addScanPath(const IRA::CString& scan);
	bool activate();
	bool deactivate();
private:
	typedef std::list<IRA::CString> TTnternalList;
	typedef std::list<IRA::CString>::iterator TInternalListIter;
	void generatePostfix();
	bool createLock();
	bool removeLock();
	bool backupSchedule();
	bool writeReport();
	/**
	 * removes the trailing '/', if present
	 */
	void removeTrailingSlash(IRA::CString& path);
	IRA::CString m_lastError;
	IRA::CString m_logFilePath;
	IRA::CString m_reportPath;
	IRA::CString m_backupPath;
	IRA::CString m_lockFile;
	IRA::CString m_scheduleBaseName;
	IRA::CString m_scheduleBasePath;
	IRA::CString m_scheduleExtension;
	IRA::CString m_currentPostfix;
	bool m_active;
	bool m_reportEnabled;
	bool m_lockFileEnabled;
	bool m_scheduleProvided;
	TTnternalList m_auxFiles;
	TTnternalList m_scanPaths;
};

#endif /* SCHEDULEREPORT_H_ */
