/*
 ************************************************************************
 *******************  CANADIAN ASTRONOMY DATA CENTRE  *******************
 **************  CENTRE CANADIEN DE DONNÉES ASTRONOMIQUES  **************
 *
 *  (c) 2014.                            (c) 2014.
 *  Government of Canada                 Gouvernement du Canada
 *  National Research Council            Conseil national de recherches
 *  Ottawa, Canada, K1A 0R6              Ottawa, Canada, K1A 0R6
 *  All rights reserved                  Tous droits réservés
 *
 *  NRC disclaims any warranties,        Le CNRC dénie toute garantie
 *  expressed, implied, or               énoncée, implicite ou légale,
 *  statutory, of any kind with          de quelque nature que ce
 *  respect to the software,             soit, concernant le logiciel,
 *  including without limitation         y compris sans restriction
 *  any warranty of merchantability      toute garantie de valeur
 *  or fitness for a particular          marchande ou de pertinence
 *  purpose. NRC shall not be            pour un usage particulier.
 *  liable in any event for any          Le CNRC ne pourra en aucun cas
 *  damages, whether direct or           être tenu responsable de tout
 *  indirect, special or general,        dommage, direct ou indirect,
 *  consequential or incidental,         particulier ou général,
 *  arising from the use of the          accessoire ou fortuit, résultant
 *  software.  Neither the name          de l'utilisation du logiciel. Ni
 *  of the National Research             le nom du Conseil National de
 *  Council of Canada nor the            Recherches du Canada ni les noms
 *  names of its contributors may        de ses  participants ne peuvent
 *  be used to endorse or promote        être utilisés pour approuver ou
 *  products derived from this           promouvoir les produits dérivés
 *  software without specific prior      de ce logiciel sans autorisation
 *  written permission.                  préalable et particulière
 *                                       par écrit.
 *
 *  This file is part of the             Ce fichier fait partie du projet
 *  OpenCADC project.                    OpenCADC.
 *
 *  OpenCADC is free software:           OpenCADC est un logiciel libre ;
 *  you can redistribute it and/or       vous pouvez le redistribuer ou le
 *  modify it under the terms of         modifier suivant les termes de
 *  the GNU Affero General Public        la “GNU Affero General Public
 *  License as published by the          License” telle que publiée
 *  Free Software Foundation,            par la Free Software Foundation
 *  either version 3 of the              : soit la version 3 de cette
 *  License, or (at your option)         licence, soit (à votre gré)
 *  any later version.                   toute version ultérieure.
 *
 *  OpenCADC is distributed in the       OpenCADC est distribué
 *  hope that it will be useful,         dans l’espoir qu’il vous
 *  but WITHOUT ANY WARRANTY;            sera utile, mais SANS AUCUNE
 *  without even the implied             GARANTIE : sans même la garantie
 *  warranty of MERCHANTABILITY          implicite de COMMERCIALISABILITÉ
 *  or FITNESS FOR A PARTICULAR          ni d’ADÉQUATION À UN OBJECTIF
 *  PURPOSE.  See the GNU Affero         PARTICULIER. Consultez la Licence
 *  General Public License for           Générale Publique GNU Affero
 *  more details.                        pour plus de détails.
 *
 *  You should have received             Vous devriez avoir reçu une
 *  a copy of the GNU Affero             copie de la Licence Générale
 *  General Public License along         Publique GNU Affero avec
 *  with OpenCADC.  If not, see          OpenCADC ; si ce n’est
 *  <http://www.gnu.org/licenses/>.      pas le cas, consultez :
 *                                       <http://www.gnu.org/licenses/>.
 *
 *  $Revision: 4 $
 *
 ************************************************************************
 */
package ca.nrc.cadc.ac.client;

import java.io.*;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.Principal;
import java.util.Iterator;
import java.util.Set;

import javax.security.auth.Subject;
import javax.security.auth.x500.X500Principal;

import ca.nrc.cadc.ac.*;
import ca.nrc.cadc.auth.HttpPrincipal;

import org.apache.log4j.Logger;

import ca.nrc.cadc.ac.xml.UserReader;
import ca.nrc.cadc.auth.AuthenticationUtil;
import ca.nrc.cadc.auth.CookiePrincipal;
import ca.nrc.cadc.auth.NumericPrincipal;
import ca.nrc.cadc.net.HttpDownload;


/**
 * Client class for performing user searching and user actions
 * with the access control web service.
 */
public class UserClient
{
    private static final Logger log = Logger.getLogger(UserClient.class);

    // socket factory to use when connecting
    private String baseURL;

    /**
     * Constructor.
     *
     * @param baseURL The URL of the supporting access control web service
     *                obtained from the registry.
     */
    public UserClient(final String baseURL)
            throws IllegalArgumentException
    {
        if (baseURL == null)
        {
            throw new IllegalArgumentException("baseURL is required");
        }
        try
        {
            new URL(baseURL);
        }
        catch (MalformedURLException e)
        {
            throw new IllegalArgumentException("URL is malformed: " +
                                               e.getMessage());
        }

        if (baseURL.endsWith("/"))
        {
            this.baseURL = baseURL.substring(0, baseURL.length() - 1);
        }
        else
        {
            this.baseURL = baseURL;
        }
    }

    /**
     * This method takes a subject with at least one valid principal, 
     * uses the ac user web service to get all the other 
     * associated principals which are then added to the subject.
     *
     * @param subject           The Subject to pull Princials for.
     */
    public void augmentSubject(Subject subject)
    {
    	Principal principal = this.getPrincipal(subject);
        URL url = this.getURL(principal);
    	log.debug("augmentSubject request to " + url.toString());    	
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        HttpDownload download = new HttpDownload(url, out);
        download.run();
     
        this.handleThrowable(download);
        this.augmentSubject(subject, this.getPrincipals(out));
    }
    
    protected void augmentSubject(Subject subject, Set<Principal> principals)
    {
        if (!principals.iterator().hasNext())
        {
        	String name = subject.getPrincipals().iterator().next().getName();
        	String msg = "No UserIdentity in LDAP server for principal: " + name;
        	throw new IllegalStateException(msg);
        }
        
    	for (Principal principal : principals)
    	{
    		if (principal instanceof HttpPrincipal)
    		{
    			subject.getPrincipals().add((HttpPrincipal)principal);
    		}
    		else if (principal instanceof X500Principal)
    		{
    			subject.getPrincipals().add((X500Principal)principal);
    		}
    		else if (principal instanceof NumericPrincipal)
    		{
    			subject.getPrincipals().add((NumericPrincipal)principal);
    		}
    		else if (principal instanceof CookiePrincipal)
    		{
    			subject.getPrincipals().add((CookiePrincipal)principal);
    		}
            else
            {
        		final String msg = "Subject has unsupported principal " +
        				principal.getName() + 
        				", not one of (X500, Cookie, HTTP or Cadc).";
		        throw new IllegalStateException(msg);
            }
    	}
    }
    
    protected Principal getPrincipal(final Subject subject)
    {
    	Set<Principal> principals = subject.getPrincipals();
    	Iterator<Principal> iterator = principals.iterator();
    	if (iterator.hasNext())
    	{
    		Principal principal = iterator.next();
    		log.debug("alinga-- UserClient.getPrincipal(): principal = " + principal);
    		if (iterator.hasNext())
    		{
    			Principal principal1 = iterator.next();
        		log.debug("alinga-- UserClient.getPrincipal(): principal1 = " + principal1);			
    			log.debug("alinga-- UserClient.getPrincipal(): number of principals = " + principals.size());
    			// Should only have one principal
        		final String msg = "Subject has more than one principal.";
		        throw new IllegalArgumentException(msg);
            }
            
            return principal;
    	}
    	else
    	{
    		final String msg = "Subject has no principal.";
    		throw new IllegalArgumentException(msg);
    	}
    }
    
    protected Set<Principal> getPrincipals(ByteArrayOutputStream out)
    {
    	try
    	{
	        String userXML = new String(out.toByteArray(), "UTF-8");
	        log.debug("userXML Input to getPrincipals(): " + userXML);
	        
	        User<Principal> user = new UserReader().read(userXML);
	        return user.getIdentities();
    	}
    	catch (Exception e)
    	{
    		throw new RuntimeException(e);
    	}
    }
    
    protected void handleThrowable(HttpDownload download)
    {
    	Throwable throwable = download.getThrowable();
        if (throwable != null)
        {
            log.debug("handleThrowable(): throwable (" + download
                    .getResponseCode() + ")", throwable);
            throw new IllegalStateException(throwable.getMessage());
        }
    }
    
    protected URL getURL(Principal principal)
    {
		try 
		{
		    String userID = principal.getName();
			URL url = new URL(this.baseURL + "/users/" + userID + 
					"?idType=" + this.getIdType(principal) + "&detail=identity");
			log.debug("getURL(): returned url ="
					+ ""
					+ " " + url.toString());
			return url;
		} 
		catch (MalformedURLException e)
		{
			throw new RuntimeException(e);
		}
    }
    
    protected String getIdType(Principal principal)
    {
		String idTypeStr = AuthenticationUtil.getPrincipalType(principal);
        if (idTypeStr == null)
        {
    		final String msg = "Subject has unsupported principal " +
    				principal.getName();
	        throw new IllegalArgumentException(msg);
        }
        
        return idTypeStr;
    }
}
