
import java.util.logging.Logger;
import java.util.Map;
import java.util.List;
import java.util.ArrayList;

import java.io.IOException;
import java.io.PrintWriter;
import java.nio.charset.Charset;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import vo.parameter.*;
import vo.error.*;

public class SearchServlet extends javax.servlet.http.HttpServlet
{
   private static final Logger         LOGGER   = Logger.getLogger(SearchServlet.class.getName());
   private static final SearchSettings settings = SearchSettings.getInstance("search.properties");

   final String RESPONSE_ENCODING   = "utf-8";
   final String DEFAULT_SKY_SYSTEM  = "ICRS";
   final String DEFAULT_SPEC_SYSTEM = "WAVE_Barycentric";
   final String DEFAULT_TIME_SYSTEM = "MJD_UTC";


   public void init() throws ServletException
   {
      LOGGER.config("DB: " + settings.dbConn.toString());
   }

   protected void doGet(HttpServletRequest request, HttpServletResponse response)
      throws ServletException, IOException, UnsupportedEncodingException
   {
      LOGGER.fine("trace");
      legacyLogEntry(request);

      try
      {
         Map<String, String[]> params = request.getParameterMap();

         QueryArgs qArgs = new QueryArgs();
         qArgs.pos  = Pos.parsePos(params, DEFAULT_SKY_SYSTEM);
         qArgs.band = Band.parseBand(params, DEFAULT_SPEC_SYSTEM);
         qArgs.time = Time.parseTime(params, DEFAULT_TIME_SYSTEM);
         qArgs.pol  = Pol.parsePol(params);

         qArgs.fov     = Interval.parseInterval(params, "FOV");
         qArgs.spatres = Interval.parseInterval(params, "SPATRES");
         qArgs.specrp  = Interval.parseInterval(params, "SPECRP");
         qArgs.exptime = Interval.parseInterval(params, "EXPTIME");
         qArgs.timeres = Interval.parseInterval(params, "TIMERES");

         qArgs.id = SingleStringParam.parseSingleStringParam(params, "ID");

         qArgs.collection = SingleStringParam.parseSingleStringParam(params, "COLLECTION");
         qArgs.facility   = SingleStringParam.parseSingleStringParam(params, "FACILITY");
         qArgs.instrument = SingleStringParam.parseSingleStringParam(params, "INSTRUMENT");
         qArgs.dptype     = SingleStringParam.parseSingleStringParam(params, "DPTYPE");

         qArgs.target = SingleStringParam.parseSingleStringParam(params, "TARGET");
         qArgs.format = SingleStringParam.parseSingleStringParam(params, "FORMAT");

         String str   = SingleStringParam.parseSingleStringParam(params, "CALIB");
         qArgs.calib  = (str == null) ? null : Integer.parseInt(str);
         if((qArgs.calib != null) && ((qArgs.calib < 0) || (qArgs.calib > 4)))
            throw new IllegalArgumentException("CALIB out of range: " + qArgs.calib + " but allowed is 0..4");

         str = SingleStringParam.parseSingleStringParam(params, "MAXREC");
         qArgs.maxrec  = (str == null) ? null : Integer.parseInt(str);
         if((qArgs.maxrec != null) && (qArgs.maxrec < 0))
            throw new IllegalArgumentException("MAXREC must not be negative: " + qArgs.maxrec);

         /* query Obscore table */

         DbPSearch dbps;
         synchronized(DbPSearch.class)
         {
            dbps = new DbPSearch(settings.dbConn);
         }
         String[] pubdidArr = dbps.queryOverlapingPubdid(qArgs);

         LOGGER.info("Found " + pubdidArr.length + " records");

         /* if filters installed response will be wrapped */

         if(response instanceof FormatResponseWrapper)
         {
            LOGGER.finest("response-type is FormatResponseWrapper");

            response.setContentType("text/plain");
            response.setCharacterEncoding(RESPONSE_ENCODING);

            /* collect all search description and set to wrapped response */

            FormatResponseWrapper responseWrapper = (FormatResponseWrapper) response;
            responseWrapper.setPubdidArr(pubdidArr);
            responseWrapper.setDBConn(settings.dbConn);
         }
         else
         {
            LOGGER.finest("response-type is HttpServletResponse");

            response.setContentType("text/plain");
            response.setCharacterEncoding(RESPONSE_ENCODING);

            PrintWriter writer = response.getWriter();
            for(String pubdid : pubdidArr)
            {
               writer.println(pubdid);
            }
            writer.close();
         }
      }
      catch(MultiValuedParamNotSupported ex)
      {
         LOGGER.warning("MultiValuedParamNotSupported: " + ex.getMessage());

         response.setStatus(HttpServletResponse.SC_BAD_REQUEST);
         response.setContentType("text/plain");

         PrintWriter writer = response.getWriter();
         Lib.doMultiValuedParamNotSupported(ex.getMessage(), writer);
         writer.close();
      }
      catch(IllegalArgumentException ex)
      {
         LOGGER.warning("IllegalArgumentException: " + ex.getMessage());

         response.setStatus(HttpServletResponse.SC_BAD_REQUEST);
         response.setContentType("text/plain");

         PrintWriter writer = response.getWriter();
         Lib.doUsageError(ex.getMessage(), writer);
         writer.close();
      }
      catch(Exception ex)
      {
         LOGGER.warning("Exception: " + ex.getMessage());
         ex.printStackTrace();

         response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
         response.setContentType("text/plain");

         PrintWriter writer = response.getWriter();
         Lib.doError(ex.toString(), writer);
         writer.close();
      }

      return;
   }


   private void legacyLogEntry(HttpServletRequest request) throws UnsupportedEncodingException
   {
      StringBuffer requestURL = request.getRequestURL();
      if (request.getQueryString() != null)
      {
         requestURL.append("?").append(request.getQueryString());
         String completeURL = requestURL.toString();
         String className = this.getClass().getSimpleName();
         LOGGER.finest(className + " vlkb req from: "
               + request.getRemoteAddr()
               + " doGet: " + URLDecoder.decode(completeURL, "UTF-8"));
      }
   }


}

