/*
 * Decompiled with CFR 0.152.
 */
package nom.tam.fits.compression.algorithm.quant;

import java.nio.ByteBuffer;
import java.nio.DoubleBuffer;
import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import nom.tam.fits.compression.algorithm.api.ICompressor;
import nom.tam.fits.compression.algorithm.quant.Quantize;
import nom.tam.fits.compression.algorithm.quant.QuantizeOption;

public class QuantizeProcessor {
    private static final double MAX_INT_AS_DOUBLE = 2.147483647E9;
    private static final long N_RESERVED_VALUES = 10L;
    private static final double ROUNDING_HALF = 0.5;
    private static final int ZERO_VALUE = -2147483646;
    private final boolean centerOnZero;
    private final PixelFilter pixelFilter;
    private double bScale;
    private double bZero;
    private Quantize quantize;
    protected final QuantizeOption quantizeOption;

    public QuantizeProcessor(QuantizeOption quantizeOption) {
        this.quantizeOption = quantizeOption;
        this.bScale = quantizeOption.getBScale();
        this.bZero = quantizeOption.getBZero();
        PixelFilter filter = null;
        boolean localCenterOnZero = quantizeOption.isCenterOnZero();
        if (quantizeOption.isDither2()) {
            filter = new DitherFilter(quantizeOption.getSeed());
            localCenterOnZero = true;
            quantizeOption.setCheckZero(true);
        } else {
            filter = quantizeOption.isDither() ? new DitherFilter(quantizeOption.getSeed()) : new BaseFilter();
        }
        if (quantizeOption.isCheckZero()) {
            filter = new ZeroFilter(filter);
        }
        if (quantizeOption.isCheckNull()) {
            final NullFilter nullFilter = new NullFilter(quantizeOption.getNullValue(), quantizeOption.getNullValueIndicator(), filter);
            filter = nullFilter;
            this.quantize = new Quantize(quantizeOption){

                @Override
                protected int findNextValidPixelWithNullCheck(int nx, Quantize.DoubleArrayPointer rowpix, int ii) {
                    while (ii < nx && nullFilter.isNull(rowpix.get(ii))) {
                        ++ii;
                    }
                    return ii;
                }

                @Override
                protected boolean isNull(double d) {
                    return nullFilter.isNull(d);
                }
            };
        } else {
            this.quantize = new Quantize(quantizeOption);
        }
        this.pixelFilter = filter;
        this.centerOnZero = localCenterOnZero;
    }

    private void calculateBZeroAndBscale() {
        this.bScale = this.quantizeOption.getBScale();
        if (Double.isNaN(this.quantizeOption.getBZero())) {
            this.bZero = this.zeroCenter();
            this.quantizeOption.setIntMinValue(this.nint((this.quantizeOption.getMinValue() - this.bZero) / this.bScale));
            this.quantizeOption.setIntMaxValue(this.nint((this.quantizeOption.getMaxValue() - this.bZero) / this.bScale));
            this.quantizeOption.setBZero(this.bZero);
        } else {
            this.bZero = this.quantizeOption.getBZero();
        }
    }

    public Quantize getQuantize() {
        return this.quantize;
    }

    private int nint(double x) {
        return x >= 0.0 ? (int)(x + 0.5) : (int)(x - 0.5);
    }

    public boolean quantize(double[] doubles, IntBuffer quants) {
        boolean success = this.quantize.quantize(doubles, this.quantizeOption.getTileWidth(), this.quantizeOption.getTileHeight());
        if (success) {
            this.calculateBZeroAndBscale();
            this.quantize(DoubleBuffer.wrap(doubles, 0, this.quantizeOption.getTileWidth() * this.quantizeOption.getTileHeight()), quants);
        }
        return success;
    }

    public void quantize(DoubleBuffer fdata, IntBuffer intData) {
        while (fdata.hasRemaining()) {
            intData.put(this.pixelFilter.toInt(fdata.get()));
            this.pixelFilter.nextPixel();
        }
    }

    public void unquantize(IntBuffer intData, DoubleBuffer fdata) {
        while (fdata.hasRemaining()) {
            fdata.put(this.pixelFilter.toDouble(intData.get()));
            this.pixelFilter.nextPixel();
        }
    }

    private double zeroCenter() {
        double evaluatedBZero;
        double minValue = this.quantizeOption.getMinValue();
        double maxValue = this.quantizeOption.getMaxValue();
        if (!this.quantizeOption.isCheckNull() && !this.centerOnZero) {
            if ((maxValue - minValue) / this.bScale < 2.147483637E9) {
                evaluatedBZero = minValue;
                long iqfactor = (long)(evaluatedBZero / this.bScale + 0.5);
                evaluatedBZero = (double)iqfactor * this.bScale;
            } else {
                evaluatedBZero = (minValue + maxValue) / 2.0;
            }
        } else {
            evaluatedBZero = minValue - this.bScale * -2.147483637E9;
        }
        return evaluatedBZero;
    }

    private class ZeroFilter
    extends PixelFilter {
        ZeroFilter(PixelFilter next) {
            super(next);
        }

        @Override
        protected double toDouble(int pixel) {
            if (pixel == -2147483646) {
                return 0.0;
            }
            return super.toDouble(pixel);
        }

        @Override
        protected int toInt(double pixel) {
            if (pixel == 0.0) {
                return -2147483646;
            }
            return super.toInt(pixel);
        }
    }

    private class PixelFilter {
        private final PixelFilter next;

        protected PixelFilter(PixelFilter next) {
            this.next = next;
        }

        protected void nextPixel() {
            this.next.nextPixel();
        }

        protected double toDouble(int pixel) {
            return this.next.toDouble(pixel);
        }

        protected int toInt(double pixel) {
            return this.next.toInt(pixel);
        }
    }

    private class NullFilter
    extends PixelFilter {
        private final double nullValue;
        private final boolean isNaN;
        private final int nullValueIndicator;

        NullFilter(double nullValue, int nullValueIndicator, PixelFilter next) {
            super(next);
            this.nullValue = nullValue;
            this.isNaN = Double.isNaN(this.nullValue);
            this.nullValueIndicator = nullValueIndicator;
        }

        public final boolean isNull(double pixel) {
            return this.isNaN ? Double.isNaN(pixel) : this.nullValue == pixel;
        }

        @Override
        protected double toDouble(int pixel) {
            if (pixel == this.nullValueIndicator) {
                return this.nullValue;
            }
            return super.toDouble(pixel);
        }

        @Override
        protected int toInt(double pixel) {
            if (this.isNull(pixel)) {
                return this.nullValueIndicator;
            }
            return super.toInt(pixel);
        }
    }

    public static class FloatQuantCompressor
    extends QuantizeProcessor
    implements ICompressor<FloatBuffer> {
        private final ICompressor<IntBuffer> postCompressor;

        public FloatQuantCompressor(QuantizeOption quantizeOption, ICompressor<IntBuffer> postCompressor) {
            super(quantizeOption);
            this.postCompressor = postCompressor;
        }

        @Override
        public boolean compress(FloatBuffer buffer, ByteBuffer compressed) {
            float[] floats = new float[this.quantizeOption.getTileHeight() * this.quantizeOption.getTileWidth()];
            double[] doubles = new double[this.quantizeOption.getTileHeight() * this.quantizeOption.getTileWidth()];
            buffer.get(floats);
            for (int index = 0; index < doubles.length; ++index) {
                doubles[index] = floats[index];
            }
            IntBuffer intData = IntBuffer.wrap(new int[this.quantizeOption.getTileHeight() * this.quantizeOption.getTileWidth()]);
            if (!this.quantize(doubles, intData)) {
                return false;
            }
            intData.rewind();
            this.postCompressor.compress(intData, compressed);
            return true;
        }

        @Override
        public void decompress(ByteBuffer compressed, FloatBuffer buffer) {
            IntBuffer intData = IntBuffer.wrap(new int[this.quantizeOption.getTileHeight() * this.quantizeOption.getTileWidth()]);
            this.postCompressor.decompress(compressed, intData);
            intData.rewind();
            double[] doubles = new double[this.quantizeOption.getTileHeight() * this.quantizeOption.getTileWidth()];
            DoubleBuffer doubleBuffer = DoubleBuffer.wrap(doubles);
            this.unquantize(intData, doubleBuffer);
            for (double d : doubles) {
                buffer.put((float)d);
            }
        }
    }

    public static class DoubleQuantCompressor
    extends QuantizeProcessor
    implements ICompressor<DoubleBuffer> {
        private final ICompressor<IntBuffer> postCompressor;

        public DoubleQuantCompressor(QuantizeOption quantizeOption, ICompressor<IntBuffer> postCompressor) {
            super(quantizeOption);
            this.postCompressor = postCompressor;
        }

        @Override
        public boolean compress(DoubleBuffer buffer, ByteBuffer compressed) {
            IntBuffer intData = IntBuffer.wrap(new int[this.quantizeOption.getTileHeight() * this.quantizeOption.getTileWidth()]);
            double[] doubles = new double[this.quantizeOption.getTileHeight() * this.quantizeOption.getTileWidth()];
            buffer.get(doubles);
            if (!this.quantize(doubles, intData)) {
                return false;
            }
            intData.rewind();
            this.postCompressor.compress(intData, compressed);
            return true;
        }

        @Override
        public void decompress(ByteBuffer compressed, DoubleBuffer buffer) {
            IntBuffer intData = IntBuffer.wrap(new int[this.quantizeOption.getTileHeight() * this.quantizeOption.getTileWidth()]);
            this.postCompressor.decompress(compressed, intData);
            intData.rewind();
            this.unquantize(intData, buffer);
        }
    }

    private class DitherFilter
    extends PixelFilter {
        private static final int LAST_RANDOM_VALUE = 1043618065;
        private static final double MAX_INT_AS_DOUBLE = 2.147483647E9;
        private static final int N_RANDOM = 10000;
        private static final int RANDOM_MULTIPLICATOR = 500;
        private static final double RANDOM_START_VALUE = 16807.0;
        private int iseed;
        private int nextRandom;
        private final double[] randomValues;

        DitherFilter(long seed) {
            super(null);
            this.iseed = 0;
            this.nextRandom = 0;
            this.randomValues = this.initRandoms();
            this.initialize(seed);
        }

        public void initialize(long ditherSeed) {
            this.iseed = (int)((ditherSeed - 1L) % 10000L);
            this.nextRandom = (int)(this.randomValues[this.iseed] * 500.0);
        }

        private double[] initRandoms() {
            double a = 16807.0;
            double m = 2.147483647E9;
            double[] randomValue = new double[10000];
            double seed = 1.0;
            for (int ii = 0; ii < 10000; ++ii) {
                double temp = a * seed;
                seed = temp - m * (double)((int)(temp / m));
                randomValue[ii] = seed / m;
            }
            if ((int)seed != 1043618065) {
                throw new IllegalArgumentException("randomValue generated incorrect random number sequence");
            }
            return randomValue;
        }

        @Override
        protected void nextPixel() {
            ++this.nextRandom;
            if (this.nextRandom == 10000) {
                ++this.iseed;
                if (this.iseed == 10000) {
                    this.iseed = 0;
                }
                this.nextRandom = (int)(this.randomValues[this.iseed] * 500.0);
            }
        }

        public double nextRandom() {
            return this.randomValues[this.nextRandom];
        }

        @Override
        protected double toDouble(int pixel) {
            return ((double)pixel - this.nextRandom() + 0.5) * QuantizeProcessor.this.bScale + QuantizeProcessor.this.bZero;
        }

        @Override
        protected int toInt(double pixel) {
            return QuantizeProcessor.this.nint((pixel - QuantizeProcessor.this.bZero) / QuantizeProcessor.this.bScale + this.nextRandom() - 0.5);
        }
    }

    private class BaseFilter
    extends PixelFilter {
        BaseFilter() {
            super(null);
        }

        @Override
        protected void nextPixel() {
        }

        @Override
        protected double toDouble(int pixel) {
            return ((double)pixel + 0.5) * QuantizeProcessor.this.bScale + QuantizeProcessor.this.bZero;
        }

        @Override
        protected int toInt(double pixel) {
            return QuantizeProcessor.this.nint((pixel - QuantizeProcessor.this.bZero) / QuantizeProcessor.this.bScale + 0.5);
        }
    }
}

