
/* NOTE originally was in search/output : designed for serializing search output xml */

import java.util.logging.Logger;

/* for Csv-loadSubsurveys */
import java.io.IOException;
import com.opencsv.*;
import com.opencsv.exceptions.*;
import java.io.FileReader;
import java.io.FileNotFoundException;
import java.util.Map;
import java.util.List;
import java.util.ArrayList;

class Subsurvey
{
   private static final Logger LOGGER = Logger.getLogger("Subsurvey");

   String description;
   String surveyname;
   String species;
   String transition;
   double rf; // rest frequency
   String rf_unit;
   String vel_unit;
   String storage_path;


   Subsurvey() {}
   Subsurvey(Subsurvey ss)
   {
    this.description = ss.description;;
    this.surveyname = ss.surveyname;
    this.species = ss.species;
    this.transition = ss.transition;
    this.rf = ss.rf; 
    this.rf_unit = ss.rf_unit;
    this.vel_unit = ss.vel_unit;
    this.storage_path = ss.storage_path;
   }

   String id() { return (this.surveyname + " " + this.species + " "  + this.transition); }

   boolean matches(String id) { return id.equals(this.id()); }




   static public Subsurvey findSubsurvey(Subsurvey[] dbSubsurveys, String subsurvey_id)
   {
      for(Subsurvey curr : dbSubsurveys)
      {
         if(curr.matches(subsurvey_id))
         {
            return curr;
         }
      }

      throw new AssertionError(subsurvey_id + " not found in surveys table");
   }




   static public FitsCard[] subsurveysFindCards(Subsurvey[] subsurveys, String subsurveyId)
   {
      if(subsurveys == null) return null;

      Subsurvey subsurvey = Subsurvey.findSubsurvey(subsurveys, subsurveyId);

      FitsCard[] keyRecords = FitsCard.convertToFitsCard(subsurvey.rf, subsurvey.vel_unit);

      return keyRecords;
   }




   public static Subsurvey[] loadSubsurveys(String csvFilename)
   {
      LOGGER.info("loadSubsurvey from: " + csvFilename);

      /* avoid access files-system if csv-filename not configured */
      if( (csvFilename == null) || ( (csvFilename != null) && (csvFilename.length() < 1) ) )
      {
         LOGGER.warning("csvFilename is null, metadata not loaded");
         return null;
      }

      try
      {
         CSVReaderHeaderAware csvReader = new CSVReaderHeaderAware(new FileReader(csvFilename));

         List<Subsurvey> subsurveyList = new ArrayList<>();
         Map<String, String> values;

         while ((values = csvReader.readMap()) != null)
         {
            Subsurvey subsurvey = new Subsurvey();

            subsurvey.description   = values.get("description");
            subsurvey.surveyname    = values.get("name");
            subsurvey.species       = values.get("species");
            subsurvey.transition    = values.get("transition");
            subsurvey.rf            = Double.parseDouble(values.get("rest_frequency"));
            subsurvey.rf_unit       = values.get("restf_fits_unit");
            subsurvey.vel_unit      = values.get("velocity_fits_unit");
            subsurvey.storage_path  = values.get("storage_path");

            subsurveyList.add(subsurvey);
         }

         return subsurveyList.toArray(new Subsurvey[0]);

      }
      catch(IOException ex) 
      {
         LOGGER.info("Error while loading " + csvFilename + " -> " + ex.getMessage());
         return null;
         //throw new IllegalStateException("Error while loading " + csvFilename + " file", ex);
      }
      catch(CsvValidationException ex) 
      {
         LOGGER.info("Error while reading " + csvFilename + " -> " + ex.getMessage());
         return null;
         //throw new IllegalStateException("Error while reading " + csvFilename + " file", ex);
      }
   }

}

