/* vim:set ft=c ts=2 sw=2 sts=2 et cindent: */

#include "rpc_amqp_utils.hpp"

#include <string>
#include <stdexcept>

#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <ctype.h>
#include <syslog.h>

#include <amqp.h>
#include <amqp_framing.h>


void die(const char *fmt, ...)
{
  const size_t BUFF_SIZE = 256;
  char buff[BUFF_SIZE];
  buff[BUFF_SIZE] = 0;
  va_list ap;
  va_start(ap, fmt);
  vsnprintf(buff, BUFF_SIZE-1, fmt, ap);
  va_end(ap);
  throw std::runtime_error(buff);
}

void die_on_error(int x, char const *context)
{
  if (x < 0) {
    die("%s: %s\n", context, amqp_error_string2(x));
  }
}

// If synchronous AMQP API methods fail (return NULL) use this.
// Get the last global amqp_rpc_reply (per-connection-global amqp_rpc_reply_t)
// normal operation: AMQP_RESPONSE_NORMAL -> RPC completed successfully.
// error: AMQP_RESPONSE_SERVER_EXCEPTION (conn closed, channel closed, library exception)
void die_on_amqp_error(amqp_rpc_reply_t x, char const *context)
{
  switch (x.reply_type) {

    case AMQP_RESPONSE_NORMAL:
      return;


    case AMQP_RESPONSE_NONE:
      die("%s: missing RPC reply type!\n", context);
      break;


    case AMQP_RESPONSE_LIBRARY_EXCEPTION:
      die("%s: %s\n", context, amqp_error_string2(x.library_error));
      break;


    case AMQP_RESPONSE_SERVER_EXCEPTION:
      switch (x.reply.id) {
        case AMQP_CONNECTION_CLOSE_METHOD: {
                                             amqp_connection_close_t *m = (amqp_connection_close_t *) x.reply.decoded;
                                             die("%s: server connection error %uh, message: %.*s\n",
                                                 context,
                                                 m->reply_code,
                                                 (int) m->reply_text.len, (char *) m->reply_text.bytes);
                                             break;
                                           }
        case AMQP_CHANNEL_CLOSE_METHOD: {
                                          amqp_channel_close_t *m = (amqp_channel_close_t *) x.reply.decoded;
                                          die("%s: server channel error %uh, message: %.*s\n",
                                              context,
                                              m->reply_code,
                                              (int) m->reply_text.len, (char *) m->reply_text.bytes);
                                          break;
                                        }
        default:
                                        die("%s: unknown server error, method id 0x%08X\n", context, x.reply.id);
                                        break;
      }
      break;


    default:
      die("%s: unknown server error, reply_type 0x%08X\n", context, x.reply_type);
      break;
  }
  // code never reaches here
}


void throw_ex_on_amqp_error(amqp_connection_state_t conn, amqp_rpc_reply_t x, char const *context)
{
  const size_t buff_size = 255;
  char buff[buff_size+1];

  buff[buff_size+1] = 0;


  switch (x.reply_type) {

    case AMQP_RESPONSE_NORMAL:
      return;


    case AMQP_RESPONSE_NONE:
      snprintf(buff, buff_size,"%s: missing RPC reply type!\n", context);
      break;


    case AMQP_RESPONSE_LIBRARY_EXCEPTION:
      snprintf(buff, buff_size,"%s: %s\n", context, amqp_error_string2(x.library_error));
      break;


    case AMQP_RESPONSE_SERVER_EXCEPTION:
      switch (x.reply.id) {
        case AMQP_CONNECTION_CLOSE_METHOD: {
                                             amqp_connection_close_t *m = (amqp_connection_close_t *) x.reply.decoded;
                                             snprintf(buff, buff_size,"%s: server connection error %uh, message: %.*s\n",
                                                 context,
                                                 m->reply_code,
                                                 (int) m->reply_text.len, (char *) m->reply_text.bytes);
                                             break;
                                           }
        case AMQP_CHANNEL_CLOSE_METHOD: {
                                          amqp_channel_close_t *m = (amqp_channel_close_t *) x.reply.decoded;
                                          snprintf(buff, buff_size,"%s: server channel error %uh, message: %.*s\n",
                                              context,
                                              m->reply_code,
                                              (int) m->reply_text.len, (char *) m->reply_text.bytes);
                                          break;
                                        }
        default:
                                        snprintf(buff, buff_size,"%s: unknown server error, method id 0x%08X\n", context, x.reply.id);
                                        break;
      }
      break;


    default:
      snprintf(buff, buff_size,"%s: unknown server error, reply_type 0x%08X\n", context, x.reply_type);
      break;
  }

  if (AMQP_RESPONSE_NORMAL != x.reply_type)
  {
    die_on_error(amqp_destroy_connection(conn), "Ending connection before throw runtime exception");
    throw std::runtime_error(buff);
  }
}







// output


static void dump_row(long count, int numinrow, int *chs)
{
  int i;

  printf("%08lX:", count - numinrow);

  if (numinrow > 0) {
    for (i = 0; i < numinrow; i++) {
      if (i == 8) {
        printf(" :");
      }
      printf(" %02X", chs[i]);
    }
    for (i = numinrow; i < 16; i++) {
      if (i == 8) {
        printf(" :");
      }
      printf("   ");
    }
    printf("  ");
    for (i = 0; i < numinrow; i++) {
      if (isprint(chs[i])) {
        printf("%c", chs[i]);
      } else {
        printf(".");
      }
    }
  }
  printf("\n");
}

static int rows_eq(int *a, int *b)
{
  int i;

  for (i=0; i<16; i++)
    if (a[i] != b[i]) {
      return 0;
    }

  return 1;
}

void amqp_dump(void const *buffer, size_t len)
{
  unsigned char *buf = (unsigned char *) buffer;
  long count = 0;
  int numinrow = 0;
  int chs[16];
  int oldchs[16] = {0};
  int showed_dots = 0;
  size_t i;

  for (i = 0; i < len; i++) {
    int ch = buf[i];

    if (numinrow == 16) {
      int j;

      if (rows_eq(oldchs, chs)) {
        if (!showed_dots) {
          showed_dots = 1;
          printf("          .. .. .. .. .. .. .. .. : .. .. .. .. .. .. .. ..\n");
        }
      } else {
        showed_dots = 0;
        dump_row(count, numinrow, chs);
      }

      for (j=0; j<16; j++) {
        oldchs[j] = chs[j];
      }

      numinrow = 0;
    }

    count++;
    chs[numinrow++] = ch;
  }

  dump_row(count, numinrow, chs);

  if (numinrow != 0) {
    printf("%08lX:\n", count);
  }
}
