/***********************************************************************
*_Title mocxtrack.h - MOC Cross Track Summing table maps
*_Args NONE

*_Descr This include file defines mapping of MGS MOC WA output pixels
*       to hardware pixels.  It is taken from the "Software Interface
*       Specification Narrow Angle and Wide Angle Standard Data Products"
*       publication, September 1999, formatted April 7, 2000, Appendix F.
*       This document was provided by Malin Space Science Systems, Inc.,
*       (MSSS) and specifies how on-board MOC WA varable summing relates
*       to output pixel generation.  These data are critical in proper
*       radiometric calibration and geometric rectifications for these
*       special WA cross track summing modes.

*       Note that it is very important to understand that the values
*       in the starting and ending pixel are relative to the starting
*       hardware detector pixel as specified in the FIRST_LINE_SAMPLE
*       ISIS label keyword (EDIT_MODE_ID in MGS PDS labels).  Thus,
*       for example, the actual hardware pixel or pixels used to
*       generate the first output pixel is:
*
*        starting_hw_pixel = (FIRST_LINE_SAMPLE - 1) + table[0].starting_pixel
*        ending_hw_pixel   = (FIRST_LINE_SAMPLE - 1) + table[0].ending_pixel
*
*       where "table" is either "mode27_table" or "mode13_table" depending
*       on the value of CROSSTRACK_SUMMING in the ISIS label, 27 or 13,
*       respectively.  Generally, these images will always have the
*       indicated number of samples per line for each mode.
*/

#ifndef MOCXTRACK_H
#define MOCXTRACK_H

/******************************************************************
  Define the table structure for WA crosstrack summing modes
*******************************************************************/

/// @cond INTERNAL
typedef struct moc_xts_table {
  int output_pixel;        // Output pixel
  int n_pixels;            // Number of detectors in sum
  int starting_pixel;      // Starting hardware detector
  int ending_pixel;        // Ending hardware detector
  double center_pixel;     // Center hardware detector
  double pixel_difference; // Difference between current
  // center_pixel and previous
  // center_pixel
} MOC_XTS_TABLE;
/// @endcond

/********************************************************************/
/* 10.  Appendix F: global map crosstrack summing tables            */
/*                                                                  */
/* These tables map output pixel  in  the  global  map  to  the     */
/* number of input pixels summed and the offset of these pixels     */
/* from the starting pixel.  Using the EDIT_MODE_ID  to  deter-     */
/* mine  the  start pixel of the image, the mapping to and from     */
/* detector pixel and global map pixel is determined.               */
/*                                                                  */
/* The tables have four columns: the output pixel number start-     */
/* ing from 0, the summing for that pixel, and the starting and     */
/* ending hardware pixel offsets relative to EDIT_MODE_ID  that     */
/* are summed to form the output pixel.                             */
/*                                                                  */
/* In ISIS, the EDIT_MODE_ID is actually the FIRST_LINE_SAMPLE      */
/* plus 1 as it is a 1-based entity not 0-based as the MGS PDS      */
/* keyword EDIT_MODE_ID is.  Note that the tables are 0-based       */
/* indicies.                                                        */
/********************************************************************/

/* 7.5 km/pixel nominal resolution; output width 384.  This is      */
/* indicated by CROSSTRACK_SUMMING = 27 in the labels.              */

#define MOC_XTS_MODE_27                  384
static MOC_XTS_TABLE mode27_table[MOC_XTS_MODE_27] = {
  {    0,    1,    0,    0,       0.0,       1.0 },
  {    1,    1,    1,    1,       1.0,       1.0 },
  {    2,    1,    2,    2,       2.0,       1.0 },
  {    3,    1,    3,    3,       3.0,       1.0 },
  {    4,    1,    4,    4,       4.0,       1.0 },
  {    5,    1,    5,    5,       5.0,       1.0 },
  {    6,    1,    6,    6,       6.0,       1.0 },
  {    7,    1,    7,    7,       7.0,       1.0 },
  {    8,    1,    8,    8,       8.0,       1.0 },
  {    9,    1,    9,    9,       9.0,       1.0 },
  {   10,    1,   10,   10,      10.0,       1.0 },
  {   11,    1,   11,   11,      11.0,       1.0 },
  {   12,    1,   12,   12,      12.0,       1.0 },
  {   13,    1,   13,   13,      13.0,       1.0 },
  {   14,    1,   14,   14,      14.0,       1.0 },
  {   15,    1,   15,   15,      15.0,       1.0 },
  {   16,    1,   16,   16,      16.0,       1.0 },
  {   17,    1,   17,   17,      17.0,       1.0 },
  {   18,    1,   18,   18,      18.0,       1.0 },
  {   19,    1,   19,   19,      19.0,       1.0 },
  {   20,    1,   20,   20,      20.0,       1.0 },
  {   21,    1,   21,   21,      21.0,       1.0 },
  {   22,    1,   22,   22,      22.0,       1.0 },
  {   23,    1,   23,   23,      23.0,       1.0 },
  {   24,    1,   24,   24,      24.0,       1.0 },
  {   25,    1,   25,   25,      25.0,       1.0 },
  {   26,    1,   26,   26,      26.0,       1.0 },
  {   27,    1,   27,   27,      27.0,       1.0 },
  {   28,    1,   28,   28,      28.0,       1.0 },
  {   29,    1,   29,   29,      29.0,       1.0 },
  {   30,    1,   30,   30,      30.0,       1.0 },
  {   31,    1,   31,   31,      31.0,       1.0 },
  {   32,    1,   32,   32,      32.0,       1.0 },
  {   33,    1,   33,   33,      33.0,       1.0 },
  {   34,    1,   34,   34,      34.0,       1.0 },
  {   35,    1,   35,   35,      35.0,       1.0 },
  {   36,    1,   36,   36,      36.0,       1.0 },
  {   37,    1,   37,   37,      37.0,       1.0 },
  {   38,    1,   38,   38,      38.0,       1.0 },
  {   39,    1,   39,   39,      39.0,       1.0 },
  {   40,    1,   40,   40,      40.0,       1.0 },
  {   41,    1,   41,   41,      41.0,       1.0 },
  {   42,    1,   42,   42,      42.0,       1.0 },
  {   43,    1,   43,   43,      43.0,       1.0 },
  {   44,    1,   44,   44,      44.0,       1.0 },
  {   45,    1,   45,   45,      45.0,       1.0 },
  {   46,    1,   46,   46,      46.0,       1.0 },
  {   47,    1,   47,   47,      47.0,       1.0 },
  {   48,    1,   48,   48,      48.0,       1.0 },
  {   49,    2,   49,   50,      49.5,       1.5 },
  {   50,    2,   51,   52,      51.5,       2.0 },
  {   51,    2,   53,   54,      53.5,       2.0 },
  {   52,    2,   55,   56,      55.5,       2.0 },
  {   53,    2,   57,   58,      57.5,       2.0 },
  {   54,    2,   59,   60,      59.5,       2.0 },
  {   55,    2,   61,   62,      61.5,       2.0 },
  {   56,    2,   63,   64,      63.5,       2.0 },
  {   57,    2,   65,   66,      65.5,       2.0 },
  {   58,    2,   67,   68,      67.5,       2.0 },
  {   59,    2,   69,   70,      69.5,       2.0 },
  {   60,    2,   71,   72,      71.5,       2.0 },
  {   61,    2,   73,   74,      73.5,       2.0 },
  {   62,    2,   75,   76,      75.5,       2.0 },
  {   63,    2,   77,   78,      77.5,       2.0 },
  {   64,    2,   79,   80,      79.5,       2.0 },
  {   65,    2,   81,   82,      81.5,       2.0 },
  {   66,    2,   83,   84,      83.5,       2.0 },
  {   67,    2,   85,   86,      85.5,       2.0 },
  {   68,    2,   87,   88,      87.5,       2.0 },
  {   69,    2,   89,   90,      89.5,       2.0 },
  {   70,    2,   91,   92,      91.5,       2.0 },
  {   71,    2,   93,   94,      93.5,       2.0 },
  {   72,    3,   95,   97,      96.0,       2.5 },
  {   73,    3,   98,  100,      99.0,       3.0 },
  {   74,    3,  101,  103,     102.0,       3.0 },
  {   75,    3,  104,  106,     105.0,       3.0 },
  {   76,    3,  107,  109,     108.0,       3.0 },
  {   77,    3,  110,  112,     111.0,       3.0 },
  {   78,    3,  113,  115,     114.0,       3.0 },
  {   79,    3,  116,  118,     117.0,       3.0 },
  {   80,    3,  119,  121,     120.0,       3.0 },
  {   81,    3,  122,  124,     123.0,       3.0 },
  {   82,    3,  125,  127,     126.0,       3.0 },
  {   83,    3,  128,  130,     129.0,       3.0 },
  {   84,    3,  131,  133,     132.0,       3.0 },
  {   85,    3,  134,  136,     135.0,       3.0 },
  {   86,    3,  137,  139,     138.0,       3.0 },
  {   87,    3,  140,  142,     141.0,       3.0 },
  {   88,    4,  143,  146,     144.5,       3.5 },
  {   89,    4,  147,  150,     148.5,       4.0 },
  {   90,    4,  151,  154,     152.5,       4.0 },
  {   91,    4,  155,  158,     156.5,       4.0 },
  {   92,    4,  159,  162,     160.5,       4.0 },
  {   93,    4,  163,  166,     164.5,       4.0 },
  {   94,    4,  167,  170,     168.5,       4.0 },
  {   95,    4,  171,  174,     172.5,       4.0 },
  {   96,    4,  175,  178,     176.5,       4.0 },
  {   97,    4,  179,  182,     180.5,       4.0 },
  {   98,    4,  183,  186,     184.5,       4.0 },
  {   99,    4,  187,  190,     188.5,       4.0 },
  {  100,    5,  191,  195,     193.0,       4.5 },
  {  101,    5,  196,  200,     198.0,       5.0 },
  {  102,    5,  201,  205,     203.0,       5.0 },
  {  103,    5,  206,  210,     208.0,       5.0 },
  {  104,    5,  211,  215,     213.0,       5.0 },
  {  105,    5,  216,  220,     218.0,       5.0 },
  {  106,    5,  221,  225,     223.0,       5.0 },
  {  107,    5,  226,  230,     228.0,       5.0 },
  {  108,    5,  231,  235,     233.0,       5.0 },
  {  109,    6,  236,  241,     238.5,       5.5 },
  {  110,    6,  242,  247,     244.5,       6.0 },
  {  111,    6,  248,  253,     250.5,       6.0 },
  {  112,    6,  254,  259,     256.5,       6.0 },
  {  113,    7,  260,  266,     263.0,       6.5 },
  {  114,    7,  267,  273,     270.0,       7.0 },
  {  115,    7,  274,  280,     277.0,       7.0 },
  {  116,    7,  281,  287,     284.0,       7.0 },
  {  117,    7,  288,  294,     291.0,       7.0 },
  {  118,    7,  295,  301,     298.0,       7.0 },
  {  119,    7,  302,  308,     305.0,       7.0 },
  {  120,    8,  309,  316,     312.5,       7.5 },
  {  121,    8,  317,  324,     320.5,       8.0 },
  {  122,    8,  325,  332,     328.5,       8.0 },
  {  123,    8,  333,  340,     336.5,       8.0 },
  {  124,    8,  341,  348,     344.5,       8.0 },
  {  125,    8,  349,  356,     352.5,       8.0 },
  {  126,    9,  357,  365,     361.0,       8.5 },
  {  127,    9,  366,  374,     370.0,       9.0 },
  {  128,    9,  375,  383,     379.0,       9.0 },
  {  129,    9,  384,  392,     388.0,       9.0 },
  {  130,    9,  393,  401,     397.0,       9.0 },
  {  131,   10,  402,  411,     406.5,       9.5 },
  {  132,   10,  412,  421,     416.5,      10.0 },
  {  133,   10,  422,  431,     426.5,      10.0 },
  {  134,   10,  432,  441,     436.5,      10.0 },
  {  135,   10,  442,  451,     446.5,      10.0 },
  {  136,   11,  452,  462,     457.0,      10.5 },
  {  137,   11,  463,  473,     468.0,      11.0 },
  {  138,   11,  474,  484,     479.0,      11.0 },
  {  139,   11,  485,  495,     490.0,      11.0 },
  {  140,   12,  496,  507,     501.5,      11.5 },
  {  141,   13,  508,  520,     514.0,      12.5 },
  {  142,   13,  521,  533,     527.0,      13.0 },
  {  143,   13,  534,  546,     540.0,      13.0 },
  {  144,   13,  547,  559,     553.0,      13.0 },
  {  145,   14,  560,  573,     566.5,      13.5 },
  {  146,   14,  574,  587,     580.5,      14.0 },
  {  147,   14,  588,  601,     594.5,      14.0 },
  {  148,   14,  602,  615,     608.5,      14.0 },
  {  149,   15,  616,  630,     623.0,      14.5 },
  {  150,   15,  631,  645,     638.0,      15.0 },
  {  151,   15,  646,  660,     653.0,      15.0 },
  {  152,   16,  661,  676,     668.5,      15.5 },
  {  153,   16,  677,  692,     684.5,      16.0 },
  {  154,   16,  693,  708,     700.5,      16.0 },
  {  155,   17,  709,  725,     717.0,      16.5 },
  {  156,   17,  726,  742,     734.0,      17.0 },
  {  157,   18,  743,  760,     751.5,      17.5 },
  {  158,   18,  761,  778,     769.5,      18.0 },
  {  159,   19,  779,  797,     788.0,      18.5 },
  {  160,   19,  798,  816,     807.0,      19.0 },
  {  161,   19,  817,  835,     826.0,      19.0 },
  {  162,   20,  836,  855,     845.5,      19.5 },
  {  163,   20,  856,  875,     865.5,      20.0 },
  {  164,   20,  876,  895,     885.5,      20.0 },
  {  165,   21,  896,  916,     906.0,      20.5 },
  {  166,   21,  917,  937,     927.0,      21.0 },
  {  167,   21,  938,  958,     948.0,      21.0 },
  {  168,   22,  959,  980,     969.5,      21.5 },
  {  169,   22,  981, 1002,     991.5,      22.0 },
  {  170,   23, 1003, 1025,    1014.0,      22.5 },
  {  171,   24, 1026, 1049,    1037.5,      23.5 },
  {  172,   24, 1050, 1073,    1061.5,      24.0 },
  {  173,   24, 1074, 1097,    1085.5,      24.0 },
  {  174,   25, 1098, 1122,    1110.0,      24.5 },
  {  175,   25, 1123, 1147,    1135.0,      25.0 },
  {  176,   25, 1148, 1172,    1160.0,      25.0 },
  {  177,   25, 1173, 1197,    1185.0,      25.0 },
  {  178,   26, 1198, 1223,    1210.5,      25.5 },
  {  179,   26, 1224, 1249,    1236.5,      26.0 },
  {  180,   26, 1250, 1275,    1262.5,      26.0 },
  {  181,   26, 1276, 1301,    1288.5,      26.0 },
  {  182,   26, 1302, 1327,    1314.5,      26.0 },
  {  183,   27, 1328, 1354,    1341.0,      26.5 },
  {  184,   27, 1355, 1381,    1368.0,      27.0 },
  {  185,   27, 1382, 1408,    1395.0,      27.0 },
  {  186,   27, 1409, 1435,    1422.0,      27.0 },
  {  187,   27, 1436, 1462,    1449.0,      27.0 },
  {  188,   27, 1463, 1489,    1476.0,      27.0 },
  {  189,   27, 1490, 1516,    1503.0,      27.0 },
  {  190,   27, 1517, 1543,    1530.0,      27.0 },
  {  191,   27, 1544, 1570,    1557.0,      27.0 },
  {  192,   27, 1571, 1597,    1584.0,      27.0 },
  {  193,   27, 1598, 1624,    1611.0,      27.0 },
  {  194,   27, 1625, 1651,    1638.0,      27.0 },
  {  195,   27, 1652, 1678,    1665.0,      27.0 },
  {  196,   27, 1679, 1705,    1692.0,      27.0 },
  {  197,   27, 1706, 1732,    1719.0,      27.0 },
  {  198,   27, 1733, 1759,    1746.0,      27.0 },
  {  199,   27, 1760, 1786,    1773.0,      27.0 },
  {  200,   27, 1787, 1813,    1800.0,      27.0 },
  {  201,   26, 1814, 1839,    1826.5,      26.5 },
  {  202,   26, 1840, 1865,    1852.5,      26.0 },
  {  203,   26, 1866, 1891,    1878.5,      26.0 },
  {  204,   26, 1892, 1917,    1904.5,      26.0 },
  {  205,   26, 1918, 1943,    1930.5,      26.0 },
  {  206,   25, 1944, 1968,    1956.0,      25.5 },
  {  207,   25, 1969, 1993,    1981.0,      25.0 },
  {  208,   25, 1994, 2018,    2006.0,      25.0 },
  {  209,   25, 2019, 2043,    2031.0,      25.0 },
  {  210,   24, 2044, 2067,    2055.5,      24.5 },
  {  211,   24, 2068, 2091,    2079.5,      24.0 },
  {  212,   24, 2092, 2115,    2103.5,      24.0 },
  {  213,   23, 2116, 2138,    2127.0,      23.5 },
  {  214,   22, 2139, 2160,    2149.5,      22.5 },
  {  215,   22, 2161, 2182,    2171.5,      22.0 },
  {  216,   21, 2183, 2203,    2193.0,      21.5 },
  {  217,   21, 2204, 2224,    2214.0,      21.0 },
  {  218,   21, 2225, 2245,    2235.0,      21.0 },
  {  219,   20, 2246, 2265,    2255.5,      20.5 },
  {  220,   20, 2266, 2285,    2275.5,      20.0 },
  {  221,   20, 2286, 2305,    2295.5,      20.0 },
  {  222,   19, 2306, 2324,    2315.0,      19.5 },
  {  223,   19, 2325, 2343,    2334.0,      19.0 },
  {  224,   19, 2344, 2362,    2353.0,      19.0 },
  {  225,   18, 2363, 2380,    2371.5,      18.5 },
  {  226,   18, 2381, 2398,    2389.5,      18.0 },
  {  227,   17, 2399, 2415,    2407.0,      17.5 },
  {  228,   17, 2416, 2432,    2424.0,      17.0 },
  {  229,   16, 2433, 2448,    2440.5,      16.5 },
  {  230,   16, 2449, 2464,    2456.5,      16.0 },
  {  231,   16, 2465, 2480,    2472.5,      16.0 },
  {  232,   15, 2481, 2495,    2488.0,      15.5 },
  {  233,   15, 2496, 2510,    2503.0,      15.0 },
  {  234,   15, 2511, 2525,    2518.0,      15.0 },
  {  235,   14, 2526, 2539,    2532.5,      14.5 },
  {  236,   14, 2540, 2553,    2546.5,      14.0 },
  {  237,   14, 2554, 2567,    2560.5,      14.0 },
  {  238,   14, 2568, 2581,    2574.5,      14.0 },
  {  239,   13, 2582, 2594,    2588.0,      13.5 },
  {  240,   13, 2595, 2607,    2601.0,      13.0 },
  {  241,   13, 2608, 2620,    2614.0,      13.0 },
  {  242,   13, 2621, 2633,    2627.0,      13.0 },
  {  243,   12, 2634, 2645,    2639.5,      12.5 },
  {  244,   11, 2646, 2656,    2651.0,      11.5 },
  {  245,   11, 2657, 2667,    2662.0,      11.0 },
  {  246,   11, 2668, 2678,    2673.0,      11.0 },
  {  247,   11, 2679, 2689,    2684.0,      11.0 },
  {  248,   10, 2690, 2699,    2694.5,      10.5 },
  {  249,   10, 2700, 2709,    2704.5,      10.0 },
  {  250,   10, 2710, 2719,    2714.5,      10.0 },
  {  251,   10, 2720, 2729,    2724.5,      10.0 },
  {  252,   10, 2730, 2739,    2734.5,      10.0 },
  {  253,    9, 2740, 2748,    2744.0,       9.5 },
  {  254,    9, 2749, 2757,    2753.0,       9.0 },
  {  255,    9, 2758, 2766,    2762.0,       9.0 },
  {  256,    9, 2767, 2775,    2771.0,       9.0 },
  {  257,    9, 2776, 2784,    2780.0,       9.0 },
  {  258,    8, 2785, 2792,    2788.5,       8.5 },
  {  259,    8, 2793, 2800,    2796.5,       8.0 },
  {  260,    8, 2801, 2808,    2804.5,       8.0 },
  {  261,    8, 2809, 2816,    2812.5,       8.0 },
  {  262,    8, 2817, 2824,    2820.5,       8.0 },
  {  263,    8, 2825, 2832,    2828.5,       8.0 },
  {  264,    7, 2833, 2839,    2836.0,       7.5 },
  {  265,    7, 2840, 2846,    2843.0,       7.0 },
  {  266,    7, 2847, 2853,    2850.0,       7.0 },
  {  267,    7, 2854, 2860,    2857.0,       7.0 },
  {  268,    7, 2861, 2867,    2864.0,       7.0 },
  {  269,    7, 2868, 2874,    2871.0,       7.0 },
  {  270,    7, 2875, 2881,    2878.0,       7.0 },
  {  271,    6, 2882, 2887,    2884.5,       6.5 },
  {  272,    6, 2888, 2893,    2890.5,       6.0 },
  {  273,    6, 2894, 2899,    2896.5,       6.0 },
  {  274,    6, 2900, 2905,    2902.5,       6.0 },
  {  275,    5, 2906, 2910,    2908.0,       5.5 },
  {  276,    5, 2911, 2915,    2913.0,       5.0 },
  {  277,    5, 2916, 2920,    2918.0,       5.0 },
  {  278,    5, 2921, 2925,    2923.0,       5.0 },
  {  279,    5, 2926, 2930,    2928.0,       5.0 },
  {  280,    5, 2931, 2935,    2933.0,       5.0 },
  {  281,    5, 2936, 2940,    2938.0,       5.0 },
  {  282,    5, 2941, 2945,    2943.0,       5.0 },
  {  283,    5, 2946, 2950,    2948.0,       5.0 },
  {  284,    4, 2951, 2954,    2952.5,       4.5 },
  {  285,    4, 2955, 2958,    2956.5,       4.0 },
  {  286,    4, 2959, 2962,    2960.5,       4.0 },
  {  287,    4, 2963, 2966,    2964.5,       4.0 },
  {  288,    4, 2967, 2970,    2968.5,       4.0 },
  {  289,    4, 2971, 2974,    2972.5,       4.0 },
  {  290,    4, 2975, 2978,    2976.5,       4.0 },
  {  291,    4, 2979, 2982,    2980.5,       4.0 },
  {  292,    4, 2983, 2986,    2984.5,       4.0 },
  {  293,    4, 2987, 2990,    2988.5,       4.0 },
  {  294,    4, 2991, 2994,    2992.5,       4.0 },
  {  295,    4, 2995, 2998,    2996.5,       4.0 },
  {  296,    3, 2999, 3001,    3000.0,       3.5 },
  {  297,    3, 3002, 3004,    3003.0,       3.0 },
  {  298,    3, 3005, 3007,    3006.0,       3.0 },
  {  299,    3, 3008, 3010,    3009.0,       3.0 },
  {  300,    3, 3011, 3013,    3012.0,       3.0 },
  {  301,    3, 3014, 3016,    3015.0,       3.0 },
  {  302,    3, 3017, 3019,    3018.0,       3.0 },
  {  303,    3, 3020, 3022,    3021.0,       3.0 },
  {  304,    3, 3023, 3025,    3024.0,       3.0 },
  {  305,    3, 3026, 3028,    3027.0,       3.0 },
  {  306,    3, 3029, 3031,    3030.0,       3.0 },
  {  307,    3, 3032, 3034,    3033.0,       3.0 },
  {  308,    3, 3035, 3037,    3036.0,       3.0 },
  {  309,    3, 3038, 3040,    3039.0,       3.0 },
  {  310,    3, 3041, 3043,    3042.0,       3.0 },
  {  311,    3, 3044, 3046,    3045.0,       3.0 },
  {  312,    2, 3047, 3048,    3047.5,       2.5 },
  {  313,    2, 3049, 3050,    3049.5,       2.0 },
  {  314,    2, 3051, 3052,    3051.5,       2.0 },
  {  315,    2, 3053, 3054,    3053.5,       2.0 },
  {  316,    2, 3055, 3056,    3055.5,       2.0 },
  {  317,    2, 3057, 3058,    3057.5,       2.0 },
  {  318,    2, 3059, 3060,    3059.5,       2.0 },
  {  319,    2, 3061, 3062,    3061.5,       2.0 },
  {  320,    2, 3063, 3064,    3063.5,       2.0 },
  {  321,    2, 3065, 3066,    3065.5,       2.0 },
  {  322,    2, 3067, 3068,    3067.5,       2.0 },
  {  323,    2, 3069, 3070,    3069.5,       2.0 },
  {  324,    2, 3071, 3072,    3071.5,       2.0 },
  {  325,    2, 3073, 3074,    3073.5,       2.0 },
  {  326,    2, 3075, 3076,    3075.5,       2.0 },
  {  327,    2, 3077, 3078,    3077.5,       2.0 },
  {  328,    2, 3079, 3080,    3079.5,       2.0 },
  {  329,    2, 3081, 3082,    3081.5,       2.0 },
  {  330,    2, 3083, 3084,    3083.5,       2.0 },
  {  331,    2, 3085, 3086,    3085.5,       2.0 },
  {  332,    2, 3087, 3088,    3087.5,       2.0 },
  {  333,    2, 3089, 3090,    3089.5,       2.0 },
  {  334,    2, 3091, 3092,    3091.5,       2.0 },
  {  335,    1, 3093, 3093,    3093.0,       1.5 },
  {  336,    1, 3094, 3094,    3094.0,       1.0 },
  {  337,    1, 3095, 3095,    3095.0,       1.0 },
  {  338,    1, 3096, 3096,    3096.0,       1.0 },
  {  339,    1, 3097, 3097,    3097.0,       1.0 },
  {  340,    1, 3098, 3098,    3098.0,       1.0 },
  {  341,    1, 3099, 3099,    3099.0,       1.0 },
  {  342,    1, 3100, 3100,    3100.0,       1.0 },
  {  343,    1, 3101, 3101,    3101.0,       1.0 },
  {  344,    1, 3102, 3102,    3102.0,       1.0 },
  {  345,    1, 3103, 3103,    3103.0,       1.0 },
  {  346,    1, 3104, 3104,    3104.0,       1.0 },
  {  347,    1, 3105, 3105,    3105.0,       1.0 },
  {  348,    1, 3106, 3106,    3106.0,       1.0 },
  {  349,    1, 3107, 3107,    3107.0,       1.0 },
  {  350,    1, 3108, 3108,    3108.0,       1.0 },
  {  351,    1, 3109, 3109,    3109.0,       1.0 },
  {  352,    1, 3110, 3110,    3110.0,       1.0 },
  {  353,    1, 3111, 3111,    3111.0,       1.0 },
  {  354,    1, 3112, 3112,    3112.0,       1.0 },
  {  355,    1, 3113, 3113,    3113.0,       1.0 },
  {  356,    1, 3114, 3114,    3114.0,       1.0 },
  {  357,    1, 3115, 3115,    3115.0,       1.0 },
  {  358,    1, 3116, 3116,    3116.0,       1.0 },
  {  359,    1, 3117, 3117,    3117.0,       1.0 },
  {  360,    1, 3118, 3118,    3118.0,       1.0 },
  {  361,    1, 3119, 3119,    3119.0,       1.0 },
  {  362,    1, 3120, 3120,    3120.0,       1.0 },
  {  363,    1, 3121, 3121,    3121.0,       1.0 },
  {  364,    1, 3122, 3122,    3122.0,       1.0 },
  {  365,    1, 3123, 3123,    3123.0,       1.0 },
  {  366,    1, 3124, 3124,    3124.0,       1.0 },
  {  367,    1, 3125, 3125,    3125.0,       1.0 },
  {  368,    1, 3126, 3126,    3126.0,       1.0 },
  {  369,    1, 3127, 3127,    3127.0,       1.0 },
  {  370,    1, 3128, 3128,    3128.0,       1.0 },
  {  371,    1, 3129, 3129,    3129.0,       1.0 },
  {  372,    1, 3130, 3130,    3130.0,       1.0 },
  {  373,    1, 3131, 3131,    3131.0,       1.0 },
  {  374,    1, 3132, 3132,    3132.0,       1.0 },
  {  375,    1, 3133, 3133,    3133.0,       1.0 },
  {  376,    1, 3134, 3134,    3134.0,       1.0 },
  {  377,    1, 3135, 3135,    3135.0,       1.0 },
  {  378,    1, 3136, 3136,    3136.0,       1.0 },
  {  379,    1, 3137, 3137,    3137.0,       1.0 },
  {  380,    1, 3138, 3138,    3138.0,       1.0 },
  {  381,    1, 3139, 3139,    3139.0,       1.0 },
  {  382,    1, 3140, 3140,    3140.0,       1.0 },
  {  383,    1, 3141, 3141,    3141.0,       1.0 }
};

/* 3.75 km/pixel nominal resolution; output width 768.  This is indicated */
/* by the keyword CROSSTRACK_SUMMING = 13 in the labels.                  */

#define MOC_XTS_MODE_13                  768
static MOC_XTS_TABLE mode13_table[MOC_XTS_MODE_13] = {
  {    0,    1,    0,    0,       0.0,       1.0 },
  {    1,    1,    1,    1,       1.0,       1.0 },
  {    2,    1,    2,    2,       2.0,       1.0 },
  {    3,    1,    3,    3,       3.0,       1.0 },
  {    4,    1,    4,    4,       4.0,       1.0 },
  {    5,    1,    5,    5,       5.0,       1.0 },
  {    6,    1,    6,    6,       6.0,       1.0 },
  {    7,    1,    7,    7,       7.0,       1.0 },
  {    8,    1,    8,    8,       8.0,       1.0 },
  {    9,    1,    9,    9,       9.0,       1.0 },
  {   10,    1,   10,   10,      10.0,       1.0 },
  {   11,    1,   11,   11,      11.0,       1.0 },
  {   12,    1,   12,   12,      12.0,       1.0 },
  {   13,    1,   13,   13,      13.0,       1.0 },
  {   14,    1,   14,   14,      14.0,       1.0 },
  {   15,    1,   15,   15,      15.0,       1.0 },
  {   16,    1,   16,   16,      16.0,       1.0 },
  {   17,    1,   17,   17,      17.0,       1.0 },
  {   18,    1,   18,   18,      18.0,       1.0 },
  {   19,    1,   19,   19,      19.0,       1.0 },
  {   20,    1,   20,   20,      20.0,       1.0 },
  {   21,    1,   21,   21,      21.0,       1.0 },
  {   22,    1,   22,   22,      22.0,       1.0 },
  {   23,    1,   23,   23,      23.0,       1.0 },
  {   24,    1,   24,   24,      24.0,       1.0 },
  {   25,    1,   25,   25,      25.0,       1.0 },
  {   26,    1,   26,   26,      26.0,       1.0 },
  {   27,    1,   27,   27,      27.0,       1.0 },
  {   28,    1,   28,   28,      28.0,       1.0 },
  {   29,    1,   29,   29,      29.0,       1.0 },
  {   30,    1,   30,   30,      30.0,       1.0 },
  {   31,    1,   31,   31,      31.0,       1.0 },
  {   32,    1,   32,   32,      32.0,       1.0 },
  {   33,    1,   33,   33,      33.0,       1.0 },
  {   34,    1,   34,   34,      34.0,       1.0 },
  {   35,    1,   35,   35,      35.0,       1.0 },
  {   36,    1,   36,   36,      36.0,       1.0 },
  {   37,    1,   37,   37,      37.0,       1.0 },
  {   38,    1,   38,   38,      38.0,       1.0 },
  {   39,    1,   39,   39,      39.0,       1.0 },
  {   40,    1,   40,   40,      40.0,       1.0 },
  {   41,    1,   41,   41,      41.0,       1.0 },
  {   42,    1,   42,   42,      42.0,       1.0 },
  {   43,    1,   43,   43,      43.0,       1.0 },
  {   44,    1,   44,   44,      44.0,       1.0 },
  {   45,    1,   45,   45,      45.0,       1.0 },
  {   46,    1,   46,   46,      46.0,       1.0 },
  {   47,    1,   47,   47,      47.0,       1.0 },
  {   48,    1,   48,   48,      48.0,       1.0 },
  {   49,    1,   49,   49,      49.0,       1.0 },
  {   50,    1,   50,   50,      50.0,       1.0 },
  {   51,    1,   51,   51,      51.0,       1.0 },
  {   52,    1,   52,   52,      52.0,       1.0 },
  {   53,    1,   53,   53,      53.0,       1.0 },
  {   54,    1,   54,   54,      54.0,       1.0 },
  {   55,    1,   55,   55,      55.0,       1.0 },
  {   56,    1,   56,   56,      56.0,       1.0 },
  {   57,    1,   57,   57,      57.0,       1.0 },
  {   58,    1,   58,   58,      58.0,       1.0 },
  {   59,    1,   59,   59,      59.0,       1.0 },
  {   60,    1,   60,   60,      60.0,       1.0 },
  {   61,    1,   61,   61,      61.0,       1.0 },
  {   62,    1,   62,   62,      62.0,       1.0 },
  {   63,    1,   63,   63,      63.0,       1.0 },
  {   64,    1,   64,   64,      64.0,       1.0 },
  {   65,    1,   65,   65,      65.0,       1.0 },
  {   66,    1,   66,   66,      66.0,       1.0 },
  {   67,    1,   67,   67,      67.0,       1.0 },
  {   68,    1,   68,   68,      68.0,       1.0 },
  {   69,    1,   69,   69,      69.0,       1.0 },
  {   70,    1,   70,   70,      70.0,       1.0 },
  {   71,    1,   71,   71,      71.0,       1.0 },
  {   72,    1,   72,   72,      72.0,       1.0 },
  {   73,    1,   73,   73,      73.0,       1.0 },
  {   74,    1,   74,   74,      74.0,       1.0 },
  {   75,    1,   75,   75,      75.0,       1.0 },
  {   76,    1,   76,   76,      76.0,       1.0 },
  {   77,    1,   77,   77,      77.0,       1.0 },
  {   78,    1,   78,   78,      78.0,       1.0 },
  {   79,    1,   79,   79,      79.0,       1.0 },
  {   80,    1,   80,   80,      80.0,       1.0 },
  {   81,    1,   81,   81,      81.0,       1.0 },
  {   82,    1,   82,   82,      82.0,       1.0 },
  {   83,    1,   83,   83,      83.0,       1.0 },
  {   84,    1,   84,   84,      84.0,       1.0 },
  {   85,    1,   85,   85,      85.0,       1.0 },
  {   86,    1,   86,   86,      86.0,       1.0 },
  {   87,    1,   87,   87,      87.0,       1.0 },
  {   88,    1,   88,   88,      88.0,       1.0 },
  {   89,    1,   89,   89,      89.0,       1.0 },
  {   90,    1,   90,   90,      90.0,       1.0 },
  {   91,    1,   91,   91,      91.0,       1.0 },
  {   92,    1,   92,   92,      92.0,       1.0 },
  {   93,    1,   93,   93,      93.0,       1.0 },
  {   94,    1,   94,   94,      94.0,       1.0 },
  {   95,    1,   95,   95,      95.0,       1.0 },
  {   96,    1,   96,   96,      96.0,       1.0 },
  {   97,    1,   97,   97,      97.0,       1.0 },
  {   98,    1,   98,   98,      98.0,       1.0 },
  {   99,    1,   99,   99,      99.0,       1.0 },
  {  100,    1,  100,  100,     100.0,       1.0 },
  {  101,    1,  101,  101,     101.0,       1.0 },
  {  102,    1,  102,  102,     102.0,       1.0 },
  {  103,    1,  103,  103,     103.0,       1.0 },
  {  104,    1,  104,  104,     104.0,       1.0 },
  {  105,    1,  105,  105,     105.0,       1.0 },
  {  106,    1,  106,  106,     106.0,       1.0 },
  {  107,    1,  107,  107,     107.0,       1.0 },
  {  108,    1,  108,  108,     108.0,       1.0 },
  {  109,    1,  109,  109,     109.0,       1.0 },
  {  110,    1,  110,  110,     110.0,       1.0 },
  {  111,    1,  111,  111,     111.0,       1.0 },
  {  112,    1,  112,  112,     112.0,       1.0 },
  {  113,    1,  113,  113,     113.0,       1.0 },
  {  114,    1,  114,  114,     114.0,       1.0 },
  {  115,    1,  115,  115,     115.0,       1.0 },
  {  116,    1,  116,  116,     116.0,       1.0 },
  {  117,    1,  117,  117,     117.0,       1.0 },
  {  118,    1,  118,  118,     118.0,       1.0 },
  {  119,    1,  119,  119,     119.0,       1.0 },
  {  120,    1,  120,  120,     120.0,       1.0 },
  {  121,    1,  121,  121,     121.0,       1.0 },
  {  122,    1,  122,  122,     122.0,       1.0 },
  {  123,    1,  123,  123,     123.0,       1.0 },
  {  124,    1,  124,  124,     124.0,       1.0 },
  {  125,    1,  125,  125,     125.0,       1.0 },
  {  126,    1,  126,  126,     126.0,       1.0 },
  {  127,    1,  127,  127,     127.0,       1.0 },
  {  128,    1,  128,  128,     128.0,       1.0 },
  {  129,    1,  129,  129,     129.0,       1.0 },
  {  130,    1,  130,  130,     130.0,       1.0 },
  {  131,    1,  131,  131,     131.0,       1.0 },
  {  132,    1,  132,  132,     132.0,       1.0 },
  {  133,    1,  133,  133,     133.0,       1.0 },
  {  134,    1,  134,  134,     134.0,       1.0 },
  {  135,    1,  135,  135,     135.0,       1.0 },
  {  136,    1,  136,  136,     136.0,       1.0 },
  {  137,    1,  137,  137,     137.0,       1.0 },
  {  138,    1,  138,  138,     138.0,       1.0 },
  {  139,    1,  139,  139,     139.0,       1.0 },
  {  140,    1,  140,  140,     140.0,       1.0 },
  {  141,    1,  141,  141,     141.0,       1.0 },
  {  142,    1,  142,  142,     142.0,       1.0 },
  {  143,    1,  143,  143,     143.0,       1.0 },
  {  144,    1,  144,  144,     144.0,       1.0 },
  {  145,    1,  145,  145,     145.0,       1.0 },
  {  146,    1,  146,  146,     146.0,       1.0 },
  {  147,    1,  147,  147,     147.0,       1.0 },
  {  148,    1,  148,  148,     148.0,       1.0 },
  {  149,    1,  149,  149,     149.0,       1.0 },
  {  150,    1,  150,  150,     150.0,       1.0 },
  {  151,    1,  151,  151,     151.0,       1.0 },
  {  152,    1,  152,  152,     152.0,       1.0 },
  {  153,    1,  153,  153,     153.0,       1.0 },
  {  154,    1,  154,  154,     154.0,       1.0 },
  {  155,    1,  155,  155,     155.0,       1.0 },
  {  156,    1,  156,  156,     156.0,       1.0 },
  {  157,    1,  157,  157,     157.0,       1.0 },
  {  158,    1,  158,  158,     158.0,       1.0 },
  {  159,    1,  159,  159,     159.0,       1.0 },
  {  160,    1,  160,  160,     160.0,       1.0 },
  {  161,    1,  161,  161,     161.0,       1.0 },
  {  162,    1,  162,  162,     162.0,       1.0 },
  {  163,    1,  163,  163,     163.0,       1.0 },
  {  164,    2,  164,  165,     164.5,       1.5 },
  {  165,    2,  166,  167,     166.5,       2.0 },
  {  166,    2,  168,  169,     168.5,       2.0 },
  {  167,    2,  170,  171,     170.5,       2.0 },
  {  168,    2,  172,  173,     172.5,       2.0 },
  {  169,    2,  174,  175,     174.5,       2.0 },
  {  170,    2,  176,  177,     176.5,       2.0 },
  {  171,    2,  178,  179,     178.5,       2.0 },
  {  172,    2,  180,  181,     180.5,       2.0 },
  {  173,    2,  182,  183,     182.5,       2.0 },
  {  174,    2,  184,  185,     184.5,       2.0 },
  {  175,    2,  186,  187,     186.5,       2.0 },
  {  176,    2,  188,  189,     188.5,       2.0 },
  {  177,    2,  190,  191,     190.5,       2.0 },
  {  178,    2,  192,  193,     192.5,       2.0 },
  {  179,    2,  194,  195,     194.5,       2.0 },
  {  180,    2,  196,  197,     196.5,       2.0 },
  {  181,    2,  198,  199,     198.5,       2.0 },
  {  182,    2,  200,  201,     200.5,       2.0 },
  {  183,    2,  202,  203,     202.5,       2.0 },
  {  184,    2,  204,  205,     204.5,       2.0 },
  {  185,    2,  206,  207,     206.5,       2.0 },
  {  186,    2,  208,  209,     208.5,       2.0 },
  {  187,    2,  210,  211,     210.5,       2.0 },
  {  188,    2,  212,  213,     212.5,       2.0 },
  {  189,    2,  214,  215,     214.5,       2.0 },
  {  190,    2,  216,  217,     216.5,       2.0 },
  {  191,    2,  218,  219,     218.5,       2.0 },
  {  192,    2,  220,  221,     220.5,       2.0 },
  {  193,    2,  222,  223,     222.5,       2.0 },
  {  194,    2,  224,  225,     224.5,       2.0 },
  {  195,    2,  226,  227,     226.5,       2.0 },
  {  196,    2,  228,  229,     228.5,       2.0 },
  {  197,    2,  230,  231,     230.5,       2.0 },
  {  198,    2,  232,  233,     232.5,       2.0 },
  {  199,    2,  234,  235,     234.5,       2.0 },
  {  200,    2,  236,  237,     236.5,       2.0 },
  {  201,    2,  238,  239,     238.5,       2.0 },
  {  202,    2,  240,  241,     240.5,       2.0 },
  {  203,    2,  242,  243,     242.5,       2.0 },
  {  204,    2,  244,  245,     244.5,       2.0 },
  {  205,    2,  246,  247,     246.5,       2.0 },
  {  206,    2,  248,  249,     248.5,       2.0 },
  {  207,    2,  250,  251,     250.5,       2.0 },
  {  208,    2,  252,  253,     252.5,       2.0 },
  {  209,    2,  254,  255,     254.5,       2.0 },
  {  210,    2,  256,  257,     256.5,       2.0 },
  {  211,    3,  258,  260,     259.0,       2.5 },
  {  212,    3,  261,  263,     262.0,       3.0 },
  {  213,    3,  264,  266,     265.0,       3.0 },
  {  214,    3,  267,  269,     268.0,       3.0 },
  {  215,    3,  270,  272,     271.0,       3.0 },
  {  216,    3,  273,  275,     274.0,       3.0 },
  {  217,    3,  276,  278,     277.0,       3.0 },
  {  218,    3,  279,  281,     280.0,       3.0 },
  {  219,    3,  282,  284,     283.0,       3.0 },
  {  220,    3,  285,  287,     286.0,       3.0 },
  {  221,    3,  288,  290,     289.0,       3.0 },
  {  222,    3,  291,  293,     292.0,       3.0 },
  {  223,    3,  294,  296,     295.0,       3.0 },
  {  224,    3,  297,  299,     298.0,       3.0 },
  {  225,    3,  300,  302,     301.0,       3.0 },
  {  226,    3,  303,  305,     304.0,       3.0 },
  {  227,    3,  306,  308,     307.0,       3.0 },
  {  228,    3,  309,  311,     310.0,       3.0 },
  {  229,    3,  312,  314,     313.0,       3.0 },
  {  230,    3,  315,  317,     316.0,       3.0 },
  {  231,    3,  318,  320,     319.0,       3.0 },
  {  232,    3,  321,  323,     322.0,       3.0 },
  {  233,    3,  324,  326,     325.0,       3.0 },
  {  234,    4,  327,  330,     328.5,       3.5 },
  {  235,    4,  331,  334,     332.5,       4.0 },
  {  236,    4,  335,  338,     336.5,       4.0 },
  {  237,    4,  339,  342,     340.5,       4.0 },
  {  238,    4,  343,  346,     344.5,       4.0 },
  {  239,    4,  347,  350,     348.5,       4.0 },
  {  240,    4,  351,  354,     352.5,       4.0 },
  {  241,    4,  355,  358,     356.5,       4.0 },
  {  242,    4,  359,  362,     360.5,       4.0 },
  {  243,    4,  363,  366,     364.5,       4.0 },
  {  244,    4,  367,  370,     368.5,       4.0 },
  {  245,    4,  371,  374,     372.5,       4.0 },
  {  246,    4,  375,  378,     376.5,       4.0 },
  {  247,    4,  379,  382,     380.5,       4.0 },
  {  248,    4,  383,  386,     384.5,       4.0 },
  {  249,    4,  387,  390,     388.5,       4.0 },
  {  250,    4,  391,  394,     392.5,       4.0 },
  {  251,    4,  395,  398,     396.5,       4.0 },
  {  252,    4,  399,  402,     400.5,       4.0 },
  {  253,    4,  403,  406,     404.5,       4.0 },
  {  254,    4,  407,  410,     408.5,       4.0 },
  {  255,    4,  411,  414,     412.5,       4.0 },
  {  256,    4,  415,  418,     416.5,       4.0 },
  {  257,    4,  419,  422,     420.5,       4.0 },
  {  258,    5,  423,  427,     425.0,       4.5 },
  {  259,    5,  428,  432,     430.0,       5.0 },
  {  260,    5,  433,  437,     435.0,       5.0 },
  {  261,    5,  438,  442,     440.0,       5.0 },
  {  262,    5,  443,  447,     445.0,       5.0 },
  {  263,    5,  448,  452,     450.0,       5.0 },
  {  264,    5,  453,  457,     455.0,       5.0 },
  {  265,    5,  458,  462,     460.0,       5.0 },
  {  266,    5,  463,  467,     465.0,       5.0 },
  {  267,    5,  468,  472,     470.0,       5.0 },
  {  268,    5,  473,  477,     475.0,       5.0 },
  {  269,    5,  478,  482,     480.0,       5.0 },
  {  270,    5,  483,  487,     485.0,       5.0 },
  {  271,    5,  488,  492,     490.0,       5.0 },
  {  272,    5,  493,  497,     495.0,       5.0 },
  {  273,    5,  498,  502,     500.0,       5.0 },
  {  274,    5,  503,  507,     505.0,       5.0 },
  {  275,    5,  508,  512,     510.0,       5.0 },
  {  276,    6,  513,  518,     515.5,       5.5 },
  {  277,    6,  519,  524,     521.5,       6.0 },
  {  278,    6,  525,  530,     527.5,       6.0 },
  {  279,    6,  531,  536,     533.5,       6.0 },
  {  280,    6,  537,  542,     539.5,       6.0 },
  {  281,    6,  543,  548,     545.5,       6.0 },
  {  282,    6,  549,  554,     551.5,       6.0 },
  {  283,    6,  555,  560,     557.5,       6.0 },
  {  284,    6,  561,  566,     563.5,       6.0 },
  {  285,    6,  567,  572,     569.5,       6.0 },
  {  286,    6,  573,  578,     575.5,       6.0 },
  {  287,    6,  579,  584,     581.5,       6.0 },
  {  288,    7,  585,  591,     588.0,       6.5 },
  {  289,    7,  592,  598,     595.0,       7.0 },
  {  290,    7,  599,  605,     602.0,       7.0 },
  {  291,    7,  606,  612,     609.0,       7.0 },
  {  292,    7,  613,  619,     616.0,       7.0 },
  {  293,    7,  620,  626,     623.0,       7.0 },
  {  294,    7,  627,  633,     630.0,       7.0 },
  {  295,    7,  634,  640,     637.0,       7.0 },
  {  296,    7,  641,  647,     644.0,       7.0 },
  {  297,    7,  648,  654,     651.0,       7.0 },
  {  298,    7,  655,  661,     658.0,       7.0 },
  {  299,    7,  662,  668,     665.0,       7.0 },
  {  300,    7,  669,  675,     672.0,       7.0 },
  {  301,    7,  676,  682,     679.0,       7.0 },
  {  302,    8,  683,  690,     686.5,       7.5 },
  {  303,    8,  691,  698,     694.5,       8.0 },
  {  304,    8,  699,  706,     702.5,       8.0 },
  {  305,    8,  707,  714,     710.5,       8.0 },
  {  306,    8,  715,  722,     718.5,       8.0 },
  {  307,    8,  723,  730,     726.5,       8.0 },
  {  308,    8,  731,  738,     734.5,       8.0 },
  {  309,    8,  739,  746,     742.5,       8.0 },
  {  310,    8,  747,  754,     750.5,       8.0 },
  {  311,    8,  755,  762,     758.5,       8.0 },
  {  312,    8,  763,  770,     766.5,       8.0 },
  {  313,    9,  771,  779,     775.0,       8.5 },
  {  314,    9,  780,  788,     784.0,       9.0 },
  {  315,    9,  789,  797,     793.0,       9.0 },
  {  316,    9,  798,  806,     802.0,       9.0 },
  {  317,    9,  807,  815,     811.0,       9.0 },
  {  318,    9,  816,  824,     820.0,       9.0 },
  {  319,    9,  825,  833,     829.0,       9.0 },
  {  320,    9,  834,  842,     838.0,       9.0 },
  {  321,    9,  843,  851,     847.0,       9.0 },
  {  322,    9,  852,  860,     856.0,       9.0 },
  {  323,   10,  861,  870,     865.5,       9.5 },
  {  324,   10,  871,  880,     875.5,      10.0 },
  {  325,   10,  881,  890,     885.5,      10.0 },
  {  326,   10,  891,  900,     895.5,      10.0 },
  {  327,   10,  901,  910,     905.5,      10.0 },
  {  328,   10,  911,  920,     915.5,      10.0 },
  {  329,   10,  921,  930,     925.5,      10.0 },
  {  330,   10,  931,  940,     935.5,      10.0 },
  {  331,   10,  941,  950,     945.5,      10.0 },
  {  332,   10,  951,  960,     955.5,      10.0 },
  {  333,   10,  961,  970,     965.5,      10.0 },
  {  334,   10,  971,  980,     975.5,      10.0 },
  {  335,   11,  981,  991,     986.0,      10.5 },
  {  336,   11,  992, 1002,     997.0,      11.0 },
  {  337,   11, 1003, 1013,    1008.0,      11.0 },
  {  338,   11, 1014, 1024,    1019.0,      11.0 },
  {  339,   11, 1025, 1035,    1030.0,      11.0 },
  {  340,   11, 1036, 1046,    1041.0,      11.0 },
  {  341,   12, 1047, 1058,    1052.5,      11.5 },
  {  342,   12, 1059, 1070,    1064.5,      12.0 },
  {  343,   12, 1071, 1082,    1076.5,      12.0 },
  {  344,   12, 1083, 1094,    1088.5,      12.0 },
  {  345,   12, 1095, 1106,    1100.5,      12.0 },
  {  346,   12, 1107, 1118,    1112.5,      12.0 },
  {  347,   12, 1119, 1130,    1124.5,      12.0 },
  {  348,   12, 1131, 1142,    1136.5,      12.0 },
  {  349,   12, 1143, 1154,    1148.5,      12.0 },
  {  350,   12, 1155, 1166,    1160.5,      12.0 },
  {  351,   12, 1167, 1178,    1172.5,      12.0 },
  {  352,   12, 1179, 1190,    1184.5,      12.0 },
  {  353,   12, 1191, 1202,    1196.5,      12.0 },
  {  354,   12, 1203, 1214,    1208.5,      12.0 },
  {  355,   12, 1215, 1226,    1220.5,      12.0 },
  {  356,   13, 1227, 1239,    1233.0,      12.5 },
  {  357,   13, 1240, 1252,    1246.0,      13.0 },
  {  358,   13, 1253, 1265,    1259.0,      13.0 },
  {  359,   13, 1266, 1278,    1272.0,      13.0 },
  {  360,   13, 1279, 1291,    1285.0,      13.0 },
  {  361,   13, 1292, 1304,    1298.0,      13.0 },
  {  362,   13, 1305, 1317,    1311.0,      13.0 },
  {  363,   13, 1318, 1330,    1324.0,      13.0 },
  {  364,   13, 1331, 1343,    1337.0,      13.0 },
  {  365,   13, 1344, 1356,    1350.0,      13.0 },
  {  366,   13, 1357, 1369,    1363.0,      13.0 },
  {  367,   13, 1370, 1382,    1376.0,      13.0 },
  {  368,   13, 1383, 1395,    1389.0,      13.0 },
  {  369,   13, 1396, 1408,    1402.0,      13.0 },
  {  370,   13, 1409, 1421,    1415.0,      13.0 },
  {  371,   13, 1422, 1434,    1428.0,      13.0 },
  {  372,   13, 1435, 1447,    1441.0,      13.0 },
  {  373,   13, 1448, 1460,    1454.0,      13.0 },
  {  374,   13, 1461, 1473,    1467.0,      13.0 },
  {  375,   13, 1474, 1486,    1480.0,      13.0 },
  {  376,   13, 1487, 1499,    1493.0,      13.0 },
  {  377,   13, 1500, 1512,    1506.0,      13.0 },
  {  378,   13, 1513, 1525,    1519.0,      13.0 },
  {  379,   13, 1526, 1538,    1532.0,      13.0 },
  {  380,   13, 1539, 1551,    1545.0,      13.0 },
  {  381,   13, 1552, 1564,    1558.0,      13.0 },
  {  382,   13, 1565, 1577,    1571.0,      13.0 },
  {  383,   13, 1578, 1590,    1584.0,      13.0 },
  {  384,   13, 1591, 1603,    1597.0,      13.0 },
  {  385,   13, 1604, 1616,    1610.0,      13.0 },
  {  386,   13, 1617, 1629,    1623.0,      13.0 },
  {  387,   13, 1630, 1642,    1636.0,      13.0 },
  {  388,   13, 1643, 1655,    1649.0,      13.0 },
  {  389,   13, 1656, 1668,    1662.0,      13.0 },
  {  390,   13, 1669, 1681,    1675.0,      13.0 },
  {  391,   13, 1682, 1694,    1688.0,      13.0 },
  {  392,   13, 1695, 1707,    1701.0,      13.0 },
  {  393,   13, 1708, 1720,    1714.0,      13.0 },
  {  394,   13, 1721, 1733,    1727.0,      13.0 },
  {  395,   13, 1734, 1746,    1740.0,      13.0 },
  {  396,   13, 1747, 1759,    1753.0,      13.0 },
  {  397,   13, 1760, 1772,    1766.0,      13.0 },
  {  398,   13, 1773, 1785,    1779.0,      13.0 },
  {  399,   13, 1786, 1798,    1792.0,      13.0 },
  {  400,   13, 1799, 1811,    1805.0,      13.0 },
  {  401,   13, 1812, 1824,    1818.0,      13.0 },
  {  402,   13, 1825, 1837,    1831.0,      13.0 },
  {  403,   13, 1838, 1850,    1844.0,      13.0 },
  {  404,   13, 1851, 1863,    1857.0,      13.0 },
  {  405,   13, 1864, 1876,    1870.0,      13.0 },
  {  406,   13, 1877, 1889,    1883.0,      13.0 },
  {  407,   13, 1890, 1902,    1896.0,      13.0 },
  {  408,   13, 1903, 1915,    1909.0,      13.0 },
  {  409,   13, 1916, 1928,    1922.0,      13.0 },
  {  410,   13, 1929, 1941,    1935.0,      13.0 },
  {  411,   13, 1942, 1954,    1948.0,      13.0 },
  {  412,   12, 1955, 1966,    1960.5,      12.5 },
  {  413,   12, 1967, 1978,    1972.5,      12.0 },
  {  414,   12, 1979, 1990,    1984.5,      12.0 },
  {  415,   12, 1991, 2002,    1996.5,      12.0 },
  {  416,   12, 2003, 2014,    2008.5,      12.0 },
  {  417,   12, 2015, 2026,    2020.5,      12.0 },
  {  418,   12, 2027, 2038,    2032.5,      12.0 },
  {  419,   12, 2039, 2050,    2044.5,      12.0 },
  {  420,   12, 2051, 2062,    2056.5,      12.0 },
  {  421,   12, 2063, 2074,    2068.5,      12.0 },
  {  422,   12, 2075, 2086,    2080.5,      12.0 },
  {  423,   12, 2087, 2098,    2092.5,      12.0 },
  {  424,   12, 2099, 2110,    2104.5,      12.0 },
  {  425,   12, 2111, 2122,    2116.5,      12.0 },
  {  426,   12, 2123, 2134,    2128.5,      12.0 },
  {  427,   11, 2135, 2145,    2140.0,      11.5 },
  {  428,   11, 2146, 2156,    2151.0,      11.0 },
  {  429,   11, 2157, 2167,    2162.0,      11.0 },
  {  430,   11, 2168, 2178,    2173.0,      11.0 },
  {  431,   11, 2179, 2189,    2184.0,      11.0 },
  {  432,   11, 2190, 2200,    2195.0,      11.0 },
  {  433,   10, 2201, 2210,    2205.5,      10.5 },
  {  434,   10, 2211, 2220,    2215.5,      10.0 },
  {  435,   10, 2221, 2230,    2225.5,      10.0 },
  {  436,   10, 2231, 2240,    2235.5,      10.0 },
  {  437,   10, 2241, 2250,    2245.5,      10.0 },
  {  438,   10, 2251, 2260,    2255.5,      10.0 },
  {  439,   10, 2261, 2270,    2265.5,      10.0 },
  {  440,   10, 2271, 2280,    2275.5,      10.0 },
  {  441,   10, 2281, 2290,    2285.5,      10.0 },
  {  442,   10, 2291, 2300,    2295.5,      10.0 },
  {  443,   10, 2301, 2310,    2305.5,      10.0 },
  {  444,   10, 2311, 2320,    2315.5,      10.0 },
  {  445,    9, 2321, 2329,    2325.0,       9.5 },
  {  446,    9, 2330, 2338,    2334.0,       9.0 },
  {  447,    9, 2339, 2347,    2343.0,       9.0 },
  {  448,    9, 2348, 2356,    2352.0,       9.0 },
  {  449,    9, 2357, 2365,    2361.0,       9.0 },
  {  450,    9, 2366, 2374,    2370.0,       9.0 },
  {  451,    9, 2375, 2383,    2379.0,       9.0 },
  {  452,    9, 2384, 2392,    2388.0,       9.0 },
  {  453,    9, 2393, 2401,    2397.0,       9.0 },
  {  454,    9, 2402, 2410,    2406.0,       9.0 },
  {  455,    8, 2411, 2418,    2414.5,       8.5 },
  {  456,    8, 2419, 2426,    2422.5,       8.0 },
  {  457,    8, 2427, 2434,    2430.5,       8.0 },
  {  458,    8, 2435, 2442,    2438.5,       8.0 },
  {  459,    8, 2443, 2450,    2446.5,       8.0 },
  {  460,    8, 2451, 2458,    2454.5,       8.0 },
  {  461,    8, 2459, 2466,    2462.5,       8.0 },
  {  462,    8, 2467, 2474,    2470.5,       8.0 },
  {  463,    8, 2475, 2482,    2478.5,       8.0 },
  {  464,    8, 2483, 2490,    2486.5,       8.0 },
  {  465,    8, 2491, 2498,    2494.5,       8.0 },
  {  466,    7, 2499, 2505,    2502.0,       7.5 },
  {  467,    7, 2506, 2512,    2509.0,       7.0 },
  {  468,    7, 2513, 2519,    2516.0,       7.0 },
  {  469,    7, 2520, 2526,    2523.0,       7.0 },
  {  470,    7, 2527, 2533,    2530.0,       7.0 },
  {  471,    7, 2534, 2540,    2537.0,       7.0 },
  {  472,    7, 2541, 2547,    2544.0,       7.0 },
  {  473,    7, 2548, 2554,    2551.0,       7.0 },
  {  474,    7, 2555, 2561,    2558.0,       7.0 },
  {  475,    7, 2562, 2568,    2565.0,       7.0 },
  {  476,    7, 2569, 2575,    2572.0,       7.0 },
  {  477,    7, 2576, 2582,    2579.0,       7.0 },
  {  478,    7, 2583, 2589,    2586.0,       7.0 },
  {  479,    7, 2590, 2596,    2593.0,       7.0 },
  {  480,    6, 2597, 2602,    2599.5,       6.5 },
  {  481,    6, 2603, 2608,    2605.5,       6.0 },
  {  482,    6, 2609, 2614,    2611.5,       6.0 },
  {  483,    6, 2615, 2620,    2617.5,       6.0 },
  {  484,    6, 2621, 2626,    2623.5,       6.0 },
  {  485,    6, 2627, 2632,    2629.5,       6.0 },
  {  486,    6, 2633, 2638,    2635.5,       6.0 },
  {  487,    6, 2639, 2644,    2641.5,       6.0 },
  {  488,    6, 2645, 2650,    2647.5,       6.0 },
  {  489,    6, 2651, 2656,    2653.5,       6.0 },
  {  490,    6, 2657, 2662,    2659.5,       6.0 },
  {  491,    6, 2663, 2668,    2665.5,       6.0 },
  {  492,    5, 2669, 2673,    2671.0,       5.5 },
  {  493,    5, 2674, 2678,    2676.0,       5.0 },
  {  494,    5, 2679, 2683,    2681.0,       5.0 },
  {  495,    5, 2684, 2688,    2686.0,       5.0 },
  {  496,    5, 2689, 2693,    2691.0,       5.0 },
  {  497,    5, 2694, 2698,    2696.0,       5.0 },
  {  498,    5, 2699, 2703,    2701.0,       5.0 },
  {  499,    5, 2704, 2708,    2706.0,       5.0 },
  {  500,    5, 2709, 2713,    2711.0,       5.0 },
  {  501,    5, 2714, 2718,    2716.0,       5.0 },
  {  502,    5, 2719, 2723,    2721.0,       5.0 },
  {  503,    5, 2724, 2728,    2726.0,       5.0 },
  {  504,    5, 2729, 2733,    2731.0,       5.0 },
  {  505,    5, 2734, 2738,    2736.0,       5.0 },
  {  506,    5, 2739, 2743,    2741.0,       5.0 },
  {  507,    5, 2744, 2748,    2746.0,       5.0 },
  {  508,    5, 2749, 2753,    2751.0,       5.0 },
  {  509,    5, 2754, 2758,    2756.0,       5.0 },
  {  510,    4, 2759, 2762,    2760.5,       4.5 },
  {  511,    4, 2763, 2766,    2764.5,       4.0 },
  {  512,    4, 2767, 2770,    2768.5,       4.0 },
  {  513,    4, 2771, 2774,    2772.5,       4.0 },
  {  514,    4, 2775, 2778,    2776.5,       4.0 },
  {  515,    4, 2779, 2782,    2780.5,       4.0 },
  {  516,    4, 2783, 2786,    2784.5,       4.0 },
  {  517,    4, 2787, 2790,    2788.5,       4.0 },
  {  518,    4, 2791, 2794,    2792.5,       4.0 },
  {  519,    4, 2795, 2798,    2796.5,       4.0 },
  {  520,    4, 2799, 2802,    2800.5,       4.0 },
  {  521,    4, 2803, 2806,    2804.5,       4.0 },
  {  522,    4, 2807, 2810,    2808.5,       4.0 },
  {  523,    4, 2811, 2814,    2812.5,       4.0 },
  {  524,    4, 2815, 2818,    2816.5,       4.0 },
  {  525,    4, 2819, 2822,    2820.5,       4.0 },
  {  526,    4, 2823, 2826,    2824.5,       4.0 },
  {  527,    4, 2827, 2830,    2828.5,       4.0 },
  {  528,    4, 2831, 2834,    2832.5,       4.0 },
  {  529,    4, 2835, 2838,    2836.5,       4.0 },
  {  530,    4, 2839, 2842,    2840.5,       4.0 },
  {  531,    4, 2843, 2846,    2844.5,       4.0 },
  {  532,    4, 2847, 2850,    2848.5,       4.0 },
  {  533,    4, 2851, 2854,    2852.5,       4.0 },
  {  534,    3, 2855, 2857,    2856.0,       3.5 },
  {  535,    3, 2858, 2860,    2859.0,       3.0 },
  {  536,    3, 2861, 2863,    2862.0,       3.0 },
  {  537,    3, 2864, 2866,    2865.0,       3.0 },
  {  538,    3, 2867, 2869,    2868.0,       3.0 },
  {  539,    3, 2870, 2872,    2871.0,       3.0 },
  {  540,    3, 2873, 2875,    2874.0,       3.0 },
  {  541,    3, 2876, 2878,    2877.0,       3.0 },
  {  542,    3, 2879, 2881,    2880.0,       3.0 },
  {  543,    3, 2882, 2884,    2883.0,       3.0 },
  {  544,    3, 2885, 2887,    2886.0,       3.0 },
  {  545,    3, 2888, 2890,    2889.0,       3.0 },
  {  546,    3, 2891, 2893,    2892.0,       3.0 },
  {  547,    3, 2894, 2896,    2895.0,       3.0 },
  {  548,    3, 2897, 2899,    2898.0,       3.0 },
  {  549,    3, 2900, 2902,    2901.0,       3.0 },
  {  550,    3, 2903, 2905,    2904.0,       3.0 },
  {  551,    3, 2906, 2908,    2907.0,       3.0 },
  {  552,    3, 2909, 2911,    2910.0,       3.0 },
  {  553,    3, 2912, 2914,    2913.0,       3.0 },
  {  554,    3, 2915, 2917,    2916.0,       3.0 },
  {  555,    3, 2918, 2920,    2919.0,       3.0 },
  {  556,    3, 2921, 2923,    2922.0,       3.0 },
  {  557,    2, 2924, 2925,    2924.5,       2.5 },
  {  558,    2, 2926, 2927,    2926.5,       2.0 },
  {  559,    2, 2928, 2929,    2928.5,       2.0 },
  {  560,    2, 2930, 2931,    2930.5,       2.0 },
  {  561,    2, 2932, 2933,    2932.5,       2.0 },
  {  562,    2, 2934, 2935,    2934.5,       2.0 },
  {  563,    2, 2936, 2937,    2936.5,       2.0 },
  {  564,    2, 2938, 2939,    2938.5,       2.0 },
  {  565,    2, 2940, 2941,    2940.5,       2.0 },
  {  566,    2, 2942, 2943,    2942.5,       2.0 },
  {  567,    2, 2944, 2945,    2944.5,       2.0 },
  {  568,    2, 2946, 2947,    2946.5,       2.0 },
  {  569,    2, 2948, 2949,    2948.5,       2.0 },
  {  570,    2, 2950, 2951,    2950.5,       2.0 },
  {  571,    2, 2952, 2953,    2952.5,       2.0 },
  {  572,    2, 2954, 2955,    2954.5,       2.0 },
  {  573,    2, 2956, 2957,    2956.5,       2.0 },
  {  574,    2, 2958, 2959,    2958.5,       2.0 },
  {  575,    2, 2960, 2961,    2960.5,       2.0 },
  {  576,    2, 2962, 2963,    2962.5,       2.0 },
  {  577,    2, 2964, 2965,    2964.5,       2.0 },
  {  578,    2, 2966, 2967,    2966.5,       2.0 },
  {  579,    2, 2968, 2969,    2968.5,       2.0 },
  {  580,    2, 2970, 2971,    2970.5,       2.0 },
  {  581,    2, 2972, 2973,    2972.5,       2.0 },
  {  582,    2, 2974, 2975,    2974.5,       2.0 },
  {  583,    2, 2976, 2977,    2976.5,       2.0 },
  {  584,    2, 2978, 2979,    2978.5,       2.0 },
  {  585,    2, 2980, 2981,    2980.5,       2.0 },
  {  586,    2, 2982, 2983,    2982.5,       2.0 },
  {  587,    2, 2984, 2985,    2984.5,       2.0 },
  {  588,    2, 2986, 2987,    2986.5,       2.0 },
  {  589,    2, 2988, 2989,    2988.5,       2.0 },
  {  590,    2, 2990, 2991,    2990.5,       2.0 },
  {  591,    2, 2992, 2993,    2992.5,       2.0 },
  {  592,    2, 2994, 2995,    2994.5,       2.0 },
  {  593,    2, 2996, 2997,    2996.5,       2.0 },
  {  594,    2, 2998, 2999,    2998.5,       2.0 },
  {  595,    2, 3000, 3001,    3000.5,       2.0 },
  {  596,    2, 3002, 3003,    3002.5,       2.0 },
  {  597,    2, 3004, 3005,    3004.5,       2.0 },
  {  598,    2, 3006, 3007,    3006.5,       2.0 },
  {  599,    2, 3008, 3009,    3008.5,       2.0 },
  {  600,    2, 3010, 3011,    3010.5,       2.0 },
  {  601,    2, 3012, 3013,    3012.5,       2.0 },
  {  602,    2, 3014, 3015,    3014.5,       2.0 },
  {  603,    2, 3016, 3017,    3016.5,       2.0 },
  {  604,    1, 3018, 3018,    3018.0,       1.5 },
  {  605,    1, 3019, 3019,    3019.0,       1.0 },
  {  606,    1, 3020, 3020,    3020.0,       1.0 },
  {  607,    1, 3021, 3021,    3021.0,       1.0 },
  {  608,    1, 3022, 3022,    3022.0,       1.0 },
  {  609,    1, 3023, 3023,    3023.0,       1.0 },
  {  610,    1, 3024, 3024,    3024.0,       1.0 },
  {  611,    1, 3025, 3025,    3025.0,       1.0 },
  {  612,    1, 3026, 3026,    3026.0,       1.0 },
  {  613,    1, 3027, 3027,    3027.0,       1.0 },
  {  614,    1, 3028, 3028,    3028.0,       1.0 },
  {  615,    1, 3029, 3029,    3029.0,       1.0 },
  {  616,    1, 3030, 3030,    3030.0,       1.0 },
  {  617,    1, 3031, 3031,    3031.0,       1.0 },
  {  618,    1, 3032, 3032,    3032.0,       1.0 },
  {  619,    1, 3033, 3033,    3033.0,       1.0 },
  {  620,    1, 3034, 3034,    3034.0,       1.0 },
  {  621,    1, 3035, 3035,    3035.0,       1.0 },
  {  622,    1, 3036, 3036,    3036.0,       1.0 },
  {  623,    1, 3037, 3037,    3037.0,       1.0 },
  {  624,    1, 3038, 3038,    3038.0,       1.0 },
  {  625,    1, 3039, 3039,    3039.0,       1.0 },
  {  626,    1, 3040, 3040,    3040.0,       1.0 },
  {  627,    1, 3041, 3041,    3041.0,       1.0 },
  {  628,    1, 3042, 3042,    3042.0,       1.0 },
  {  629,    1, 3043, 3043,    3043.0,       1.0 },
  {  630,    1, 3044, 3044,    3044.0,       1.0 },
  {  631,    1, 3045, 3045,    3045.0,       1.0 },
  {  632,    1, 3046, 3046,    3046.0,       1.0 },
  {  633,    1, 3047, 3047,    3047.0,       1.0 },
  {  634,    1, 3048, 3048,    3048.0,       1.0 },
  {  635,    1, 3049, 3049,    3049.0,       1.0 },
  {  636,    1, 3050, 3050,    3050.0,       1.0 },
  {  637,    1, 3051, 3051,    3051.0,       1.0 },
  {  638,    1, 3052, 3052,    3052.0,       1.0 },
  {  639,    1, 3053, 3053,    3053.0,       1.0 },
  {  640,    1, 3054, 3054,    3054.0,       1.0 },
  {  641,    1, 3055, 3055,    3055.0,       1.0 },
  {  642,    1, 3056, 3056,    3056.0,       1.0 },
  {  643,    1, 3057, 3057,    3057.0,       1.0 },
  {  644,    1, 3058, 3058,    3058.0,       1.0 },
  {  645,    1, 3059, 3059,    3059.0,       1.0 },
  {  646,    1, 3060, 3060,    3060.0,       1.0 },
  {  647,    1, 3061, 3061,    3061.0,       1.0 },
  {  648,    1, 3062, 3062,    3062.0,       1.0 },
  {  649,    1, 3063, 3063,    3063.0,       1.0 },
  {  650,    1, 3064, 3064,    3064.0,       1.0 },
  {  651,    1, 3065, 3065,    3065.0,       1.0 },
  {  652,    1, 3066, 3066,    3066.0,       1.0 },
  {  653,    1, 3067, 3067,    3067.0,       1.0 },
  {  654,    1, 3068, 3068,    3068.0,       1.0 },
  {  655,    1, 3069, 3069,    3069.0,       1.0 },
  {  656,    1, 3070, 3070,    3070.0,       1.0 },
  {  657,    1, 3071, 3071,    3071.0,       1.0 },
  {  658,    1, 3072, 3072,    3072.0,       1.0 },
  {  659,    1, 3073, 3073,    3073.0,       1.0 },
  {  660,    1, 3074, 3074,    3074.0,       1.0 },
  {  661,    1, 3075, 3075,    3075.0,       1.0 },
  {  662,    1, 3076, 3076,    3076.0,       1.0 },
  {  663,    1, 3077, 3077,    3077.0,       1.0 },
  {  664,    1, 3078, 3078,    3078.0,       1.0 },
  {  665,    1, 3079, 3079,    3079.0,       1.0 },
  {  666,    1, 3080, 3080,    3080.0,       1.0 },
  {  667,    1, 3081, 3081,    3081.0,       1.0 },
  {  668,    1, 3082, 3082,    3082.0,       1.0 },
  {  669,    1, 3083, 3083,    3083.0,       1.0 },
  {  670,    1, 3084, 3084,    3084.0,       1.0 },
  {  671,    1, 3085, 3085,    3085.0,       1.0 },
  {  672,    1, 3086, 3086,    3086.0,       1.0 },
  {  673,    1, 3087, 3087,    3087.0,       1.0 },
  {  674,    1, 3088, 3088,    3088.0,       1.0 },
  {  675,    1, 3089, 3089,    3089.0,       1.0 },
  {  676,    1, 3090, 3090,    3090.0,       1.0 },
  {  677,    1, 3091, 3091,    3091.0,       1.0 },
  {  678,    1, 3092, 3092,    3092.0,       1.0 },
  {  679,    1, 3093, 3093,    3093.0,       1.0 },
  {  680,    1, 3094, 3094,    3094.0,       1.0 },
  {  681,    1, 3095, 3095,    3095.0,       1.0 },
  {  682,    1, 3096, 3096,    3096.0,       1.0 },
  {  683,    1, 3097, 3097,    3097.0,       1.0 },
  {  684,    1, 3098, 3098,    3098.0,       1.0 },
  {  685,    1, 3099, 3099,    3099.0,       1.0 },
  {  686,    1, 3100, 3100,    3100.0,       1.0 },
  {  687,    1, 3101, 3101,    3101.0,       1.0 },
  {  688,    1, 3102, 3102,    3102.0,       1.0 },
  {  689,    1, 3103, 3103,    3103.0,       1.0 },
  {  690,    1, 3104, 3104,    3104.0,       1.0 },
  {  691,    1, 3105, 3105,    3105.0,       1.0 },
  {  692,    1, 3106, 3106,    3106.0,       1.0 },
  {  693,    1, 3107, 3107,    3107.0,       1.0 },
  {  694,    1, 3108, 3108,    3108.0,       1.0 },
  {  695,    1, 3109, 3109,    3109.0,       1.0 },
  {  696,    1, 3110, 3110,    3110.0,       1.0 },
  {  697,    1, 3111, 3111,    3111.0,       1.0 },
  {  698,    1, 3112, 3112,    3112.0,       1.0 },
  {  699,    1, 3113, 3113,    3113.0,       1.0 },
  {  700,    1, 3114, 3114,    3114.0,       1.0 },
  {  701,    1, 3115, 3115,    3115.0,       1.0 },
  {  702,    1, 3116, 3116,    3116.0,       1.0 },
  {  703,    1, 3117, 3117,    3117.0,       1.0 },
  {  704,    1, 3118, 3118,    3118.0,       1.0 },
  {  705,    1, 3119, 3119,    3119.0,       1.0 },
  {  706,    1, 3120, 3120,    3120.0,       1.0 },
  {  707,    1, 3121, 3121,    3121.0,       1.0 },
  {  708,    1, 3122, 3122,    3122.0,       1.0 },
  {  709,    1, 3123, 3123,    3123.0,       1.0 },
  {  710,    1, 3124, 3124,    3124.0,       1.0 },
  {  711,    1, 3125, 3125,    3125.0,       1.0 },
  {  712,    1, 3126, 3126,    3126.0,       1.0 },
  {  713,    1, 3127, 3127,    3127.0,       1.0 },
  {  714,    1, 3128, 3128,    3128.0,       1.0 },
  {  715,    1, 3129, 3129,    3129.0,       1.0 },
  {  716,    1, 3130, 3130,    3130.0,       1.0 },
  {  717,    1, 3131, 3131,    3131.0,       1.0 },
  {  718,    1, 3132, 3132,    3132.0,       1.0 },
  {  719,    1, 3133, 3133,    3133.0,       1.0 },
  {  720,    1, 3134, 3134,    3134.0,       1.0 },
  {  721,    1, 3135, 3135,    3135.0,       1.0 },
  {  722,    1, 3136, 3136,    3136.0,       1.0 },
  {  723,    1, 3137, 3137,    3137.0,       1.0 },
  {  724,    1, 3138, 3138,    3138.0,       1.0 },
  {  725,    1, 3139, 3139,    3139.0,       1.0 },
  {  726,    1, 3140, 3140,    3140.0,       1.0 },
  {  727,    1, 3141, 3141,    3141.0,       1.0 },
  {  728,    1, 3142, 3142,    3142.0,       1.0 },
  {  729,    1, 3143, 3143,    3143.0,       1.0 },
  {  730,    1, 3144, 3144,    3144.0,       1.0 },
  {  731,    1, 3145, 3145,    3145.0,       1.0 },
  {  732,    1, 3146, 3146,    3146.0,       1.0 },
  {  733,    1, 3147, 3147,    3147.0,       1.0 },
  {  734,    1, 3148, 3148,    3148.0,       1.0 },
  {  735,    1, 3149, 3149,    3149.0,       1.0 },
  {  736,    1, 3150, 3150,    3150.0,       1.0 },
  {  737,    1, 3151, 3151,    3151.0,       1.0 },
  {  738,    1, 3152, 3152,    3152.0,       1.0 },
  {  739,    1, 3153, 3153,    3153.0,       1.0 },
  {  740,    1, 3154, 3154,    3154.0,       1.0 },
  {  741,    1, 3155, 3155,    3155.0,       1.0 },
  {  742,    1, 3156, 3156,    3156.0,       1.0 },
  {  743,    1, 3157, 3157,    3157.0,       1.0 },
  {  744,    1, 3158, 3158,    3158.0,       1.0 },
  {  745,    1, 3159, 3159,    3159.0,       1.0 },
  {  746,    1, 3160, 3160,    3160.0,       1.0 },
  {  747,    1, 3161, 3161,    3161.0,       1.0 },
  {  748,    1, 3162, 3162,    3162.0,       1.0 },
  {  749,    1, 3163, 3163,    3163.0,       1.0 },
  {  750,    1, 3164, 3164,    3164.0,       1.0 },
  {  751,    1, 3165, 3165,    3165.0,       1.0 },
  {  752,    1, 3166, 3166,    3166.0,       1.0 },
  {  753,    1, 3167, 3167,    3167.0,       1.0 },
  {  754,    1, 3168, 3168,    3168.0,       1.0 },
  {  755,    1, 3169, 3169,    3169.0,       1.0 },
  {  756,    1, 3170, 3170,    3170.0,       1.0 },
  {  757,    1, 3171, 3171,    3171.0,       1.0 },
  {  758,    1, 3172, 3172,    3172.0,       1.0 },
  {  759,    1, 3173, 3173,    3173.0,       1.0 },
  {  760,    1, 3174, 3174,    3174.0,       1.0 },
  {  761,    1, 3175, 3175,    3175.0,       1.0 },
  {  762,    1, 3176, 3176,    3176.0,       1.0 },
  {  763,    1, 3177, 3177,    3177.0,       1.0 },
  {  764,    1, 3178, 3178,    3178.0,       1.0 },
  {  765,    1, 3179, 3179,    3179.0,       1.0 },
  {  766,    1, 3180, 3180,    3180.0,       1.0 },
  {  767,    1, 3181, 3181,    3181.0,       1.0 }
};

#endif

/*
*_Hist Jan 24 2001 Kris Becker USGS, Flagstaff Original Version
*      Dec 20 2002 Janet Barrett, Added components for piecewise linear
*                  equations for mapping between the image sample and
*                  detector.
*_End
*************************************************************************/
