/**
 * @file
 * $Revision$ 
 * $Date$ 
 *
 *   Unless noted otherwise, the portions of Isis written by the USGS are public
 *   domain. See individual third-party library and package descriptions for
 *   intellectual property information,user agreements, and related information.
 *
 *   Although Isis has been used by the USGS, no warranty, expressed or implied,
 *   is made by the USGS as to the accuracy and functioning of such software
 *   and related material nor shall the fact of distribution constitute any such
 *   warranty, and no responsibility is assumed by the USGS in connection
 *   therewith.
 *
 *   For additional information, launch
 *   $ISISROOT/doc//documents/Disclaimers/Disclaimers.html in a browser or see
 *   the Privacy &amp; Disclaimers page on the Isis website,
 *   http://isis.astrogeology.usgs.gov, and the USGS privacy and disclaimers on
 *   http://www.usgs.gov/privacy.html.
 */

#include <boost/foreach.hpp>
#include "opencv2/opencv.hpp"
#include "opencv2/xfeatures2d.hpp"

#include "BRISKAlgorithm.h"
#include <QVariant>

namespace Isis {


  /**
   * Constructs the algorithm with default variables.
   */
  BRISKAlgorithm::BRISKAlgorithm() : 
                 Feature2DAlgorithm("BRISK", "Feature2D",
                                    BRISKType::create()) {  
    setupParameters();
  }


  /**
   * Constructs the algorithm with the input variables
   * 
   * @param cvars  The variables and values the algorithm will use.
   *               Variables that are not included will be set to their default.
   * @param config The config string used to construct cvars.
   */
  BRISKAlgorithm::BRISKAlgorithm(const PvlFlatMap &cvars, const QString &config) :
                  Feature2DAlgorithm("BRISK", "Feature2D", 
                                     BRISKType::create(), cvars) {
   setConfig(config);
   PvlFlatMap variables = setupParameters();
   variables.merge(cvars);

   // Constructs the algorithm for a custom pattern
   if (variables.exists("RadiusList") && variables.exists("NumberList")) {
     std::vector<float> radiusList;
     QStringList radiusStrings = variables.get("RadiusList").split(",");
     BOOST_FOREACH(QString radius, radiusStrings) {
       radiusList.push_back(radius.toFloat());
     }

     std::vector<int> numberList;
     QStringList numberStrings = variables.get("NumberList").split(",");
     BOOST_FOREACH(QString number, numberStrings) {
       numberList.push_back(toInt(number));
     }
     const float dMax = variables.get("DMax", "5.85").toFloat();
     const float dMin = variables.get("DMin", "8.2").toFloat();

     std::vector<int> indexChange;
     if (!variables.get("IndexChange", "").isEmpty()) {
       QStringList indexStrings = variables.get("IndexChange").split(",");
       BOOST_FOREACH(QString index, indexStrings) {
         indexChange.push_back(toInt(index));
       }
     }
     m_algorithm = BRISKType::create(radiusList, numberList, dMax, dMin, indexChange);
   }
   // Constructs the algorithm with the input variables
   else {
     const int thresh = toInt(variables.get("Threshold"));
     const int octaves = toInt(variables.get("NOctaves"));
     const float patternScale = variables.get("PatternScale").toFloat();  
   
     m_algorithm = BRISKType::create(thresh, octaves, patternScale);
   }
    m_variables.merge(variables);
  }


  /**
   * Destroys the algorithm
   */
  BRISKAlgorithm::~BRISKAlgorithm() { }
 

  /**
   * Sets up the algorithm parameters with default values. 
   * 
   * @return PvlFlatMap Algorithm parameters and their default values. 
   */
  PvlFlatMap BRISKAlgorithm::setupParameters() {
    PvlFlatMap variables;
    variables.add("Threshold",    "30");
    variables.add("NOctaves",     "3");
    variables.add("PatternScale", "1.0");
    m_variables = variables;
    return (m_variables);
  }


  /**
   * Returns a description of the algorithm.
   * 
   * @return @b QString A description of the algorithm.
   */
  QString BRISKAlgorithm::description() const {
    QString desc = "The OpenCV BRISK Feature2D detector/extractor algorithm."
                   " See the documentation at "
     "http://docs.opencv.org/3.1.0/de/dbf/classcv_1_1BRISK.html";
    return (desc);
  }


  /**
   * Creates an instance of the algorithm.
   * 
   * @param cvars  The variables and values the algorithm will use.
   *               Variables that are not included will be set to their default.
   * @param config The config string used to construct cvars.
   */
  Feature2DAlgorithm *BRISKAlgorithm::create(const PvlFlatMap &vars, const QString &config) {
    return ( new BRISKAlgorithm(vars, config) );
  }


  /**
   * Returns true if the algorithm has a detector. 
   *  
   * @return @b true if the algorithm has a detector. 
   */
  bool BRISKAlgorithm::hasDetector() const { 
    return true; 
  }


  /**
   * Returns true if the algorithm has an extractor. 
   *  
   * @return @b true if the algorithm has an extractor. 
   */
  bool BRISKAlgorithm::hasExtractor() const { 
    return true; 
  }


  /**
   * Returns true if the algorithm has a matcher. 
   *  
   * @return @b true if the algorithm has a matcher. 
   */
  bool BRISKAlgorithm::hasMatcher() const { 
    return false; 
  }


  /**
   * Returns the variables and their values used by the algorithm.
   * 
   * @return @b PvlFlatMap The variables and their values as keyword, value pairs.
   */
  PvlFlatMap BRISKAlgorithm::getAlgorithmVariables( ) const {
    return (variables());
  }


/**
 * @brief Set parameters as provided by the variables
 * 
 * @param variables Container of parameters to set
 * 
 * @return @b int Always -1, variables cannot be set after initialization.
 * 
 * @throws IException::Programmer "BRISKAlgorithm does not have the ability
 *                                 to set algorithm parameters."
 */
  int BRISKAlgorithm::setAlgorithmVariables(const PvlFlatMap &variables) {
    QString msg = "BRISKAlgorithm does not have the ability to set algorithm parameters.";
    throw IException(IException::Programmer, msg, _FILEINFO_);

    return (-1);
  }

};  // namespace Isis
