import pytest
import numpy as np
import os
import unittest
from unittest.mock import PropertyMock, patch

import json

import ale
from ale import util
from ale.formatters.formatter import to_isd
from ale.drivers.rosetta_drivers import RosettaVirtisIsisLabelNaifSpiceDriver

from conftest import get_image_label, get_image_kernels, convert_kernels, compare_dicts, get_isd

kerns = ['DE405.BSP', '21_LUTETIA_2004_2016.BSP', 'ORHS_______________00109.BSP',
         'ORHR_______________00122.BSP', 'ROS_VIRTIS_ZERO_V1.BC',
         'ATNR_P040302093352_00127.BC', 'ROS_V38.TF', 'PCK00010.TPC',
         'ROS_LUTETIA_RSOC_V03.TPC', 'ROS_LUTETIA_RSOC_V03.TF',
         'ROS_VIRTIS_V14.TI', 'virtisAddendum_v002.ti', 'naif0012.tls',
         'ROS_160929_STEP.TSC']

opt_ang = [-0.0016759493628859521, -0.0016759493628859521, -0.001934056133738587,
            -0.0021781834811647137, -0.002436241290910525, -0.002680418438204356,
            -0.0029248199369388004, -0.0031825721682645918, -0.003426778606181821,
            -0.0036845626852124807, -0.0036845626852124807, -0.0036845626852124807,
            -0.0036845626852124807, -0.0036845626852124807, -0.0036845626852124807,
            -0.0036845626852124807, -0.00419233, -0.0036845626852124807,
            -0.0016759493628859521, -0.001934056133738587, -0.0021781834811647137,
            -0.002436241290910525, -0.002680418438204356, -0.0029248199369388004,
            -0.0031825721682645918, -0.003426778606181821, -0.0036845626852124807,
            -0.0036845626852124807, -0.0036845626852124807, -0.0036845626852124807,
            -0.0036845626852124807, -0.0036845626852124807,-0.00354844,
            -0.0036845626852124807, -0.0036845626852124807, -0.0016759493628859521,
            -0.001934056133738587, -0.0021781834811647137, -0.002436241290910525,
            -0.002680418438204356, -0.0029248199369388004, -0.0031825721682645918,
            -0.003426778606181821, -0.0036845626852124807, -0.0036845626852124807,
            -0.0036845626852124807, -0.0036845626852124807, -0.0036845626852124807,
            -0.00372128, -0.0036845626852124807, -0.0036845626852124807,
            -0.0036845626852124807, -0.0016759493628859521, -0.001934056133738587,
            -0.0021781834811647137, -0.002436241290910525, -0.002680418438204356,
            -0.0029248199369388004, -0.0031825721682645918, -0.003426778606181821,
            -0.0036845626852124807, -0.0036845626852124807, -0.0036845626852124807,
            -0.0036845626852124807, -0.00367381, -0.0036845626852124807,
            -0.0036845626852124807, -0.0036845626852124807, -0.0036845626852124807,
            -0.0016759493628859521, -0.001934056133738587, -0.0021781834811647137,
            -0.002436241290910525, -0.002680418438204356, -0.0029248199369388004,
            -0.0031825721682645918, -0.003426778606181821, -0.0036845626852124807,
            -0.0036845626852124807, -0.0036845626852124807, -0.00369084,
            -0.0036845626852124807, -0.0036845626852124807, -0.0036845626852124807,
            -0.0036845626852124807, -0.0036845626852124807, -0.0016759493628859521,
            -0.001934056133738587, -0.0021781834811647137, -0.002436241290910525,
            -0.002680418438204356, -0.0029248199369388004, -0.0031825721682645918,
            -0.003426778606181821, -0.0036845626852124807, -0.0036845626852124807,
            -0.00367021, -0.0036845626852124807, -0.0036845626852124807,
            -0.0036845626852124807, -0.0036845626852124807, -0.0036845626852124807,
            -0.0036845626852124807, -0.0016759493628859521, -0.001934056133738587,
            -0.0021781834811647137, -0.002436241290910525, -0.002680418438204356,
            -0.0029248199369388004, -0.0031825721682645918, -0.003426778606181821,
            -0.0036845626852124807,-0.00373568, -0.0036845626852124807,
            -0.0036845626852124807, -0.0036845626852124807, -0.0036845626852124807,
            -0.0036845626852124807, -0.0036845626852124807, -0.0036845626852124807,
            -0.0016759493628859521, -0.001934056133738587, -0.0021781834811647137,
            -0.002436241290910525, -0.002680418438204356, -0.0029248199369388004,
            -0.0031825721682645918, -0.003426778606181821,-0.00360218,
            -0.0036845626852124807, -0.0036845626852124807, -0.0036845626852124807,
            -0.0036845626852124807, -0.0036845626852124807, -0.0036845626852124807,
            -0.0036845626852124807, -0.0036845626852124807, -0.0016759493628859521,
            -0.001934056133738587, -0.0021781834811647137, -0.002436241290910525,
            -0.002680418438204356, -0.0029248199369388004, -0.0031825721682645918,
            -0.0032917, -0.003426778606181821, -0.0036845626852124807,
            -0.0036845626852124807, -0.0036845626852124807, -0.0036845626852124807,
            -0.0036845626852124807, -0.0036845626852124807, -0.0036845626852124807,
            -0.0036845626852124807, -0.0016759493628859521, -0.001934056133738587,
            -0.0021781834811647137, -0.002436241290910525, -0.002680418438204356,
            -0.0029248199369388004,-0.00305905, -0.0031825721682645918,
            -0.003426778606181821, -0.0036845626852124807, -0.0036845626852124807,
            -0.0036845626852124807, -0.0036845626852124807, -0.0036845626852124807,
            -0.0036845626852124807, -0.0036845626852124807, -0.0036845626852124807,
            -0.0016759493628859521, -0.001934056133738587, -0.0021781834811647137,
            -0.002436241290910525, -0.002680418438204356,-0.0028401,
            -0.0029248199369388004]

hk_ephemeris_times = [331981320.7458468, 331981350.3517909, 331981370.36564744,
                    331981390.37948877, 331981410.39358944, 331981430.361395,
                    331981450.37525153, 331981470.3890776, 331981490.35715777,
                    331981510.3709991, 331981530.3848556, 331981550.3529358,
                    331981570.3667618, 331981590.3806184, 331981610.3947343,
                    331981629.25990915, 331981650.53505737, 331981670.3764743,
                    331981690.3781391, 331981709.2628604, 331981730.36004525,
                    331981750.37390184, 331981770.3877279, 331981790.3558081,
                    331981810.36966467, 331981830.38350594, 331981850.3973473,
                    331981870.36542743, 331981890.379284, 331981910.39312524,
                    331981930.36120546, 331981949.2611856, 331981970.54762524,
                    331981990.38902694, 331982010.3570919, 331982030.3587109,
                    331982050.3725675, 331982070.386424, 331982090.40028054,
                    331982109.2605268, 331982130.382202, 331982150.3960738,
                    331982170.3641235, 331982189.14984584, 331982210.39183664,
                    331982230.35990155, 331982250.37375814, 331982270.3876146,
                    331982290.5601626, 331982310.40157956, 331982330.3696598,
                    331982349.2620563, 331982370.3851506, 331982390.3992513,
                    331982410.36708736, 331982430.3809439, 331982450.39476997,
                    331982470.36285007, 331982490.37670666, 331982510.39056325,
                    331982530.3586434, 331982550.3725, 331982570.38635653,
                    331982590.40045726, 331982610.5269542, 331982629.2605677,
                    331982650.3822582, 331982670.39611477, 331982690.3641644,
                    331982710.3658292, 331982730.37968576, 331982750.39354235,
                    331982770.3616225, 331982790.3754791, 331982809.26267236,
                    331982830.4031769, 331982850.37125707, 331982870.38511366,
                    331982890.3989702, 331982909.2604524, 331982930.53959835,
                    331982950.3810459, 331982970.39488715, 331982990.3629674,
                    331983010.3768086, 331983030.3906652, 331983050.39232993,
                    331983070.4061865, 331983090.3742667, 331983109.26257384,
                    331983129.263964, 331983150.37004477, 331983170.3839013,
                    331983190.3977884, 331983210.3658685, 331983230.3797251,
                    331983250.5522731, 331983270.39369005, 331983290.40754664,
                    331983309.2613384, 331983330.3894834, 331983350.4033857,
                    331983370.3714354, 331983390.3731001, 331983410.3869567,
                    331983429.26386553, 331983450.36889344, 331983470.38275,
                    331983490.3965913, 331983510.36467147, 331983530.3785586,
                    331983550.3924151, 331983570.56496304, 331983590.4066699,
                    331983610.37449074, 331983630.38834727, 331983649.26402014,
                    331983670.370284, 331983690.3841406, 331983710.3980124,
                    331983730.39966196, 331983750.36774206, 331983770.38159865,
                    331983790.3954552, 331983810.4093423, 331983830.37742245,
                    331983850.3912638, 331983869.2643274, 331983890.5319072,
                    331983910.37340045, 331983930.38719594, 331983950.401083,
                    331983970.3691632, 331983990.38301975, 331984009.26351446,
                    331984030.4107482, 331984050.37882835, 331984070.3804779,
                    331984090.39433444, 331984110.4082368, 331984129.26146394,
                    331984150.39015824, 331984170.40403, 331984190.37211025,
                    331984210.5446887, 331984230.3861057, 331984249.2638385,
                    331984269.2652286, 331984290.3819295, 331984310.3957708,
                    331984330.4096731, 331984350.37773806, 331984370.39160985,
                    331984389.2643988, 331984410.4071312, 331984430.3752114,
                    331984450.3890832, 331984470.4029398, 331984490.37103516,
                    331984510.38489175, 331984530.55747026, 331984550.3988872,
                    331984570.41303366, 331984590.3808545, 331984610.3947263,
                    331984630.40858287, 331984650.376663, 331984670.3905501,
                    331984690.40439135, 331984710.3725021, 331984730.3863586,
                    331984750.3880082, 331984770.4018953, 331984790.4157518,
                    331984810.3838473, 331984830.3977038, 331984850.5702823,
                    331984869.6100787]

image_dict = {
    "i1_00237330013": get_isd("rosetta_virtis")
}

@pytest.fixture(scope='module')
def test_kernels():
    updated_kernels = {}
    binary_kernels = {}
    for image in image_dict.keys():
        kernels = get_image_kernels(image)
        updated_kernels[image], binary_kernels[image] = convert_kernels(kernels)
    yield updated_kernels
    for kern_list in binary_kernels.values():
        for kern in kern_list:
            os.remove(kern)

@pytest.mark.parametrize("label_type", ['isis3'])
@pytest.mark.parametrize("image", image_dict.keys())
def test_load(test_kernels, label_type, image):
    with patch('ale.drivers.rosetta_drivers.RosettaVirtisIsisLabelNaifSpiceDriver.hk_ephemeris_time', hk_ephemeris_times),  \
    patch('ale.drivers.rosetta_drivers.RosettaVirtisIsisLabelNaifSpiceDriver.optical_angle', opt_ang):

        label_file = get_image_label(image, label_type)
        isd_str = ale.loads(label_file, props={'kernels': test_kernels[image]})
        isd_obj = json.loads(isd_str)
        compare_dict = image_dict[image]
        assert compare_dicts(isd_obj, compare_dict) == []

class test_rosetta_virtis_isis_naif(unittest.TestCase):


    def setUp(self):
        label = get_image_label("i1_00237330013", "isis3")
        self.driver = RosettaVirtisIsisLabelNaifSpiceDriver(label)

    def test_instrument_id(self):
        assert self.driver.instrument_id == "ROS_VIRTIS-M_IR"

    def test_sensor_model_version(self):
        assert self.driver.sensor_model_version == 1

    def test_spacecraft_name(self):
        assert self.driver.spacecraft_name == "ROSETTA"

    def test_ephemeris_start_time(self):
        with patch('ale.drivers.rosetta_drivers.RosettaVirtisIsisLabelNaifSpiceDriver.hk_ephemeris_time', hk_ephemeris_times):
            assert self.driver.ephemeris_start_time == 331981320.7448468

    def test_ephemeris_stop_time(self):
        with patch('ale.drivers.rosetta_drivers.RosettaVirtisIsisLabelNaifSpiceDriver.hk_ephemeris_time', hk_ephemeris_times):
            assert self.driver.ephemeris_stop_time == 331984869.6110787

    def test_line_exposure_duration(self):
        assert self.driver.line_exposure_duration == 0.002
    
    def test_line_summing(self):
        assert self.driver.line_summing == 1
    
    def test_sample_summing(self):
        assert self.driver.sample_summing == 1

    def test_scan_rate(self):
        assert self.driver.scan_rate == 20
    
    def test_odtk(self):
        assert self.driver.odtk == [0.0, 0.038, 0.38]
    
    def test_sensor_frame_id(self):
        with patch('ale.drivers.rosetta_drivers.RosettaVirtisIsisLabelNaifSpiceDriver.has_articulation_kernel', True), \
        patch('ale.base.data_naif.spice.bods2c', return_value=12345) as bods2c:
            assert self.driver.sensor_frame_id == 12345
            bods2c.assert_called_with('ROS_VIRTIS-M_IR')
    
    def test_is_calibrated(self):
        assert self.driver.is_calibrated is False

    def test_has_articulation_kernel(self):
        with patch('ale.drivers.rosetta_drivers.RosettaVirtisIsisLabelNaifSpiceDriver.kernels', kerns):
            assert self.driver.has_articulation_kernel is False
