#!/usr/bin/perl
use Getopt::Long;
use FileHandle;		# Buffer Flushing
use File::Basename;	# File Parsing
use Cwd;		# Current Working Directory Module
#
# Program: vims_titan_cube.pl
# By: Jason Soderblom
# Date: 24-Jan-2007
#
# Purpose: 
#          
#This procedure runs a VIMS .ir.cub through 
#  -spiceinit [default will not run spiceinit], if -spice is entered, all input files
#             will be run through spiceinit
#  -phocube  (ema, inc, phase)
#  -cubeavg  (runs 8 times, once for each of the 8 titan methane windows)
#  -cubit  (combines 8 planes & backplane info to a single cube)
#  -cam2map  (only runs if -simp flag is set...for level1 input)
#  -grid     (creates a grid on the 'map' output only)
#  -isis2std  (creates a jpg file of the simp version only)
#
#NOTES:
#-spiceinit failures will be listed in a nospice.log file
#
#
# Original program, vimslev0.pl, written by TBecker 27-Aug-2007
#
# 2008-Apr-14 TLB - Renamed Jason's vims_titan_cube.pl to vims_mwinavg.pl
# 2011-Mar-02 TLB - Modified to allow level2 input images for the cubeavg output
#                 - Check channel for IR; moved camstats to simp option
#------------------------------------------------------------------------
my ($progname) = ($0 =~ m#([^/]+)$#);  # get the name of this program

my $usage = "
Help: $progname [-h] or [-d=S02_T0] or [-l=input.lis] or [-f=input.cub] [-simp -map]
where
   -h                    Help, prints documentation only
   -dir=                 Process all *.ir.cub files in a directory (shortcut: -d=)
   -list=                List of input cubes (shortcut: -l=)
   -from=                Single Image (shortcut: -f=)
   -spice                Default does NOT run spiceinit; 
   -simp                 Generate individual Simps-default res/lat/lon (Optional for lev1 input)
   -nojpg                Do not create simp color jpegs (Optional for lev1 input)
   -map                  Generate individual Simps at same resolution/global range (Optional for lev1 input)
   -grid                 Generate a lat/lon grid to the Simp map (Optional for lev1 input)

Hyphens are REQUIRED; 
-dir, -list or -from= one of these parameters is required for specified input

*Input is required to be a calibrated IR cube (if no spiceinit applied; enter -spice)
*Input can be a level1 or level2 image
 
*Run on an entire directory
     Command:  $progname -dir=S02_T0 -simp -map (assumes *.ir.cub input files)
*Run a list of input cubes
     Command:  $progname -list=single_column_input.lis -simp -map
*Run a single file 
     Command:  $progname -from=dir/vims_input.cub 

***All output files go to current directory and will be overwritten if they already exist
";

my $summary = "
Help:  $progname [-h] ~or~ [-dir=S02_T0]
where
   -h                    Help, prints documentation only
   -dir=                 Process all *.ir.cub files in a directory <required or> (shortcut: -d=)
   -list=                List of input cubes  <required or> (shortcut: -l=)
   -from=                Single Image <required or> (shortcut: -f=)
   -spice                [default=no spiceinit] this switch includes spiceinit
   -simp                 Generate individual Simp, default res/lat/lon (Optional for lev1 input)
   -nojpg                Do not create jpg color of Simp images(Optional for lev1 input; use with -simp)
   -map                  Generate individual Simps at same res/global range (Optional for lev1 input)
   -grid                 Generate a lat/lon grid for Map file (Optional for lev1 input; use with -map)

Hyphens are REQUIRED
-dir, -list or -from= one of these parameters is required for specified input

*Input is required to be calibrated IR cube (if no spiceinit applied; enter -spice)
*Input can be a level1 or level2 image

***All output files go to current directory and will be overwritten if they already exist

Command:   $progname -dir=S02_T0 -simp -map ~or~ $progname -d=S02_T0 -simp -map
";
#####################################################################
#  MAIN APPLICATION SECTION
#####################################################################

if (@ARGV == 0)
  {
    print "$summary\n";
    exit;
  }

my $help = '';          # Help option

# Get options
my $opt     = GetOptions ( "h"         => \$help,
                           "list=s"    => \$list,
                           "from=s"    => \$single,
                           "dir=s"     => \$dir,
                           "spice"     => \$spice,
                           "simp"      => \$simp,
                           "nojpg"     => \$nojpg,
                           "map"       => \$global,
		           "grid"      => \$grid);


if (@ARGV != 0)
  {
    print "$summary\n";
    print "[ERROR] Unrecognized parameter!\n";
    print "Remember hyphens before option parameters!\n"; 
    exit;  
  }

if (!$opt)
  {
    print "$usage\n";
    print "Remember hyphens before option parameters!\n"; 
    exit;
  }

if ($help)
  {
    print "$usage\n";
    exit;
  } 

if (defined $dir)
  {
    if (defined $list || defined $single)
      {
	print "[ERROR] Select only one option; -dir or -list or -from\n";
        print "Help Command:  $progname -h\n";
	exit;
       }
    $dir =~ s/\s+//;
    chomp $dir;

    $currdir = cwd();

    $work_dir = $dir;

    if (!-e $work_dir) 
      {
        print "\n[ERROR] Working Directory:  \n  $currdir/$work_dir  does NOT exist!\n\n";
        exit;
      }
   }  #defined $dir


if ($work_dir eq "" && !$list && !$single)
  {
    print "\n[Error] Unrecognized parameters..Did you forget ";
    print "to the hyphen before the \"dir\" parameter? \n Command: $progname -dir=S02_T0\n\n";
    exit;
  }


#Set up desired bands for extract planes to average

$p1 = "+4";       #plane 4 eq <channel 4>
$p2 = "+12-13";   #plane 2 eq <channel 12-13>
$p3 = "+25-26";   #plane 3 eq <channel 25-26>
$p4 = "+42-45";   #plane 4 eq <channel 42-45>
$p5 = "+69-73";   #plane 5 eq <channel 69-73>
$p6 = "+110-112"; #plane 6 eq <channel 110-112>
$p7 = "+116-117"; #plane 7 eq <channel 116-117>
$p8 = "+243-253"; #plane 8 eq <channel 243-253>

#Set up desired bands for the jpg color files
$rb = "+5";
$gb = "+4";
$bb = "+3";

#Map projection settings for global map files
$lonsys = 360;
$proj = "SimpleCylindrical";
$clat = 0.0;
$clon = 180.0;
$pix = 8000.0;   #meters/pixel  (8 km/px)
$slat = -90.0;
$elat = 90.0;
$slon = 0.0;
$elon = 360.0;

$pd_const = 44942;    #meters/pixel based on Titan's diameter

if ($global)  #global map output files desired
  {
    $map = "simp.map";
    if (-e $map) { unlink $map; }
    open (MFL,">>$map");
    print MFL "Group = Mapping\n";
    print MFL "  ProjectionName  = $proj\n";
    print MFL "  CenterLongitude = $clon\n";
    print MFL "  PixelResolution = $pix\n <Meters/pixel>"; 
    print MFL "  LongitudeDomain = $lonsys\n";
    print MFL "  MinimumLatitude = $slat\n";
    print MFL "  MaximumLatitude = $elat\n";
    print MFL "  MinimumLongitude= $slon\n";
    print MFL "  MaximumLongitude= $elon\n";
    print MFL "End_Group\n";
    print MFL "End\n";
  }
    
#delete print file for now...
if (-e print.prt) { unlink print.prt; }

$date = scalar localtime;

#------------------------------------------------------------------
# Continue with the directory option and load img array with
# the directory listing of .cub(s)
#------------------------------------------------------------------

if (defined $dir)
  {  
     @img = glob("$work_dir/*ir*.cub");
     if (@img == 0)
       {
         print "[ERROR] No input cubes to process!! .cub doesn't exist in $work_dir/ ?\n\n";
         exit;
        }
     
     $prt = "$work_dir/" . "vims_mwinavg.prt";
     $err = "$work_dir/" . "vims_mwinavg.err";
     $log = "$work_dir/" . "vims_mwinavg.log";
     $spicelog = "$work_dir/" . "vims_nospice.log";

     print "\n Working Directory: $work_dir \n\n";

  }

#------------------------------------------------------------------
# List Option; load img array with input list
#------------------------------------------------------------------

if (defined $list)
  {
    if (defined $dir || defined $single)
      {
	print "[ERROR] Select only one option; -dir or -list or -from\n";
        print "Help Command:  $progname -h\n";
	exit;
      }

    chomp $list;

    if (!-e $list)
      {
	print "\n[ERROR] Input list: $list does not exist\n";
	exit;
      }
    else
      {
	    open(ILIST,$list) or die ("\n[ERROR] Problem opening input list: $list, $!\n");
        @img = <ILIST>;
      }

    ($iroot,$a) = split('\.',$list);
    $prt = $iroot . "_vims_mwinavg.prt";
    $err = $iroot . "_vims_mwinavg.err";
    $log = $iroot . "_vims_mwinavg.log";
    $spicelog = $iroot . "_vims_nospice.log";

    print "\n List Input: $list \n\n";
  }


#------------------------------------------------------------------
# Single image option; load img array with input file
#------------------------------------------------------------------

if (defined $single)
  {
    if (defined $dir || defined $list)
      {
	    print "[ERROR] Select only one option; -dir or -list or -from\n";
        print "Help Command:  $progname -h\n";
	exit;
      }
    chomp $single;
    if (!-e $single)
      {
	print "[ERROR] $single input file does not exist\n\n";
	exit;
      }
    @img = $single;

    ($iroot,$a) = split('\.',$single);
    $prt = $iroot . "_vims_mwinavg.prt";
    $err = $iroot . "_vims_mwinavg.err";
    $log = $iroot . "_vims_mwinavg.log";
    $spicelog = $iroot . "_vims_nospice.log";

    print "\n Single input file: $single \n\n";
  }

$prtlink = "ln -s -f $prt print.prt";
system($prtlink);

open ERR, ">> $err";
print ERR "\n------------------------------------------------------\n";
print ERR " Errors encountered during VIMS methane band processing    \n";
print ERR " Date:  $date\n";
print ERR "------------------------------------------------------\n";

open LOG, ">> $log";
print LOG "\n-------------------------------------------------------------------\n";
print LOG " Note worthy activity encountered during VIMS methane band processing\n";
print LOG " Date:  $date\n";
print LOG "---------------------------------------------------------------------\n";

if ($single) { print LOG "\nSingle input file: $single \n\n";   }
if ($dir) { print LOG "\nWorking Directory: $work_dir \n\n"; }
if ($list) { print LOG "\nList Input: $list \n\n"; }

print LOG "Bands selected for averaging: \n";
print LOG "  p1=$p1, p2=$p2, p3=$p3, p4=$p4, p5=$p5, p6=$p6, p7=$p7, p8=$p8\n\n";
print LOG "RGB Bands for Jpeg files:\n";
print LOG "  Red=$rb  Green=$gb   Blue=$bb    \n\n";

print LOG "Pixel/Deg Constant: $pd_const\n\n";
print LOG "[if px/dg >=  0 and px/dg < 2   => Scale=4]\n";
print LOG "[if px/dg >=  2 and px/dg < 4   => Scale=8]\n";
print LOG "[if px/dg >=  4 and px/dg < 8   => Scale=16]\n";
print LOG "[if px/dg >=  8 and px/dg < 16  => Scale=32]\n";
print LOG "[if px/dg >= 16 and px/dg < 32  => Scale=64]\n";
print LOG "[if px/dg >= 32 and px/dg < 64  => Scale=128]\n";
print LOG "[if px/dg >= 64 and px/dg < 128 => Scale=256]\n";
print LOG "[if px/dg >= 128                => Scale=512]\n\n";

if ($spice)
  { 
    open SPLOG, ">> $spicelog";
    print SPLOG "\n-------------------------------------\n";
    print SPLOG " Files that had spiceinit problems    \n";
    print SPLOG " Date:  $date\n";
    print SPLOG "-------------------------------------\n";
  }

SKIP: for ($i=0; $i<=$#img; $i++)
  {
    chomp $img[$i];

    if (!-e $img[$i]) { 
      print "\n[ERROR] $img[$i] does not exist\n";
      exit;
    }

   $channel = `getkey from=$img[$i] grpname=Instrument keyword=Channel`;
   chomp $channel; 
   if ($channel ne "IR") {
      print "\n[ERROR] Channel is not IR ($channel) for $img[$i]\n";
      exit;
    }
    
    $cmd = "getkey  from=$img[$i] grpname=Mapping keyword=ProjectionName";
    $result = system($cmd);
    if ($result == 0) {
      $lev2 = 1;     #input image has a mapping group; its a level2 img
      print LOG "$img[$i] is a Level2 image\n";
     }
    else {
	print "Level's check...$img[$i] is a Level1 image\n";
        $lev2 = 0;
     }

    $ir = $img[$i];

#    ($froot,$a,$b) = split('\.',$img[$i]);
#    @irchk = split('\.',$froot);
#    $ir = $froot.".ir.cub";

    ($froot,$a) = split('\.cub',$img[$i]);
    $base = basename($froot);
    chomp $base;

#print "input=$img[$i]\n";
#print "froot=$froot\n";
#print "base=$base\n";

        if ($spice)   #Default is to not run spiceinit; assumed to be run previously in lev0
	  {
            $cmd = "spiceinit from=$ir ckrecon=yes ckpredicted=yes spkrecon=yes spkpredicted=yes";
#print "$cmd\n";
            $result = system($cmd);
            if ($result != 0)
              {
                print SPLOG "Spiceinit failed on $ir\n";
		next SKIP;
               }
	   }

	 $stats = $base . ".stats";         
         $geo = $base . "_geo.cub";
         $ir_input = $ir . "+71";         #pass 71st band only

         if (-e $geo) { unlink $geo; }
         $cmd = "phocube from=$ir_input to=$geo latitude=no longitude=no";
#print "$cmd\n";
         $result = system($cmd);
         if ($result != 0)
           {
             print ERR "[ERROR] phocube failed on $ir\n";
             next SKIP;
           }
         else
           {
             $make_planes = 1;
MAKE_PLANES: while ($make_planes)
               {
                 $plane1 = $base . "_plane1.cub";
                 $ir_input = $ir . $p1;        
                 if (-e $plane1) { unlink $plane1; }
                 $cmd = "cubeavg from=$ir_input to=$plane1 bandbin=compute";
#print "$cmd\n";
                 $result = system($cmd);
                 if ($result != 0)
                   {
                     print ERR "[ERROR] cubeavg failed on $ir plane1\n";
                     $make_planes = 0;
                     goto MAKE_PLANES;
                    }

                 $plane2 = $base . "_plane2.cub";
                 $ir_input = $ir . $p2; 
                 if (-e $plane2) { unlink $plane2; }
                 $cmd = "cubeavg from=$ir_input to=$plane2 bandbin=compute";
#print "$cmd\n";
                 $result = system($cmd);
                 if ($result != 0)
                   {
                     print ERR "[ERROR] cubeavg failed on $ir plane2\n";
                     $make_planes = 0;
                     goto MAKE_PLANES;
                    }

                 $plane3 = $base . "_plane3.cub";
                 $ir_input = $ir . $p3; 
                 if (-e $plane3) { unlink $plane3; }
                 $cmd = "cubeavg from=$ir_input to=$plane3 bandbin=compute";
#print "$cmd\n";
                 $result = system($cmd);
                 if ($result != 0)
                   {
                     print ERR "[ERROR] cubeavg failed on $ir plane3\n";
                     $make_planes = 0;
                     goto MAKE_PLANES;
                   }

                 $plane4 = $base . "_plane4.cub";
                 $ir_input = $ir . $p4; 
                 if (-e $plane4) { unlink $plane4; }
                 $cmd = "cubeavg from=$ir_input to=$plane4 bandbin=compute";
#print "$cmd\n";
                 $result = system($cmd);
                 if ($result != 0)
                   {
                     print ERR "[ERROR] cubeavg failed on $ir plane4\n";
                     $make_planes = 0;
                     goto MAKE_PLANES;
                   }

                 $plane5 = $base . "_plane5.cub";
                 $ir_input = $ir . $p5; 
                 if (-e $plane5) { unlink $plane5; }
                 $cmd = "cubeavg from=$ir_input to=$plane5 bandbin=compute";
#print "$cmd\n";
                 $result = system($cmd);
                 if ($result != 0)
                   {
                     print ERR "[ERROR] cubeavg failed on $ir plane5\n";
                     $make_planes = 0;
                     goto MAKE_PLANES;
                   }

                 $plane6 = $base . "_plane6.cub";
                 $ir_input = $ir . $p6;
                 if (-e $plane6) { unlink $plane6; }
                 $cmd = "cubeavg from=$ir_input to=$plane6 bandbin=compute";
#print "$cmd\n";
                 $result = system($cmd);
                 if ($result != 0)
                   {
                     print ERR "[ERROR] cubeavg failed on $ir plane6\n";
                     $make_planes = 0;
                     goto MAKE_PLANES;
                    }

                 $plane7 = $base . "_plane7.cub";
                 $ir_input = $ir . $p7; 
                 if (-e $plane7) { unlink $plane7; }
                 $cmd = "cubeavg from=$ir_input to=$plane7 bandbin=compute";
#print "$cmd\n";
                 $result = system($cmd);
                 if ($result != 0)
                   {
                     print ERR "[ERROR] cubeavg failed on $ir plane7\n";
                     $make_planes = 0;
                     goto MAKE_PLANES;
                   }

                 $plane8 = $base . "_plane8.cub";
                 $ir_input = $ir . $p8; 
                 if (-e $plane8) { unlink $plane8; }
                 $cmd = "cubeavg from=$ir_input to=$plane8 bandbin=compute";
#print "$cmd\n";
                 $result = system($cmd);
                 if ($result != 0)
                   {
                     print ERR "[ERROR] cubeavg failed on $ir plane8\n";
                     $make_planes = 0;
                     goto MAKE_PLANES;
                   }

                 $make_planes = 0;

                 $planelist = $base . "_ir_planes.lis";
                 if (-e $planelist) { unlink $planelist; }
                 open LIST, ">> $planelist";
                 print LIST "$plane1\n";
                 print LIST "$plane2\n";
                 print LIST "$plane3\n";
                 print LIST "$plane4\n";
                 print LIST "$plane5\n";
                 print LIST "$plane6\n";
                 print LIST "$plane7\n";
                 print LIST "$plane8\n";
                 print LIST "$geo\n";
                 close LIST;

                 $mwinavg = $base . "_mwinavg.cub";
                 if (-e $mwinavg) { unlink $mwinavg; }
                 $cmd = "cubeit fromlist=$planelist to=$mwinavg";
#print "$cmd\n";
                 $result = system($cmd);
                 if ($result != 0)
                   {
                     print ERR "[ERROR] cubeit failed on $ir\n";
                   }
                 else
		  {
		    unlink ($geo);
                  }

                 if ($simp) {
                     if ($lev2 == 1) {
                        print "[NOTICE] cam2map will not be run. $mwinavg is a level2 image already\n";
		     }
                     else {
   	               $cmd = "camstats from=$ir_input to=$stats";
	               system($cmd) == 0 || die "[ERROR] camstats bombed on $ir_input\n";
       
	               $minres = `getkey from=$stats grpname=Resolution keyword=ResolutionMinimum`;
                       chomp $minres;
                       $maxres = `getkey from=$stats grpname=Resolution keyword=ResolutionMaximum`;
                       chomp $maxres;
                       $avgres = `getkey from=$stats grpname=Resolution keyword=ResolutionAverage`;
                       chomp $avgres;

		       $pix_deg = int($pd_const / $minres);
     
                       if ($pix_deg >= 0 && $pix_deg < 2) { $res=4; }
                       if ($pix_deg >= 2 && $pix_deg < 4) { $res=8; }
                       if ($pix_deg >= 4 && $pix_deg < 8) { $res=16; }
                       if ($pix_deg >= 8 && $pix_deg < 16) { $res=32; }
                       if ($pix_deg >= 16 && $pix_deg < 32) { $res=64; }
                       if ($pix_deg >= 32 && $pix_deg < 64) { $res=128; }
                       if ($pix_deg >= 64 && $pix_deg < 128) { $res=256; }
                       if ($pix_deg >= 128) { $res=512; }

                       print LOG "$mwinavg MinRes=$minres PixDeg==$pix_deg => Scale=$res\n";

                       $cyl = $base . "_mwinavg_simp.cub";
                       if (-e $cyl) { unlink $cyl; }
                       $cmd = "cam2map from=$mwinavg map=\$ISIS3DATA/base/templates/maps/simplecylindrical.map to=$cyl pixres=ppd res=$res defaultrange=CAMERA" ;
#print "$cmd\n";
                       $result = system($cmd);
                       if ($result != 0) {
                           print ERR "[ERROR] cam2map failed on $mwinavg\n";
                       }
                       else {
                        if (!$nojpg)
                          {
                            $jpg = $base . "_mwinavg_simp.jpg";
                            $red   = $cyl . $rb;
                            $green = $cyl . $gb;
                            $blue  = $cyl . $bb;
                            if (-e $jpg) { unlink $jpg; }
                            $cmd = "isis2std red=$red green=$green blue=$blue to=$jpg mode=rgb format=jpeg";
                            $result = system($cmd);
                            if ($result != 0) {
                                print ERR "[ERROR] isis2std (jpg) failed on $ir\n";
                              }
                          } #end-if ($nojpg)
		      }

#                     else {
#remove gridding the simp for now
#                         if ($add_grids) {
#                             $cyl_grid = $base . "_mwinavg_simp_gr.cub";
#                             if (-e $cyl_grid) { unlink $cyl_grid; }
#                             $cmd = "grid from=$cyl to=$cyl_grid mode=ground latinc=30 loninc=30";
#                             $result = system($cmd);
#                             if ($result != 0) {
#                                 print ERR "[ERROR] grid failed on $cyl\n";
#                                }
#                             else  {
#                                  if (-e $cyl) { unlink $cyl; }
#                                  if (-e $cyl_grid) { rename $cyl_grid, $cyl; }
#                                }
#                            } #end-if ($add_grids)
#                        } #end-else
		      } #end-else for level's check
                    } #end-if ($simp)

                 if ($global) {
                     if ($lev2 == 1) {
                        print "\n\n[NOTICE] cam2map will not be run. $mwinavg is a level2 image already\n";
		     }
                     else {
		       $cylmap = $base . "_mwinavg_smap.cub";
                       if (-e $cylmap) { unlink $cylmap }
                       $cmd = "cam2map from=$mwinavg map=$map to=$cylmap pixres=map defaultrange=map";
                       $result = system($cmd);
                       if ($result != 0)
                         {
                          print ERR "[ERROR] cam2map for global map failed on $mwinavg\n";
                         }
                       else
                        {
                          if (!$nogrid)
                            {
                             $cyl_grid = $base . "_mwinavg_smap_gr.cub";
                             if (-e $cyl_grid) { unlink $cyl_grid; }
                             $cmd = "grid from=$cylmap to=$cyl_grid mode=ground latinc=30 loninc=30";
                             $result = system($cmd);
                             if ($result != 0)
                               {
                                 print ERR "[ERROR] grid failed on $cylmap/$cyl_grid\n";
                                }
                             $jpgin = $cyl_grid;
                            } #end-if ($nogrid)
                         else {
                             $jpgin = $cylmap;
                           }
                       } #end-else

                       if (!$nojpg) {
                            $jpg = $base . "_mwinavg_smap.jpg";
                            $red   = $jpgin . $rb;
                            $green = $jpgin . $gb;
                            $blue  = $jpgin . $bb;
                            if (-e $jpg) { unlink $jpg; }
                            $cmd = "isis2std red=$red green=$green blue=$blue to=$jpg mode=rgb format=jpeg";
                            $result = system($cmd);
                            if ($result != 0)
                              {
                                print ERR "[ERROR] isis2std (jpg) failed on $jpgin/$jpg\n";
                              }
                          } #end-if ($nojpg)

                    } #end-if of levels check
		  } #end-if ($global)
                 } #end-while

          } #end-else
 if (-e $plane1) { unlink $plane1; }
 if (-e $plane2) { unlink $plane2; }
 if (-e $plane3) { unlink $plane3; }
 if (-e $plane4) { unlink $plane4; }
 if (-e $plane5) { unlink $plane5; }
 if (-e $plane6) { unlink $plane6; }
 if (-e $plane7) { unlink $plane7; }
 if (-e $plane8) { unlink $plane8; }
 if (-e $planelist) { unlink $planelist; }

  } #end-for ($i=0; $i<=$#img; $i++)

unlink "print.prt";
close LOG;
close ERR;
close PRT;
close SPLOG;
exit;
