#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Functions managing the several directory names.
"""

# System modules
import shutil
from pathlib import Path

# Third-party modules
from astropy.io import fits
from astropy.time import Time

# Custom modules
from ..config.constants import (
    DATA_FOLDER_NAME, LOG_DIRECTORY_NAME,
    FILE_PREFIX, LOG_FILE_EXTENSION, LATEST_LOG_FILENAME,
    dateobs, dir_type, ext, fits_folder,
    frame_number, imagetyp, focus_folder, focus_ext
)

PROJECT_ROOT = Path(__file__).parent.parent.parent


def date_folder():
    """Create a date folder string based on astronomical convention
    (changes at midday UTC).
    """
    now = Time.now()
    # If current UTC hour is before midday UTC, the "observing night"
    # belongs to the previous calendar date.
    if now.datetime.hour < 12: 
        folder_date_obj = now.datetime.date() - timedelta(days=1)
    else:
        folder_date_obj = now.datetime.date()
    return folder_date_obj.strftime("%Y-%m-%d")


def frame_folder(header):
    """
    Create a folder depending on the image type in FITS header
    """
    frame = header[imagetyp]
    if isinstance(frame, int):
        folder_name = dir_type[frame]
    else:
        frame_num_list = [v for k, v in frame_number.items() if k in frame]
        if not frame_num_list:
            # Fallback if frame type string is not recognized
            return "unknown_type"
        folder_name = dir_type[frame_num_list[0]]
    return folder_name


def fits_path(header, dry=False):
    """
    Create a fits file path where the file will be stored
    """
    
    root = PROJECT_ROOT / DATA_FOLDER_NAME / fits_folder

    date_str = date_folder() 
    date_path_part = Path(date_str)

    frame_path_part = Path(frame_folder(header))
    path = root / date_path_part / frame_path_part

    if not dry:
        path.mkdir(parents=True, exist_ok=True)
    return path


def get_log_dir_path(dry=False):
    """
    Returns the Path object for the log directory.
    Creates it if it doesn't exist (unless dry=True).
    """
    log_dir = PROJECT_ROOT / DATA_FOLDER_NAME / LOG_DIRECTORY_NAME
    if not dry:
        log_dir.mkdir(parents=True, exist_ok=True)
    return log_dir

def get_dated_log_filepath(date_str, dry=False):
    """
    Constructs a path for an expected date-stamped (rotated) log
    file.  date_str: A string like "YYYY-MM-DD"
    """
    
    log_dir = get_log_dir_path(dry=dry)
    # Filename: OARPAF.YYYY-MM-DD.log
    log_filename = f"{FILE_PREFIX}{date_str}{LOG_FILE_EXTENSION}"
    return log_dir / log_filename

def get_latest_log_filepath(dry=False):
    """
    Constructs the path for the "latest" log file.
    """
    log_dir = get_log_dir_path(dry=dry)
    latest_filename = f"{FILE_PREFIX}{LATEST_LOG_FILENAME}{LOG_FILE_EXTENSION}"
    return log_dir / latest_filename


def foc_path(timestamp, dry=False):
    """
    Create the focus output text file name and its path
    """
    
    root = PROJECT_ROOT / DATA_FOLDER_NAME / focus_folder
    path = root

    if not dry:
        path.mkdir(parents=True, exist_ok=True)

    # OARPAF.YYYY-MM-DD.foc
    outfile = f"{FILE_PREFIX}{timestamp}{focus_ext}"
    outpath = path / outfile
    return outpath


def save_filename(infile_path_str):
    """
    Save a fits file in its path with an ESO-style filename.
    """
    
    inpath = Path(infile_path_str)
    header = fits.getheader(inpath)

    # '2021-12-28T20:09:56.163'
    date_obs_str = header[dateobs] # DATE-OBS from FITS header
    name_for_file = Time(date_obs_str).isot
    
    outfile_name = f"{FILE_PREFIX}{name_for_file}{ext}"
    outfile = Path(outfile_name)

    outdir = fits_path(header) # This already creates the directory
    outpath = outdir / outfile

    shutil.copy2(inpath, outpath)
    return str(outpath) 
