#!/usr/bin/env python

"""Custom format log"""

# System modules
import datetime
import os  # Will be used if log_path only returns a directory
import sys

# Third-party modules
from loguru import logger

# Other templates
from .structure import log_path


def mylog():
    "logger function"

    logger.remove()  # Remove default handler to prevent duplicate console logs

    time_fmt = "{time:YYYY-MM-DD HH:mm:ss.SSSSSS!UTC} "
    level_fmt = "<level>{level: <8}</level> "
    message_fmt = "| {message} "
    stack_fmt = "<bold>({module}.{function}:{line})</bold>"
    fmt = time_fmt + level_fmt + message_fmt + stack_fmt

    # On standard output
    logger.add(
        sys.stderr,
        format=fmt,
        level="DEBUG"
    )

    full_log_file_path = log_path()

    logger.add(
        full_log_file_path,
        format=fmt,
        colorize=True,  # For file logs
        rotation="16:19",  # Local time
        # retention="7 days", # Old logs
        level="DEBUG"
    )

    # Custom levels
    logger.level("DEBUG", color="<magenta><bold>")
    logger.level("INFO", color="<green><bold>")

    return logger


# This ensures mylog() is called only once.
log = mylog()


def main():
    """Main function"""

    # log is already initialized globally
    log.debug("This is a debug message.")
    log.info("This is an info message.")
    log.warning("This is a warning message.")
    log.error("This is an error message.")
    log.critical("This is a critical message.")


if __name__ == "__main__":
    main()
