/* Distributed under the terms of GPLv3 or later. See COPYING for details. */

/*! \file Configuration.cpp
 *
 * \brief Implementation of the configuration classes.
 */

#include <cmath>
#include <cstdio>
#include <exception>
#include <fstream>
#include <hdf5.h>
#include <regex>
#include <string>

#ifndef INCLUDE_TYPES_H_
#include "../include/types.h"
#endif

#ifndef INCLUDE_ERRORS_H_
#include "../include/errors.h"
#endif

#ifndef INCLUDE_LIST_H_
#include "../include/List.h"
#endif

#ifndef INCLUDE_PARSERS_H_
#include "../include/Parsers.h"
#endif

#ifndef INCLUDE_CONFIGURATION_H_
#include "../include/Configuration.h"
#endif

#ifndef INCLUDE_FILE_IO_H_
#include "../include/file_io.h"
#endif

using namespace std;

GeometryConfiguration::GeometryConfiguration(
					     int _nsph, int _lm, int _in_pol, int _npnt, int _npntts, int _isam,
					     int _li, int _le, int _mxndm, int _iavm,
					     double *x, double *y, double *z,
					     double in_th_start, double in_th_step, double in_th_end,
					     double sc_th_start, double sc_th_step, double sc_th_end,
					     double in_ph_start, double in_ph_step, double in_ph_end,
					     double sc_ph_start, double sc_ph_step, double sc_ph_end,
					     int _jwtm
) {
  number_of_spheres = _nsph;
  l_max = _lm;
  in_pol = _in_pol;
  npnt = _npnt;
  npntts = _npntts;
  meridional_type = _isam;
  li = _li;
  le = _le;
  mxndm = _mxndm;
  iavm = _iavm;
  in_theta_start = in_th_start;
  in_theta_step = in_th_step;
  in_theta_end = in_th_end;
  in_phi_start = in_ph_start;
  in_phi_step = in_ph_step;
  in_phi_end = in_ph_end;
  sc_theta_start = sc_th_start;
  sc_theta_step = sc_th_step;
  sc_theta_end = sc_th_end;
  sc_phi_start = sc_ph_start;
  sc_phi_step = sc_ph_step;
  sc_phi_end = sc_ph_end;
  jwtm = _jwtm;
  sph_x = x;
  sph_y = y;
  sph_z = z;
}

GeometryConfiguration::~GeometryConfiguration() {
  delete[] sph_x;
  delete[] sph_y;
  delete[] sph_z;
}

GeometryConfiguration* GeometryConfiguration::from_legacy(string file_name) {
  int num_lines = 0;
  int last_read_line = 0;
  string *file_lines;
  string str_target, str_num;
  smatch m;
  try {
    file_lines = load_file(file_name, &num_lines);
  } catch (...) {
    OpenConfigurationFileException ex(file_name);
    throw ex;
  }
  int _nsph = 0, _lm = 0, _in_pol = 0, _npnt = 0, _npntts = 0, _isam = 0;
  int _li = 0, _le = 0, _mxndm = 0, _iavm = 0;
  regex re = regex("-?[0-9]+");
  str_target = file_lines[last_read_line++];
  regex_search(str_target, m, re);
  _nsph = stoi(m.str());
  if (_nsph == 1) {
    for (int ri = 0; ri < 5; ri++) {
      str_target = m.suffix().str();
      regex_search(str_target, m, re);
      if (ri == 0) _lm = stoi(m.str());
      if (ri == 1) _in_pol = stoi(m.str());
      if (ri == 2) _npnt = stoi(m.str());
      if (ri == 3) _npntts = stoi(m.str());
      if (ri == 4) _isam = stoi(m.str());
    }
  } else {
    for (int ri = 0; ri < 8; ri++) {
      str_target = m.suffix().str();
      regex_search(str_target, m, re);
      if (ri == 0) _li = stoi(m.str());
      if (ri == 1) _le = stoi(m.str());
      if (ri == 2) _mxndm = stoi(m.str());
      if (ri == 3) _in_pol = stoi(m.str());
      if (ri == 4) _npnt = stoi(m.str());
      if (ri == 5) _npntts = stoi(m.str());
      if (ri == 6) _iavm = stoi(m.str());
      if (ri == 7) _isam = stoi(m.str());
    }
  }
  double *x, *y, *z;
  x = new double[_nsph];
  y = new double[_nsph];
  z = new double[_nsph];
  if (_nsph == 1) {
    x[0] = 0.0;
    y[0] = 0.0;
    z[0] = 0.0;
  } else {
    for (int i = 0; i < _nsph; i++) {
      str_target = file_lines[last_read_line++];
      re = regex("-?[0-9]+\\.[0-9]+([eEdD][-+]?[0-9]+)?");
      for (int ri = 0; ri < 3; ri++) {
	regex_search(str_target, m, re);
	str_num = regex_replace(m.str(), regex("D"), "e");
	str_num = regex_replace(str_num, regex("d"), "e");
	if (ri == 0) x[i] = stod(str_num);
	if (ri == 1) y[i] = stod(str_num);
	if (ri == 2) z[i] = stod(str_num);
	str_target = m.suffix().str();
      }
    }
  }
  double in_th_start, in_th_end, in_th_step, sc_th_start, sc_th_end, sc_th_step;
  re = regex("-?[0-9]+\\.[0-9]+([eEdD][-+]?[0-9]+)?");
  str_target = file_lines[last_read_line++];
  for (int ri = 0; ri < 6; ri++) {
    regex_search(str_target, m, re);
    str_num = regex_replace(m.str(), regex("D"), "e");
    str_num = regex_replace(str_num, regex("d"), "e");
    if (ri == 0) in_th_start = stod(str_num);
    if (ri == 1) in_th_step = stod(str_num);
    if (ri == 2) in_th_end = stod(str_num);
    if (ri == 3) sc_th_start = stod(str_num);
    if (ri == 4) sc_th_step = stod(str_num);
    if (ri == 5) sc_th_end = stod(str_num);
    str_target = m.suffix().str();
  }

  double in_ph_start, in_ph_end, in_ph_step, sc_ph_start, sc_ph_end, sc_ph_step;
  str_target = file_lines[last_read_line++];
  for (int ri = 0; ri < 6; ri++) {
    regex_search(str_target, m, re);
    str_num = regex_replace(m.str(), regex("D"), "e");
    str_num = regex_replace(str_num, regex("d"), "e");
    if (ri == 0) in_ph_start = stod(str_num);
    if (ri == 1) in_ph_step = stod(str_num);
    if (ri == 2) in_ph_end = stod(str_num);
    if (ri == 3) sc_ph_start = stod(str_num);
    if (ri == 4) sc_ph_step = stod(str_num);
    if (ri == 5) sc_ph_end = stod(str_num);
    str_target = m.suffix().str();
  }

  int _jwtm;
  re = regex("[0-9]+");
  str_target = file_lines[last_read_line++];
  regex_search(str_target, m, re);
  _jwtm = stoi(m.str());
  GeometryConfiguration *conf = new GeometryConfiguration(
							  _nsph, _lm, _in_pol, _npnt, _npntts, _isam,
							  _li, _le, _mxndm, _iavm,
							  x, y, z,
							  in_th_start, in_th_step, in_th_end,
							  sc_th_start, sc_th_step, sc_th_end,
							  in_ph_start, in_ph_step, in_ph_end,
							  sc_ph_start, sc_ph_step, sc_ph_end,
							  _jwtm
  );
  return conf;
}

ScattererConfiguration::ScattererConfiguration(
					       int nsph,
					       double *scale_vector,
					       int nxi,
					       string variable_name,
					       int *iog_vector,
					       double *ros_vector,
					       int *nshl_vector,
					       double **rcf_vector,
					       int dielectric_func_type,
					       dcomplex ***dc_matrix,
					       bool is_external,
					       double ex,
					       double w,
					       double x
) {
  number_of_spheres = nsph;
  number_of_scales = nxi;
  reference_variable_name = variable_name;
  iog_vec = iog_vector;
  radii_of_spheres = ros_vector;
  nshl_vec = nshl_vector;
  rcf = rcf_vector;
  idfc = dielectric_func_type;
  dc0_matrix = dc_matrix;
  use_external_sphere = is_external;
  exdc = ex;
  wp = w;
  xip = x;
  if (variable_name == "XIV") scale_vec = scale_vector;
  else {
    scale_vec = new double[number_of_scales]();
    const double pi2 = 2.0 * acos(-1.0);
    const double evc = 6.5821188e-16;
    for (int si = 0; si < number_of_scales; si++) {
      double value = scale_vector[si];
      if (variable_name.compare("WNS") == 0) value *= (3.0e8 / wp);
      else if (variable_name.compare("WLS") == 0) value = pi2 / value * 3.0e8 / wp;
      else if (variable_name.compare("PUS") == 0) value /= wp;
      else if (variable_name.compare("EVS") == 0) value /= (evc * wp);
      scale_vec[si] = value;
    }
  }
}

ScattererConfiguration::~ScattererConfiguration() {
  int configurations = 0;
  for (int i = 1; i <= number_of_spheres; i++) {
    if (iog_vec[i - 1] < i) continue;
    configurations++;
  }
  for (int i = 0; i < configurations; i++) {
    for (int j = 0; j < number_of_spheres; j++) {
      delete[] dc0_matrix[i][j];
    }
  }
  delete[] dc0_matrix;
  for (int i = 0; i < configurations; i++) {
    delete[] rcf[i];
  }
  delete[] rcf;
  delete[] nshl_vec;
  delete[] radii_of_spheres;
  delete[] iog_vec;
  delete[] scale_vec;
}

ScattererConfiguration* ScattererConfiguration::from_binary(string file_name, string mode) {
  ScattererConfiguration *conf = NULL;
  if (mode.compare("LEGACY") == 0) {
    conf = ScattererConfiguration::from_legacy(file_name);
  } else if (mode.compare("HDF5") == 0) {
    conf = ScattererConfiguration::from_hdf5(file_name);
  } else {
    string message = "Unknown format mode: \"" + mode + "\"";
    throw UnrecognizedConfigurationException(message);
  }
  return conf;
}

ScattererConfiguration* ScattererConfiguration::from_dedfb(string dedfb_file_name) {
  int num_lines = 0;
  int last_read_line = 0;
  string *file_lines;
  regex re;
  smatch m;
  try {
    file_lines = load_file(dedfb_file_name, &num_lines);
  } catch (...) {
    OpenConfigurationFileException ex(dedfb_file_name);
    throw ex;
  }
  int nsph, ies;
  re = regex("[0-9]+");
  string str_target = file_lines[last_read_line];
  for (int ri = 0; ri < 2; ri++) {
    regex_search(str_target, m, re);
    if (ri == 0) nsph = stoi(m.str());
    if (ri == 1) ies = stoi(m.str());
    str_target = m.suffix().str();
  }
  if (ies != 0) ies = 1;
  double _exdc, _wp, _xip;
  str_target = file_lines[++last_read_line];
  re = regex("-?[0-9]+\\.[0-9]+([eEdD][-+]?[0-9]+)?");
  for (int ri = 0; ri < 3; ri++) {
    regex_search(str_target, m, re);
    string str_number = m.str();
    str_number = regex_replace(str_number, regex("D"), "e");
    str_number = regex_replace(str_number, regex("d"), "e");
    if (ri == 0) _exdc = stod(str_number);
    if (ri == 1) _wp = stod(str_number);
    if (ri == 2) _xip = stod(str_number);
    str_target = m.suffix().str();
  }
  int _idfc, nxi, instpc, insn;
  re = regex("-?[0-9]+");
  for (int ri = 0; ri < 4; ri++) {
    regex_search(str_target, m, re);
    if (ri == 0) _idfc = stoi(m.str());
    if (ri == 1) nxi = stoi(m.str());
    if (ri == 2) instpc = stoi(m.str());
    if (ri == 3) insn = stoi(m.str());
    str_target = m.suffix().str();
  }

  double *variable_vector;
  string variable_name;

  if (_idfc < 0) { // Diel. functions at XIP value and XI is scale factor
    variable_name = "XIV";
    if (instpc < 1) { // The variable vector is explicitly defined.
      double xi;
      List<double> xi_vector;
      str_target = file_lines[++last_read_line];
      re = regex("-?[0-9]+\\.[0-9]+([eEdD][-+]?[0-9]+)?");
      regex_search(str_target, m, re);
      string str_number = m.str();
      str_number = regex_replace(str_number, regex("D"), "e");
      str_number = regex_replace(str_number, regex("d"), "e");
      xi = stod(str_number);
      xi_vector.set(0, xi);
      for (int jxi310 = 1; jxi310 < nxi; jxi310++) {
	str_target = file_lines[++last_read_line];
	regex_search(str_target, m, re);
	str_number = m.str();
	str_number = regex_replace(str_number, regex("D"), "e");
	str_number = regex_replace(str_number, regex("d"), "e");
	xi = stod(str_number);
	xi_vector.append(xi);
      }
      variable_vector = xi_vector.to_array();
    } else { // instpc >= 1: the variable vector is defined in steps
      double xi, xi_step;
      str_target = file_lines[++last_read_line];
      re = regex("-?[0-9]+\\.[0-9]+([eEdD][-+]?[0-9]+)?");
      regex_search(str_target, m, re);
      for (int ri = 0; ri < 2; ri++) {
	regex_search(str_target, m, re);
	string str_number = m.str();
	str_number = regex_replace(str_number, regex("D"), "e");
	str_number = regex_replace(str_number, regex("d"), "e");
	if (ri == 0) xi = stod(str_number);
	if (ri == 1) xi_step = stod(str_number);
	str_target = m.suffix().str();
      }
      variable_vector = new double[nxi]();
      for (int jxi320 = 0; jxi320 < nxi; jxi320++) {
	variable_vector[jxi320] = xi;
	xi += xi_step;
      }
    }
  } else { // idfc >= 0
    variable_vector = new double[nxi]();
    if (instpc == 0) { // The variable vector is explicitly defined
      double vs;
      for (int jxi_r = 0; jxi_r < nxi; jxi_r++) {
	str_target = file_lines[++last_read_line];
	re = regex("-?[0-9]+\\.[0-9]+([eEdD][-+]?[0-9]+)?");
	regex_search(str_target, m, re);
	string str_number = m.str();
	str_number = regex_replace(str_number, regex("D"), "e");
	str_number = regex_replace(str_number, regex("d"), "e");
	vs = stod(str_number);
	variable_vector[jxi_r] = vs;
      }
      switch (insn) {
      case 1: //xi vector definition
	variable_name = "XIV";
	break;
      case 2: //wave number vector definition
	variable_name = "WNS";
	break;
      case 3: //wavelength vector definition
	variable_name = "WLS";
	break;
      case 4: //pu vector definition
	variable_name = "PUS";
	break;
      case 5: //eV vector definition
	variable_name = "EVS";
	break;
      }
    } else { // The variable vector needs to be computed in steps
      double vs, vs_step;
      str_target = file_lines[++last_read_line];
      re = regex("-?[0-9]+\\.[0-9]+([eEdD][-+]?[0-9]+)?");
      regex_search(str_target, m, re);
      for (int ri = 0; ri < 2; ri++) {
	regex_search(str_target, m, re);
	string str_number = m.str();
	str_number = regex_replace(str_number, regex("D"), "e");
	str_number = regex_replace(str_number, regex("d"), "e");
	if (ri == 0) vs = stod(str_number);
	if (ri == 1) vs_step = stod(str_number);
	str_target = m.suffix().str();
      }
      for (int jxi110w = 0; jxi110w < nxi; jxi110w++) {
	variable_vector[jxi110w] = vs;
	vs += vs_step;
      }
      switch (insn) {
      case 1: //xi vector definition
	variable_name = "XIV";
	break;
      case 2: //wave number vector definition
	variable_name = "WNS";
	break;
      case 3: //wavelength vector definition
	variable_name = "WLS";
	break;
      case 4: //pu vector definition
	variable_name = "PUS";
	break;
      case 5: //eV vector definition
	variable_name = "EVS";
	break;
      }
    }
  }
  int *iog_vector = new int[nsph]();
  str_target = file_lines[++last_read_line];
  re = regex("[0-9]+");
  int configurations = 0;
  for (int i = 1; i <= nsph; i++) {
    bool success = regex_search(str_target, m, re);
    if (success) {
      iog_vector[i - 1] = stoi(m.str());
      str_target = m.suffix().str();
      if (iog_vector[i - 1] >= i) configurations++;
    } else {
      str_target = file_lines[++last_read_line];
      i--;
    }
  }
  int index = 0;
  double *ros_vector = new double[configurations]();
  int *nshl_vector = new int[configurations]();
  double **rcf_vector = new double*[configurations];
  for (int i113 = 1; i113 <= nsph; i113++) {
    if (iog_vector[i113 - 1] < i113) continue;
    str_target = file_lines[++last_read_line];
    re = regex("[0-9]+");
    regex_search(str_target, m, re);
    nshl_vector[i113 - 1] = stoi(m.str());
    str_target = m.suffix().str();
    re = regex("-?[0-9]+\\.[0-9]+([eEdD][-+]?[0-9]+)?");
    regex_search(str_target, m, re);
    string str_number = m.str();
    str_number = regex_replace(str_number, regex("D"), "e");
    str_number = regex_replace(str_number, regex("d"), "e");
    ros_vector[i113 - 1] = stod(str_number);
    int nsh = nshl_vector[i113 - 1];
    if (i113 == 1) nsh += ies;
    rcf_vector[i113 - 1] = new double[nsh]();
    for (int ns112 = 0; ns112 < nsh; ns112++) {
      str_target = file_lines[++last_read_line];
      regex_search(str_target, m, re);
      str_number = m.str();
      str_number = regex_replace(str_number, regex("D"), "e");
      str_number = regex_replace(str_number, regex("d"), "e");
      rcf_vector[i113 - 1][ns112] = stod(str_number);
    } // ns112 loop
  } // i113 loop

  dcomplex ***dc0m = new dcomplex**[configurations];
  dcomplex *dc0 = new dcomplex[configurations]();
  int dim3 = (_idfc == 0) ? nxi : 1;
  for (int di = 0; di < configurations; di++) {
    dc0m[di] = new dcomplex*[nsph];
    for (int dj = 0; dj < nsph; dj++) dc0m[di][dj] = new dcomplex[dim3]();
  } // di loop
  for (int jxi468 = 1; jxi468 <= nxi; jxi468++) {
    if (_idfc != 0 && jxi468 > 1) continue; // jxi468 loop
    for (int i162 = 1; i162 <= nsph; i162++) {
      if (iog_vector[i162 - 1] >= i162) {
	int nsh = nshl_vector[i162 - 1];
	int ici = (nsh + 1) / 2;
	if (i162 == 1) ici += ies;
	for (int ic157 = 0; ic157 < ici; ic157++) {
	  str_target = file_lines[++last_read_line];
	  regex_search(str_target, m, re);
	  string str_number = m.str();
	  str_number = regex_replace(str_number, regex("D"), "e");
	  str_number = regex_replace(str_number, regex("d"), "e");
	  double rval = stod(str_number);
	  str_target = m.suffix().str();
	  regex_search(str_target, m, re);
	  str_number = m.str();
	  str_number = regex_replace(str_number, regex("D"), "e");
	  str_number = regex_replace(str_number, regex("d"), "e");
	  double ival = stod(str_number);
	  dc0[ic157] = rval + ival * I;
	  dc0m[ic157][i162 - 1][jxi468 - 1] = dc0[ic157];
	} // ic157 loop
      }
    } // i162 loop
  } // jxi468 loop
  delete[] dc0;

  ScattererConfiguration *config = new ScattererConfiguration(
							      nsph,
							      variable_vector,
							      nxi,
							      variable_name,
							      iog_vector,
							      ros_vector,
							      nshl_vector,
							      rcf_vector,
							      _idfc,
							      dc0m,
							      (ies > 0),
							      _exdc,
							      _wp,
							      _xip
  );
  delete[] file_lines;
  return config;
}

ScattererConfiguration* ScattererConfiguration::from_hdf5(string file_name) {
  ScattererConfiguration *conf = NULL;
  unsigned int flags = H5F_ACC_RDONLY;
  HDFFile *hdf_file = new HDFFile(file_name, flags);
  herr_t status = hdf_file->get_status();
  string str_name, str_type;
  if (status == 0) {
    int nsph, ies;
    int *iog;
    double _exdc, _wp, _xip;
    int _idfc, nxi;
    int *nshl_vector;
    double *xi_vec;
    double *ros_vector;
    double **rcf_vector;
    dcomplex ***dc0m;
    status = hdf_file->read("NSPH", "INT32_(1)", &nsph);
    status = hdf_file->read("IES", "INT32_(1)", &ies);
    status = hdf_file->read("EXDC", "FLOAT64_(1)", &_exdc);
    status = hdf_file->read("WP", "FLOAT64_(1)", &_wp);
    status = hdf_file->read("XIP", "FLOAT64_(1)", &_xip);
    status = hdf_file->read("IDFC", "INT32_(1)", &_idfc);
    status = hdf_file->read("NXI", "INT32_(1)", &nxi);
    iog = new int[nsph];
    string str_type = "INT32_(" + to_string(nsph) + ")";
    status = hdf_file->read("IOGVEC", str_type, iog);
    int configuration_count = 0;
    for (int ci = 0; ci < nsph; ci++) {
      if (iog[ci] < ci + 1) continue;
      configuration_count++;
    }
    nshl_vector = new int[configuration_count]();
    ros_vector = new double[configuration_count]();
    rcf_vector = new double*[configuration_count];
    for (int i115 = 1; i115 <= nsph; i115++) {
      if (iog[i115 - 1] < i115) continue;
      str_name = "NSHL_" + to_string(iog[i115 - 1]);
      str_type = "INT32_(1)";
      status = hdf_file->read(str_name, str_type, (nshl_vector + i115 - 1));
      str_name = "ROS_" + to_string(iog[i115 - 1]);
      str_type = "FLOAT64_(1)";
      status = hdf_file->read(str_name, str_type, (ros_vector + i115 - 1));
      int nsh = nshl_vector[i115 - 1];
      rcf_vector[i115 - 1] = new double[nsh]();
      str_name = "RCF_" +  to_string(iog[i115 - 1]);
      str_type = "FLOAT64_(" + to_string(nsh) + ")";
      status = hdf_file->read(str_name, str_type, (rcf_vector[i115 - 1]));
    }
    xi_vec = new double[nxi]();
    str_name = "XIVEC";
    str_type = "FLOAT64_(" + to_string(nxi) + ")";
    status = hdf_file->read(str_name, str_type, xi_vec);

    int dim3 = (_idfc == 0) ? nxi : 1;
    int element_size = 2 * dim3 * nsph * configuration_count;
    double *elements = new double[element_size]();
    str_name = "DC0M";
    str_type = "FLOAT64_(" + to_string(element_size) + ")";
    status = hdf_file->read(str_name, str_type, elements);
    dc0m = new dcomplex**[configuration_count];
    int dc_index = 0;
    for (int di = 0; di < configuration_count; di++) {
      dc0m[di] = new dcomplex*[nsph];
      for (int dj = 0; dj < nsph; dj++) {
	dc0m[di][dj] = new dcomplex[dim3]();
	for (int dk = 0; dk < dim3; dk++) {
	  double rval = elements[2 * dc_index];
	  double ival = elements[2 * dc_index + 1];
	  dc_index++;
	  dc0m[di][dj][dk] = rval + ival * I;
	}
      } // dj loop
    } // di loop
    delete[] elements;
    status = hdf_file->close();
    conf = new ScattererConfiguration(
				      nsph,
				      xi_vec,
				      nxi,
				      "XIV",
				      iog,
				      ros_vector,
				      nshl_vector,
				      rcf_vector,
				      _idfc,
				      dc0m,
				      (ies == 1),
				      _exdc,
				      _wp,
				      _xip
    );
  }
  
  return conf;
}

ScattererConfiguration* ScattererConfiguration::from_legacy(string file_name) {
  int _nsph;
  int *_iog_vec;
  int _ies;
  double _exdc, _wp, _xip;
  int _idfc, _nxi;
  int *_nshl_vec;
  double *_xi_vec;
  double *_ros_vec;
  double **_rcf_vec;
  dcomplex ***_dc0m;

  fstream input;
  input.open(file_name.c_str(), ios::in | ios::binary);
  input.read(reinterpret_cast<char *>(&_nsph), sizeof(int));
  input.read(reinterpret_cast<char *>(&_ies), sizeof(int));
  _iog_vec = new int[_nsph]();
  for (int i = 0; i < _nsph; i++)
    input.read(reinterpret_cast<char *>(&(_iog_vec[i])), sizeof(int));
  input.read(reinterpret_cast<char *>(&_exdc), sizeof(double));
  input.read(reinterpret_cast<char *>(&_wp), sizeof(double));
  input.read(reinterpret_cast<char *>(&_xip), sizeof(double));
  input.read(reinterpret_cast<char *>(&_idfc), sizeof(int));
  input.read(reinterpret_cast<char *>(&_nxi), sizeof(int));
  _xi_vec = new double[_nxi]();
  for (int i = 0; i < _nxi; i++)
    input.read(reinterpret_cast<char *>(&(_xi_vec[i])), sizeof(double));
  int configurations = 0;
  for (int i = 1; i <= _nsph; i++) {
    if (_iog_vec[i - 1] >= i) configurations++;
  }
  _nshl_vec = new int[configurations]();
  _ros_vec = new double[configurations]();
  _rcf_vec = new double*[configurations];
  for (int i115 = 1; i115 <= _nsph; i115++) {
    if (_iog_vec[i115 - 1] < i115) continue;
    input.read(reinterpret_cast<char *>(&(_nshl_vec[i115 - 1])), sizeof(int));
    input.read(reinterpret_cast<char *>(&(_ros_vec[i115 - 1])), sizeof(double));
    int nsh = _nshl_vec[i115 - 1];
    if (i115 == 1) nsh += _ies;
    _rcf_vec[i115 - 1] = new double[nsh]();
    for (int nsi = 0; nsi < nsh; nsi++)
      input.read(reinterpret_cast<char *>(&(_rcf_vec[i115 - 1][nsi])), sizeof(double));
  }
  _dc0m = new dcomplex**[configurations];
  int dim3 = (_idfc == 0) ? _nxi : 1;
  for (int di = 0; di < configurations; di++) {
    _dc0m[di] = new dcomplex*[_nsph];
    for (int dj = 0; dj < _nsph; dj++) _dc0m[di][dj] = new dcomplex[dim3]();
  } // di loop
  for (int jxi468 = 1; jxi468 <= _nxi; jxi468++) {
    if (_idfc != 0 && jxi468 > 1) continue;
    for (int i162 = 1; i162 <= _nsph; i162++) {
      if (_iog_vec[i162 - 1] < i162) continue;
      int nsh = _nshl_vec[i162 - 1];
      int ici = (nsh + 1) / 2;
      if (i162 == 1) ici = ici + _ies;
      for (int i157 = 0; i157 < ici; i157++) {
	double rval, ival;
	input.read(reinterpret_cast<char *>(&rval), sizeof(double));
	input.read(reinterpret_cast<char *>(&ival), sizeof(double));
	_dc0m[i157][i162 - 1][jxi468 - 1] = rval + ival * I;
      }
    }
  }
  input.close();
    
  ScattererConfiguration *conf = new ScattererConfiguration(
							    _nsph,
							    _xi_vec,
							    _nxi,
							    "XIV",
							    _iog_vec,
							    _ros_vec,
							    _nshl_vec,
							    _rcf_vec,
							    _idfc,
							    _dc0m,
							    (_ies == 1),
							    _exdc,
							    _wp,
							    _xip
  );
  return conf;
}

double ScattererConfiguration::get_param(string param_name) {
  double value;
  if (param_name.compare("number_of_spheres") == 0) value = 1.0 * number_of_spheres;
  else if (param_name.compare("nsph") == 0) value = 1.0 * number_of_spheres;
  else if (param_name.compare("number_of_scales") == 0) value = 1.0 * number_of_scales;
  else if (param_name.compare("nxi") == 0) value = 1.0 * number_of_scales;
  else if (param_name.compare("idfc") == 0) value = 1.0 * idfc;
  else if (param_name.compare("exdc") == 0) value = exdc;
  else if (param_name.compare("wp") == 0) value = wp;
  else if (param_name.compare("xip") == 0) value = xip;
  else {
    // TODO: add exception code for unknown parameter.
    return 0.0;
  }
  return value;
}

void ScattererConfiguration::print() {
  int ies = (use_external_sphere)? 1 : 0;
  int configurations = 0;
  for (int ci = 1; ci <= number_of_spheres; ci++) {
    if (iog_vec[ci - 1] >= ci) configurations++;
  }
  printf("### CONFIGURATION DATA ###\n");
  printf("NSPH  = %d\n", number_of_spheres);
  printf("ROS   = [");
  for (int i = 0; i < configurations; i++) printf("\t%lg", radii_of_spheres[i]);
  printf("\t]\n");
  printf("IOG   = [");
  for (int i = 0; i < number_of_spheres; i++) printf("\t%d", iog_vec[i]);
  printf("\t]\n");
  printf("NSHL  = [");
  for (int i = 0; i < configurations; i++) printf("\t%d", nshl_vec[i]);
  printf("\t]\n");
  printf("RCF   = [\n");
  for (int i = 1; i <= configurations; i++) {
    int nsh = nshl_vec[i - 1];
    if (i == 1) nsh += ies;
    printf("         [");
    for (int ns = 0; ns < nsh; ns++) {
      printf("\t%lg", rcf[i - 1][ns]);
    }
    printf("\t]\n");
  }
  printf("        ]\n");
  printf("SCALE = %s\n", reference_variable_name.c_str());
  printf("NXI   = %d\n", number_of_scales);
  printf("VEC   = [");
  for (int i = 0; i < number_of_scales; i++) printf("\t%lg", scale_vec[i]);
  printf("\t]\n");
  printf("DC0M  = [\n");
  for (int i = 0; i < configurations; i++) {
    printf("         [\n");
    for (int j = 0; j < number_of_spheres; j++) {
      printf("          [");
      for (int k = 0; k < number_of_scales; k++) {
	if (idfc != 0 and k > 0) continue;
	printf("\t%lg + i(%lg)", real(dc0_matrix[i][j][k]), imag(dc0_matrix[i][j][k]));
      }
      printf("\t]\n");
    }
    printf("         ]\n");
  }
  printf("        ]\n");
}

void ScattererConfiguration::write_binary(string file_name, string mode) {
  if (mode.compare("LEGACY") == 0) {
    write_legacy(file_name);
  } else if (mode.compare("HDF5") == 0) {
    write_hdf5(file_name);
  } else {
    string message = "Unknown format mode: \"" + mode + "\"";
    throw UnrecognizedConfigurationException(message);
  }
}

void ScattererConfiguration::write_hdf5(string file_name) {
  int ies = (use_external_sphere)? 1 : 0;
  List<string> rec_name_list(1);
  List<string> rec_type_list(1);
  List<void *> rec_ptr_list(1);
  string str_type, str_name;
  int configurations = 0;
  for (int ci = 1; ci <= number_of_spheres; ci++) {
    if(iog_vec[ci - 1] >= ci) configurations++;
  }
  rec_name_list.set(0, "NSPH");
  rec_type_list.set(0, "INT32_(1)");
  rec_ptr_list.set(0, &number_of_spheres);
  rec_name_list.append("IES");
  rec_type_list.append("INT32_(1)");
  rec_ptr_list.append(&ies);
  rec_name_list.append("IOGVEC");
  str_type = "INT32_(" + to_string(number_of_spheres) + ")";
  rec_type_list.append(str_type);
  rec_ptr_list.append(iog_vec);
  rec_name_list.append("EXDC");
  rec_type_list.append("FLOAT64_(1)");
  rec_ptr_list.append(&exdc);
  rec_name_list.append("WP");
  rec_type_list.append("FLOAT64_(1)");
  rec_ptr_list.append(&wp);
  rec_name_list.append("XIP");
  rec_type_list.append("FLOAT64_(1)");
  rec_ptr_list.append(&xip);
  rec_name_list.append("IDFC");
  rec_type_list.append("INT32_(1)");
  rec_ptr_list.append(&idfc);
  rec_name_list.append("NXI");
  rec_type_list.append("INT32_(1)");
  rec_ptr_list.append(&number_of_scales);
  rec_name_list.append("XIVEC");
  str_type = "FLOAT64_(" + to_string(number_of_scales) + ")";
  rec_type_list.append(str_type);
  rec_ptr_list.append(scale_vec);
  for (int i115 = 1; i115 <= number_of_spheres; i115++) {
    if (iog_vec[i115 - 1] < i115) continue;
    str_name = "NSHL_" + to_string(i115);
    rec_name_list.append(str_name);
    rec_type_list.append("INT32_(1)");
    rec_ptr_list.append(&(nshl_vec[i115 - 1])); // was not from IOG
    str_name = "ROS_" + to_string(i115);
    rec_name_list.append(str_name);
    rec_type_list.append("FLOAT64_(1)");
    rec_ptr_list.append(&(radii_of_spheres[i115 - 1])); // was not from IOG
    int nsh = nshl_vec[i115 - 1]; // was not from IOG
    if (i115 == 1) nsh += ies;
    str_name = "RCF_" + to_string(i115); // was not from IOG
    str_type = "FLOAT64_(" + to_string(nsh) + ")";
    rec_name_list.append(str_name);
    rec_type_list.append(str_type);
    rec_ptr_list.append(&(rcf[i115 - 1][0])); // was not from IOG
  }

  int dim3 = (idfc == 0) ? number_of_scales : 1;
  int dc0m_size = 2 * dim3 * number_of_spheres * configurations;
  double *dc0m = new double[dc0m_size]();
  int dc0_index = 0;
  for (int jxi468 = 1; jxi468 <= number_of_scales; jxi468++) {
    if (idfc != 0 && jxi468 > 1) continue;
    for (int i162 = 1; i162 <= number_of_spheres; i162++) {
      if (iog_vec[i162 - 1] < i162) continue;
      int nsh = nshl_vec[i162 - 1]; // was not from IOG
      int ici = (nsh + 1) / 2;
      if (i162 == 1) ici = ici + ies;
      for (int i157 = 0; i157 < ici; i157++) {
	double dc0_real, dc0_imag;
	dc0_real = real(dc0_matrix[i157][i162 - 1][jxi468 - 1]);
	dc0_imag = imag(dc0_matrix[i157][i162 - 1][jxi468 - 1]);
	dc0m[2 * dc0_index] = dc0_real;
	dc0m[2 * dc0_index + 1] = dc0_imag;
	dc0_index++;
      }
    }
  }
  str_type = "FLOAT64_(" + to_string(dc0m_size) + ")";
  rec_name_list.append("DC0M");
  rec_type_list.append(str_type);
  rec_ptr_list.append(dc0m);

  string *rec_names = rec_name_list.to_array();
  string *rec_types = rec_type_list.to_array();
  void **rec_pointers = rec_ptr_list.to_array();
  const int rec_num = rec_name_list.length();
  FileSchema schema(rec_num, rec_types, rec_names);
  HDFFile *hdf_file = HDFFile::from_schema(schema, file_name, H5F_ACC_TRUNC);
  for (int ri = 0; ri < rec_num; ri++)
    hdf_file->write(rec_names[ri], rec_types[ri], rec_pointers[ri]);
  hdf_file->close();
  
  // Clean memory
  delete[] dc0m;
  delete[] rec_names;
  delete[] rec_types;
  delete[] rec_pointers;
  delete hdf_file;
}

void ScattererConfiguration::write_legacy(string file_name) {
  fstream output;
  int ies = (use_external_sphere)? 1 : 0;
  output.open(file_name.c_str(), ios::out | ios::binary);
  output.write(reinterpret_cast<char *>(&number_of_spheres), sizeof(int));
  output.write(reinterpret_cast<char *>(&ies), sizeof(int));
  for (int i = 0; i < number_of_spheres; i++)
    output.write(reinterpret_cast<char *>(&(iog_vec[i])), sizeof(int));
  output.write(reinterpret_cast<char *>(&exdc), sizeof(double));
  output.write(reinterpret_cast<char *>(&wp), sizeof(double));
  output.write(reinterpret_cast<char *>(&xip), sizeof(double));
  output.write(reinterpret_cast<char *>(&idfc), sizeof(int));
  output.write(reinterpret_cast<char *>(&number_of_scales), sizeof(int));
  for (int i = 0; i < number_of_scales; i++)
    output.write(reinterpret_cast<char *>(&(scale_vec[i])), sizeof(double));
  for (int i115 = 1; i115 <= number_of_spheres; i115++) {
    if (iog_vec[i115 - 1] < i115) continue;
    output.write(reinterpret_cast<char *>(&(nshl_vec[i115 - 1])), sizeof(int)); // was not from IOG
    output.write(reinterpret_cast<char *>(&(radii_of_spheres[i115 - 1])), sizeof(double)); // was not from IOG
    int nsh = nshl_vec[i115 - 1]; // was not from IOG
    if (i115 == 1) nsh += ies;
    for (int nsi = 0; nsi < nsh; nsi++)
      output.write(reinterpret_cast<char *>(&(rcf[i115 - 1][nsi])), sizeof(double)); // was not from IOG
  }
  for (int jxi468 = 1; jxi468 <= number_of_scales; jxi468++) {
    if (idfc != 0 && jxi468 > 1) continue;
    for (int i162 = 1; i162 <= number_of_spheres; i162++) {
      if (iog_vec[i162 - 1] < i162) continue;
      int nsh = nshl_vec[i162 - 1]; // was not from IOG
      int ici = (nsh + 1) / 2; // QUESTION: is integer division really intended here?
      if (i162 == 1) ici = ici + ies;
      for (int i157 = 0; i157 < ici; i157++) {
	double dc0_real, dc0_img;
	dc0_real = real(dc0_matrix[i157][i162 - 1][jxi468 - 1]);
	dc0_img = imag(dc0_matrix[i157][i162 - 1][jxi468 - 1]);
	// The FORTRAN code writes the complex numbers as a 16-byte long binary stream.
	// Here we assume that the 16 bytes are equally split in 8 bytes to represent the
	// real part and 8 bytes to represent the imaginary one.
	output.write(reinterpret_cast<char *>(&dc0_real), sizeof(double));
	output.write(reinterpret_cast<char *>(&dc0_img), sizeof(double));
      }
    }
  }
  output.close();
}

void ScattererConfiguration::write_formatted(string file_name) {
  const double evc = 6.5821188e-16;
  const double two_pi = acos(0.0) * 4.0;
  double *xi_vec;
  int ici;
  int ies = (use_external_sphere)? 1: 0;
  FILE *output = fopen(file_name.c_str(), "w");
  int scale_type = -1;
  if (reference_variable_name.compare("XIV") == 0) scale_type = 0;
  else if (reference_variable_name.compare("WNS") == 0) scale_type = 1;
  else if (reference_variable_name.compare("WLS") == 0) scale_type = 2;
  else if (reference_variable_name.compare("PUS") == 0) scale_type = 3;
  else if (reference_variable_name.compare("EVS") == 0) scale_type = 4;
  if (idfc >= 0) { // Dielectric functions are constant or they depend on XI
    double  *pu_vec, *ev_vec, *wn_vec, *wl_vec;
    xi_vec = new double[number_of_scales];
    pu_vec = new double[number_of_scales];
    ev_vec = new double[number_of_scales];
    wn_vec = new double[number_of_scales];
    wl_vec = new double[number_of_scales];
    switch (scale_type) {
    case 0:
      fprintf(output, "  JXI     XIV          WNS          WLS          PUS          EVS\n");
      for (int i = 0; i < number_of_scales; i++) {
	xi_vec[i] = scale_vec[i];
	pu_vec[i] = xi_vec[i] * wp;
	ev_vec[i] = pu_vec[i] * evc;
	wn_vec[i] = pu_vec[i] / 3.0e8;
	wl_vec[i] = two_pi / wn_vec[i];
	fprintf(
		output,
		"%5d%13.4lE%13.4lE%13.4lE%13.4lE%13.4lE\n",
		(i + 1),
		xi_vec[i],
		wn_vec[i],
		wl_vec[i],
		pu_vec[i],
		ev_vec[i]
	);
      }
      break;
    case 1:
      fprintf(output, "  JXI     WNS          WLS          PUS          EVS          XIV\n");
      for (int i = 0; i < number_of_scales; i++) {
	xi_vec[i] = scale_vec[i];
	pu_vec[i] = xi_vec[i] * wp;
	wn_vec[i] = pu_vec[i] / 3.0e8;
	wl_vec[i] = two_pi / wn_vec[i];
	ev_vec[i] = pu_vec[i] * evc;
	fprintf(
		output,
		"%5d%13.4lE%13.4lE%13.4lE%13.4lE%13.4lE\n",
		(i + 1),
		wn_vec[i],
		wl_vec[i],
		pu_vec[i],
		ev_vec[i],
		xi_vec[i]
	);
      }
      break;
    case 2:
      fprintf(output, "  JXI     WLS          WNS          PUS          EVS          XIV\n");
      for (int i = 0; i < number_of_scales; i++) {
	xi_vec[i] = scale_vec[i];
	pu_vec[i] = xi_vec[i] * wp;
	wn_vec[i] = pu_vec[i] / 3.0e8;
	wl_vec[i] = two_pi / wn_vec[i];
	ev_vec[i] = pu_vec[i] * evc;
	fprintf(
		output,
		"%5d%13.4lE%13.4lE%13.4lE%13.4lE%13.4lE\n",
		(i + 1),
		wl_vec[i],
		wn_vec[i],
		pu_vec[i],
		ev_vec[i],
		xi_vec[i]
	);
      }
      break;
    case 3:
      fprintf(output, "  JXI     PUS          WNS          WLS          EVS          XIV\n");
      for (int i = 0; i < number_of_scales; i++) {
	xi_vec[i] = scale_vec[i];
	pu_vec[i] = xi_vec[i] * wp;
	wn_vec[i] = pu_vec[i] / 3.0e8;
	wl_vec[i] = two_pi / wn_vec[i];
	ev_vec[i] = pu_vec[i] * evc;
	fprintf(
		output,
		"%5d%13.4lE%13.4lE%13.4lE%13.4lE%13.4lE\n",
		(i + 1),
		pu_vec[i],
		wn_vec[i],
		wl_vec[i],
		ev_vec[i],
		xi_vec[i]
	);
      }
      break;
    case 4:
      fprintf(output, "  JXI     EVS          WNS          WLS          PUS          XIV\n");
      for (int i = 0; i < number_of_scales; i++) {
	xi_vec[i] = scale_vec[i];
	pu_vec[i] = xi_vec[i] * wp;
	wn_vec[i] = pu_vec[i] / 3.0e8;
	wl_vec[i] = two_pi / wn_vec[i];
	ev_vec[i] = pu_vec[i] * evc;
	fprintf(
		output,
		"%5d%13.4lE%13.4lE%13.4lE%13.4lE%13.4lE\n",
		(i + 1),
		ev_vec[i],
		wn_vec[i],
		wl_vec[i],
		pu_vec[i],
		xi_vec[i]
	);
      }
      break;
    default:
      throw UnrecognizedConfigurationException(
					       "Wonrg parameter set: unrecognized scale variable type "
					       + reference_variable_name
					       );
      break;
    }
    // Clean memory
    delete[] xi_vec;
    delete[] pu_vec;
    delete[] ev_vec;
    delete[] wn_vec;
    delete[] wl_vec;
  } else { // idfc < 0, Dielectric functions are at XIP and XI is scale for dimensions
    double pu, wn;
    xi_vec = scale_vec;
    pu = xip * wp;
    wn = pu / 3.0e8;
    fprintf(output, "          XIP          WN           WL           PU           EV\n");
    fprintf(output, "     %13.4lE", xip);
    fprintf(output, "%13.4lE", wn);
    fprintf(output, "%13.4lE", two_pi / wn);
    fprintf(output, "%13.4lE", pu);
    fprintf(output, "%13.4lE\n", pu * evc);
    fprintf(output, "  SCALE FACTORS XI\n");
    for (int i = 0; i < number_of_scales; i++)
      fprintf(output, "%5d%13.4lE\n", (i + 1), xi_vec[i]);
  }
  if (idfc != 0) {
    fprintf(output, "  DIELECTRIC CONSTANTS\n");
    for (int i473 = 1; i473 <= number_of_spheres; i473++) {
      if (iog_vec[i473 - 1] != i473) continue;
      ici = (nshl_vec[i473 - 1] + 1) / 2;
      if (i473 == 1) ici += ies;
      fprintf(output, " SPHERE N.%4d\n", i473);
      for (int ic472 = 0; ic472 < ici; ic472++) {
	double dc0_real = real(dc0_matrix[ic472][i473 - 1][0]);
	double dc0_img = imag(dc0_matrix[ic472][i473 - 1][0]);
	fprintf(output, "%5d %12.4lE%12.4lE\n", (ic472 + 1), dc0_real, dc0_img);
      }
    }
  } else {
    fprintf(output, "  DIELECTRIC FUNCTIONS\n");
    for (int i478 = 1; i478 <= number_of_spheres; i478++) {
      if (iog_vec[i478 - 1] != i478) continue;
      ici = (nshl_vec[i478 - 1] + 1) / 2;
      if (i478 == 1) ici += ies;
      fprintf(output, " SPHERE N.%4d\n", i478);
      for (int ic477 = 1; ic477 <= ici; ic477++) {
	fprintf(output, " NONTRANSITION LAYER N.%2d, SCALE = %3s\n", ic477, reference_variable_name.c_str());
	for (int jxi476 = 0; jxi476 < number_of_scales; jxi476++) {
	  double dc0_real = real(dc0_matrix[ic477 - 1][i478 - 1][jxi476]);
	  double dc0_img = imag(dc0_matrix[ic477 - 1][i478 - 1][jxi476]);
	  fprintf(output, "%5d %12.4lE%12.4lE\n", (jxi476 + 1), dc0_real, dc0_img);
	}
      }
    }
  }
  fclose(output);
}

bool ScattererConfiguration::operator ==(ScattererConfiguration &other) {
  if (number_of_spheres != other.number_of_spheres) {
    return false;
  }
  if (number_of_scales != other.number_of_scales) {
    return false;
  }
  if (idfc != other.idfc) {
    return false;
  }
  if (exdc != other.exdc) {
    return false;
  }
  if (wp != other.wp) {
    return false;
  }
  if (xip != other.xip) {
    return false;
  }
  //if (use_external_sphere != other.use_external_sphere) return false;
  for (int svi = 0; svi < number_of_scales; svi++)
    if (scale_vec[svi] != other.scale_vec[svi]) {
      return false;
    }
  int configurations = 0;
  int dim3 = (idfc == 0) ? number_of_scales : 1;
  for (int ci = 1; ci <= number_of_spheres; ci++) {
    if (iog_vec[ci - 1] != other.iog_vec[ci - 1]) {
      return false;
    }
    if (iog_vec[ci - 1] < ci) continue;
    configurations++;
  }
  for (int ri = 0; ri < configurations; ri++) {
    if (radii_of_spheres[ri] != other.radii_of_spheres[ri]) {
      return false;
    }
    if (nshl_vec[ri] != other.nshl_vec[ri]) {
      return false;
    }
    for (int rj = 0; rj < nshl_vec[ri]; rj++) {
      if (rcf[ri][rj] != other.rcf[ri][rj]) {
	return false;
      }
    } // rj loop
    for (int dj = 0; dj < number_of_spheres; dj++) {
      for (int dk = 0; dk < dim3; dk++) {
	if (dc0_matrix[ri][dj][dk] != other.dc0_matrix[ri][dj][dk]) {
	  return false;
	}
      } // dk loop
    } // dj loop
  } // ri loop
  return true;
}
