/* Distributed under the terms of GPLv3 or later. See COPYING for details. */

/*! \file np_sphere.cpp
 *
 * \brief Single sphere scattering problem handler.
 *
 * This program emulates the execution work-flow originally handled by the
 * FORTRAN EDFB and SPH codes, which undertook the task of solving the
 * scattering calculation for the case of a single, possibly multi-layered
 * sphere. The program is designed to work in two modes: emulation and
 * enhanced. The emulation mode is activated by invoking the program without
 * arguments. In this case, the code looks for hard-coded default input and
 * writes output in the execution folder, replicating the behaviour of the
 * original FORTRAN code. Advanced mode, instead, is activated by passing
 * command line arguments that locate the desired input files and a valid
 * folder to write the output into. The emulation mode is useful for testing,
 * while the advanced mode implements the possibility to change input and
 * output options, without having to modify the code.
 */
#include <cstdio>
#include <string>

#ifndef INCLUDE_TYPES_H_
#include "../include/types.h"
#endif

#ifndef INCLUDE_CONFIGURATION_H_
#include "../include/Configuration.h"
#endif

using namespace std;

extern void sphere(string config_file, string data_file, string output_path);

/*! \brief Main program entry point.
 *
 * This is the starting point of the execution flow. Here we may choose
 * how to configure the code, e.g. by loading a legacy configuration file
 * or some otherwise formatted configuration data set. The code can be
 * linked to a luncher script or to a GUI oriented application that performs
 * the configuration and runs the main program.
 *
 * \param argc: `int` The number of arguments given in command-line.
 * \param argv: `char **` The vector of command-line arguments.
 * \return result: `int` An exit code passed to the OS (0 for succesful execution).
 */
int main(int argc, char **argv) {
  string config_file = "../../test_data/sphere/DEDFB";
  string data_file = "../../test_data/sphere/DSPH";
  string output_path = ".";
  if (argc == 4) {
    config_file = string(argv[1]);
    data_file = string(argv[2]);
    output_path = string(argv[3]);
  }
  sphere(config_file, data_file, output_path);
  return 0;
}
