/* Copyright (C) 2024   INAF - Osservatorio Astronomico di Cagliari

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   A copy of the GNU General Public License is distributed along with
   this program in the COPYING file. If not, see: <https://www.gnu.org/licenses/>.
 */

/*! \file test_zinvert.cpp
 *
 * \brief Test of matrix inversion process.
 */
#include <cstdio>

#ifndef INCLUDE_TYPES_H_
#include "../include/types.h"
#endif

#ifdef USE_MAGMA
#include "../include/magma_calls.h"
#define SUCCESS MAGMA_SUCCESS
#else
#define SUCCESS 0
#endif

#ifdef USE_MAGMA_SVD
#include "magma_operators.h"
#endif

#include "../include/algebraic.h"

int main() {
  int result = (int)SUCCESS;
#ifdef USE_MAGMA
  magma_init();
#endif
  const np_int n = 2;
  const np_int nn = n * n;
  const double tolerance = 1.0e-6;
  dcomplex **test_matrix;
  dcomplex *vec_matrix = new dcomplex[nn];
  dcomplex *vec_inv_matrix = new dcomplex[nn];
  for (int ivi = 0; ivi < nn; ivi++) vec_matrix[ivi] = 1.0 + ivi + I * (1.0 + ivi);
  test_matrix = &vec_matrix;
  vec_inv_matrix[0] = -1.0 + I * 1.0;
  vec_inv_matrix[1] = 0.5 - I * 0.5;
  vec_inv_matrix[2] = 0.75 - I * 0.75;
  vec_inv_matrix[3] = -0.25 + I * 0.25;
#ifdef USE_MAGMA
  if (sizeof(np_int) != sizeof(magma_int_t)) {
    printf("ERROR: sizeof(np_int) = %ld; sizeof(magma_int_t) = %ld\n",
	   sizeof(np_int), sizeof(magma_int_t));
    result = 2;
  }
#endif
  if (result == 0) {
    invert_matrix(test_matrix, n, result, n);
    for (int tvi = 0; tvi < 4; tvi++) {
      dcomplex difference = vec_matrix[tvi] - vec_inv_matrix[tvi];
      if (real(difference) < -tolerance || real(difference) > tolerance) result = 1;
      if (imag(difference) < -tolerance || imag(difference) > tolerance) result = 1;
    }
    if (result != 0) printf("ERROR: failed matrix inversion test!\n");
  }
  delete[] vec_inv_matrix;
  delete[] vec_matrix;
#ifdef USE_MAGMA
  magma_finalize();
#endif
  return result;
}
