/* Copyright (C) 2024   INAF - Osservatorio Astronomico di Cagliari

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   A copy of the GNU General Public License is distributed along with
   this program in the COPYING file. If not, see: <https://www.gnu.org/licenses/>.
 */

/*! \file inclusion.cpp
 *
 * \brief Implementation of the calculation for a sphere with inclusions.
 */
#include <chrono>
#include <cstdio>
#include <exception>
#include <fstream>
#include <hdf5.h>
#include <string>
#ifdef _OPENMP
#include <omp.h>
#endif
#ifdef USE_MPI
#ifndef MPI_VERSION
#include <mpi.h>
#endif
#endif
#ifdef USE_NVTX
#include <nvtx3/nvToolsExt.h>
#endif
#ifdef USE_MAGMA
#include <cuda_runtime.h>
#endif

#ifndef INCLUDE_TYPES_H_
#include "../include/types.h"
#endif

#ifndef INCLUDE_ERRORS_H_
#include "../include/errors.h"
#endif

#ifndef INCLUDE_LOGGING_H_
#include "../include/logging.h"
#endif

#ifndef INCLUDE_CONFIGURATION_H_
#include "../include/Configuration.h"
#endif

#ifndef INCLUDE_COMMONS_H_
#include "../include/Commons.h"
#endif

#ifndef INCLUDE_SPH_SUBS_H_
#include "../include/sph_subs.h"
#endif

#ifndef INCLUDE_CLU_SUBS_H_
#include "../include/clu_subs.h"
#endif

#ifndef INCLUDE_TRANSITIONMATRIX_H_
#include "../include/TransitionMatrix.h"
#endif

#ifndef INCLUDE_ALGEBRAIC_H_
#include "../include/algebraic.h"
#endif

#ifndef INCLUDE_LIST_H_
#include "../include/List.h"
#endif

#ifndef INCLUDE_FILE_IO_H_
#include "../include/file_io.h"
#endif

#ifndef INCLUDE_UTILS_H_
#include "../include/utils.h"
#endif

using namespace std;

/*! \brief C++ implementation of INCLU
 *
 * \param config_file: `string` Name of the configuration file.
 * \param data_file: `string` Name of the input data file.
 * \param output_path: `string` Directory to write the output files in.
 * \param mpidata: `mixMPI *` Pointer to an instance of MPI data settings.
 */
void inclusion(const string& config_file, const string& data_file, const string& output_path, const mixMPI *mpidata) {
  ScattererConfiguration *sconf = ScattererConfiguration::from_dedfb(config_file);
  GeometryConfiguration *gconf = GeometryConfiguration::from_legacy(data_file);
  ParticleDescriptorSphere *pds = NULL;
  ParticleDescriptorCluster *pdc = NULL;
  ParticleDescriptorInclusion *pdi = NULL;
  if (gconf->number_of_spheres == 1) pds = new ParticleDescriptorSphere(gconf, sconf);
  else {
    if (sconf->use_external_sphere) pdi = new ParticleDescriptorInclusion(gconf, sconf);
    else pdc = new ParticleDescriptorCluster(gconf, sconf);
  }
  ParticleDescriptor *pd;
  if (pds != NULL) pd = pds;
  else if (pdc != NULL) pd = pdc;
  else pd = pdi;
  printf("INFO: %s\n", (pd->get_descriptor_type()).c_str());
  delete pd;
  delete gconf;
  delete sconf;
}
