/* Copyright (C) 2024   INAF - Osservatorio Astronomico di Cagliari

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   A copy of the GNU General Public License is distributed along with
   this program in the COPYING file. If not, see: <https://www.gnu.org/licenses/>.
 */

/*! \file sphere.cpp
 *
 * \brief Implementation of the single sphere calculation.
 */
#include <cstdio>
#include <exception>
#include <fstream>
#include <string>

#ifndef INCLUDE_TYPES_H_
#include "../include/types.h"
#endif

#ifndef INCLUDE_ERRORS_H_
#include "../include/errors.h"
#endif

#ifndef INCLUDE_LOGGING_H_
#include "../include/logging.h"
#endif

#ifndef INCLUDE_CONFIGURATION_H_
#include "../include/Configuration.h"
#endif

#ifndef INCLUDE_COMMONS_H_
#include "../include/Commons.h"
#endif

#ifndef INCLUDE_SPH_SUBS_H_
#include "../include/sph_subs.h"
#endif

#ifndef INCLUDE_TRANSITIONMATRIX_H_
#include "../include/TransitionMatrix.h"
#endif

#ifndef INCLUDE_OUTPUTS_H_
#include "../include/outputs.h"
#endif

using namespace std;

/*! \brief C++ implementation of SPH
 *
 *  \param config_file: `string` Name of the configuration file.
 *  \param data_file: `string` Name of the input data file.
 *  \param output_path: `string` Directory to write the output files in.
 */
void sphere(const string& config_file, const string& data_file, const string& output_path) {
  Logger *logger = new Logger(LOG_INFO);
  dcomplex arg, s0, tfsas;
  double th, ph;
  logger->log("INFO: making legacy configuration...\n");
  ScattererConfiguration *sconf = NULL;
  try {
    sconf = ScattererConfiguration::from_dedfb(config_file);
  } catch(const OpenConfigurationFileException &ex) {
    logger->err("\nERROR: failed to open scatterer configuration file.\n");
    string message = ex.what();
    logger->err("FILE: " + message + "\n");
    delete logger;
    exit(1);
  }
  sconf->write_formatted(output_path + "/c_OEDFB");
  sconf->write_binary(output_path + "/c_TEDF");
  sconf->write_binary(output_path + "/c_TEDF.hd5", "HDF5");
  GeometryConfiguration *gconf = NULL;
  try {
    gconf = GeometryConfiguration::from_legacy(data_file);
  } catch(const OpenConfigurationFileException &ex) {
    logger->err("\nERROR: failed to open geometry configuration file.\n");
    string message = ex.what();
    logger->err("FILE: " + message + "\n");
    if (sconf != NULL) delete sconf;
    delete logger;
    exit(1);
  }
  int s_nsph = sconf->number_of_spheres;
  int nsph = gconf->number_of_spheres;
  int configurations = sconf->configurations;
  if (s_nsph == nsph) {
    int isq, ibf;
    double *argi, *args, *gaps;
    double cost, sint, cosp, sinp;
    double costs, sints, cosps, sinps;
    double scan;
    double *duk = new double[3];
    double *u = new double[3];
    double *us = new double[3];
    double *un = new double[3];
    double *uns = new double[3];
    double *up = new double[3];
    double *ups = new double[3];
    double *upmp = new double[3];
    double *upsmp = new double[3];
    double *unmp = new double[3];
    double *unsmp = new double[3];
    double **cmul = new double*[4];
    double **cmullr = new double*[4];
    for (int i = 0; i < 4; i++) {
      cmul[i] = new double[4];
      cmullr[i] = new double[4];
    }
    dcomplex **tqspe, **tqsps;
    double **tqse, **tqss;
    tqse = new double*[2];
    tqss = new double*[2];
    tqspe = new dcomplex*[2];
    tqsps = new dcomplex*[2];
    for (int ti = 0; ti < 2; ti++) {
      tqse[ti] = new double[2]();
      tqss[ti] = new double[2]();
      tqspe[ti] = new dcomplex[2]();
      tqsps[ti] = new dcomplex[2]();
    }
    double frx = 0.0, fry = 0.0, frz = 0.0;
    double cfmp, cfsp, sfmp, sfsp;
    const dcomplex cc0 = 0.0 * I + 0.0;
    int jw;
    int l_max = gconf->l_max;
    ParticleDescriptor *c1 = new ParticleDescriptorSphere(gconf, sconf);
    int npnt = gconf->npnt;
    int npntts = gconf->npntts;
    int in_pol = gconf->in_pol;
    int meridional_type = gconf->iavm;
    int jwtm = gconf->jwtm;
    double in_theta_start = gconf->in_theta_start;
    double in_theta_step = gconf->in_theta_step;
    double in_theta_end = gconf->in_theta_end;
    double sc_theta_start = gconf->sc_theta_start;
    double sc_theta_step = gconf->sc_theta_step;
    double sc_theta_end = gconf->sc_theta_end;
    double in_phi_start = gconf->in_phi_start;
    double in_phi_step = gconf->in_phi_step;
    double in_phi_end = gconf->in_phi_end;
    double sc_phi_start = gconf->sc_phi_start;
    double sc_phi_step = gconf->sc_phi_step;
    double sc_phi_end = gconf->sc_phi_end;
    argi = new double[1];
    args = new double[1];
    gaps = new double[2];
    // FILE *output = fopen((output_path + "/c_OSPH").c_str(), "w");
    mixMPI *mpidata = new mixMPI(); // Fake MPI configuration
    SphereOutputInfo *p_output = new SphereOutputInfo(sconf, gconf, mpidata);
    double sml = 1.0e-3;
    int nth = 0, nph = 0;
    if (in_theta_step != 0.0)
      nth = int((in_theta_end - in_theta_start) / in_theta_step + sml);
    nth += 1;
    if (in_phi_step != 0.0)
      nph = int((in_phi_end - in_phi_start) / in_phi_step + sml);
    nph += 1;
    int nths = 0, nphs = 0;
    double thsca;
    if (meridional_type > 1) { // ISAM > 1, fixed scattering angle
      nths = 1;
      thsca = sc_theta_start - in_theta_start;
    } else { //ISAM <= 1
      if (in_theta_step != 0.0)
	nths = int((sc_theta_end - sc_theta_start) / sc_theta_step + sml);
      nths += 1;
      if (meridional_type == 1) { // ISAM = 1
	nphs = 1;
      } else { // ISAM < 1
	if (sc_phi_step != 0.0)
	  nphs = int((sc_phi_end - sc_phi_start) / sc_phi_step + sml);
	nphs += 1;
      }
    }
    int nk = nth * nph;
    int nks = nths * nphs;
    int nkks = nk * nks;
    double th1 = in_theta_start;
    double ph1 = in_phi_start;
    double ths1 = sc_theta_start;
    double phs1 = sc_phi_start;
    const double half_pi = acos(0.0);
    const double pi = 2.0 * half_pi;
    double gcs = 0.0;
    for (int i116 = 0; i116 < nsph; i116++) {
      int i = i116 + 1;
      int iogi = c1->iog[i116];
      if (iogi >= i) {
	double gcss = pi * c1->ros[i116] * c1->ros[i116];
	c1->gcsv[i116] = gcss;
	int nsh = c1->nshl[i116];
	for (int j115 = 0; j115 < nsh; j115++) {
	  c1->rc[i116][j115] = sconf->get_rcf(i116, j115) * c1->ros[i116];
	}
      }
      gcs += c1->gcsv[iogi - 1];
    }
    double ****zpv = new double***[l_max]; //[l_max][3][2][2]; // Matrix: dim[LM x 3 x 2 x 2]
    for (int zi = 0; zi < l_max; zi++) {
      zpv[zi] = new double**[3];
      for (int zj = 0; zj < 3; zj++) {
	zpv[zi][zj] = new double*[2];
	for (int zk = 0; zk < 2; zk++) {
	  zpv[zi][zj][zk] = new double[2]();
	}
      }
    }
    thdps(l_max, zpv);
    double exdc = sconf->exdc;
    double exri = sqrt(exdc);
    fstream tppoan;
    string tppoan_name = output_path + "/c_TPPOAN";
    tppoan.open(tppoan_name.c_str(), ios::binary|ios::out);
    if (tppoan.is_open()) {
      int imode = 10;
      tppoan.write(reinterpret_cast<char *>(&imode), sizeof(int));
      tppoan.write(reinterpret_cast<char *>(&(meridional_type)), sizeof(int));
      tppoan.write(reinterpret_cast<char *>(&(in_pol)), sizeof(int));
      tppoan.write(reinterpret_cast<char *>(&s_nsph), sizeof(int));
      tppoan.write(reinterpret_cast<char *>(&nth), sizeof(int));
      tppoan.write(reinterpret_cast<char *>(&nph), sizeof(int));
      tppoan.write(reinterpret_cast<char *>(&nths), sizeof(int));
      tppoan.write(reinterpret_cast<char *>(&nphs), sizeof(int));
      tppoan.write(reinterpret_cast<char *>(&nsph), sizeof(int));

      for (int nsi = 0; nsi < nsph; nsi++)
	tppoan.write(reinterpret_cast<char *>(&(c1->iog[nsi])), sizeof(int));
      double wp = sconf->wp;
      double xip = sconf->xip;
      double wn = wp / 3.0e8;
      double sqsfi = 1.0;
      double vk, vkarg;
      int idfc = sconf->idfc;
      int nxi = sconf->number_of_scales;
      if (idfc < 0) {
	vk = xip * wn;
	p_output->vec_vk[0] = vk;
      }
      int ndirs = nkks;
      for (int jxi488 = 0; jxi488 < nxi; jxi488++) {
	int oindex = 0;
	int jxi = jxi488 + 1;
	logger->log("INFO: running scale iteration " + to_string(jxi) + " of " + to_string(nxi) + ".\n");
	double xi = sconf->get_scale(jxi488);
	if (idfc >= 0) {
	  vk = xi * wn;
	  vkarg = vk;
	  p_output->vec_vk[jxi488] = vk;
	  p_output->vec_xi[jxi488] = xi;
	} else { // IDFC < 0
	  vkarg = xi * vk;
	  sqsfi = 1.0 / (xi * xi);
	  p_output->vec_vk[jxi488] = vk;
	  p_output->vec_xi[jxi488] = xi;
	}
	tppoan.write(reinterpret_cast<char *>(&vk), sizeof(double));
	for (int i132 = 0; i132 < nsph; i132++) {
	  int i = i132 + 1;
	  int iogi = c1->iog[i132];
	  if (iogi != i) {
	    for (int l123 = 0; l123 < l_max; l123++) {
	      c1->rmi[l123][i132] = c1->rmi[l123][iogi - 1];
	      c1->rei[l123][i132] = c1->rei[l123][iogi - 1];
	    }
	    continue; // i132
	  }
	  // label 125
	  int nsh = c1->nshl[i132];
	  int ici = (nsh + 1) / 2;
	  if (idfc == 0) {
	    for (int ic = 0; ic < ici; ic++)
	      c1->dc0[ic] = sconf->get_dielectric_constant(ic, i132, jxi488); // WARNING: IDFC=0 is not tested!
	  } else { // IDFC != 0
	    if (jxi == 1) {
	      for (int ic = 0; ic < ici; ic++) {
		c1->dc0[ic] = sconf->get_dielectric_constant(ic, i132, jxi488);
	      }
	    }
	  }
	  if (nsh % 2 == 0) c1->dc0[ici] = exdc;
	  int jer = 0;
	  int lcalc = 0;
	  dme(l_max, i, npnt, npntts, vkarg, exdc, exri, c1, jer, lcalc, arg);
	  if (jer != 0) {
	    p_output->vec_ier[jxi] = 1;
	    p_output->lcalc = lcalc;
	    p_output->arg = arg;
	    tppoan.close();
	    delete sconf;
	    delete gconf;
	    delete c1;
	    for (int zi = l_max - 1; zi > -1; zi--) {
	      for (int zj = 0; zj < 3; zj++) {
		for (int zk = 0; zk < 2; zk++) {
		  delete[] zpv[zi][zj][zk];
		}
		delete[] zpv[zi][zj];
	      }
	      delete[] zpv[zi];
	    }
	    delete[] zpv;
	    delete[] duk;
	    delete[] u;
	    delete[] us;
	    delete[] un;
	    delete[] uns;
	    delete[] up;
	    delete[] ups;
	    delete[] upmp;
	    delete[] upsmp;
	    delete[] unmp;
	    delete[] unsmp;
	    delete[] argi;
	    delete[] args;
	    delete[] gaps;
	    for (int i = 3; i > -1; i--) {
	      delete[] cmul[i];
	      delete[] cmullr[i];
	    }
	    delete[] cmul;
	    delete[] cmullr;
	    for (int ti = 1; ti > -1; ti--) {
	      delete[] tqse[ti];
	      delete[] tqss[ti];
	      delete[] tqspe[ti];
	      delete[] tqsps[ti];
	    }
	    delete[] tqse;
	    delete[] tqss;
	    delete[] tqspe;
	    delete[] tqsps;
	    delete logger;
	    return;
	  }
	} // i132
	if (idfc >= 0 and nsph == 1 and jxi == jwtm) {
	  // This is the condition that writes the transition matrix to output.
	  string ttms_name = output_path + "/c_TTMS.hd5";
	  TransitionMatrix::write_binary(
					 ttms_name, l_max, vk, exri, c1->rmi, c1->rei,
					 sconf->get_radius(0), "HDF5"
					 );
	  ttms_name = output_path + "/c_TTMS";
	  TransitionMatrix::write_binary(
					 ttms_name, l_max, vk, exri, c1->rmi, c1->rei,
					 sconf->get_radius(0)
					 );
	}
	double cs0 = 0.25 * vk * vk * vk / half_pi;
	sscr0(tfsas, nsph, l_max, vk, exri, c1);
	double sqk = vk * vk * exdc;
	aps(zpv, l_max, nsph, c1, sqk, gaps);
	rabas(in_pol, l_max, nsph, c1, tqse, tqspe, tqss, tqsps);
	int last_configuration = 0;
	for (int i170 = 0; i170 < nsph; i170++) {
	  int i = i170 + 1;
	  if (c1->iog[i170] >= i) {
	    last_configuration++;
	    oindex = jxi488 * configurations + last_configuration - 1;
	    double albeds = c1->sscs[i170] / c1->sexs[i170];
	    c1->sqscs[i170] *= sqsfi;
	    c1->sqabs[i170] *= sqsfi;
	    c1->sqexs[i170] *= sqsfi;
	    if (c1->nshl[i170] != 1) {
	      p_output->vec_sphere_ref_indices[oindex] = cc0;
	      p_output->vec_sphere_sizes[oindex] = c1->vsz[i170];
	    } else {
	      p_output->vec_sphere_ref_indices[oindex] = c1->vkt[i170];
	      p_output->vec_sphere_sizes[oindex] = c1->vsz[i170];
	    }
	    p_output->vec_scs[oindex] = c1->sscs[i170];
	    p_output->vec_abs[oindex] = c1->sabs[i170];
	    p_output->vec_exs[oindex] = c1->sexs[i170];
	    p_output->vec_albeds[oindex] = albeds;
	    p_output->vec_scsrt[oindex] = c1->sqscs[i170];
	    p_output->vec_absrt[oindex] = c1->sqabs[i170];
	    p_output->vec_exsrt[oindex] = c1->sqexs[i170];
	    p_output->vec_fsas[oindex] = c1->fsas[i170];
	    double csch = 2.0 * vk * sqsfi / c1->gcsv[i170];
	    s0 = c1->fsas[i170] * exri;
	    double qschu = csch * imag(s0);
	    double pschu = csch * real(s0);
	    double s0mag = cs0 * cabs(s0);
	    p_output->vec_qschu[oindex] = qschu;
	    p_output->vec_pschu[oindex] = pschu;
	    p_output->vec_s0mag[oindex] = s0mag;
	    double rapr = c1->sexs[i170] - gaps[i170];
	    double cosav = gaps[i170] / c1->sscs[i170];
	    p_output->vec_cosav[oindex] = cosav;
	    p_output->vec_raprs[oindex] = rapr;
	    p_output->vec_tqek1[oindex] = tqse[0][i170];
	    p_output->vec_tqsk1[oindex] = tqss[0][i170];
	    p_output->vec_tqek2[oindex] = tqse[1][i170];
	    p_output->vec_tqsk2[oindex] = tqss[1][i170];
	    tppoan.write(reinterpret_cast<char *>(&(tqse[0][i170])), sizeof(double));
	    tppoan.write(reinterpret_cast<char *>(&(tqss[0][i170])), sizeof(double));
	    double val = real(tqspe[0][i170]);
	    tppoan.write(reinterpret_cast<char *>(&val), sizeof(double));
	    val = imag(tqspe[0][i170]);
	    tppoan.write(reinterpret_cast<char *>(&val), sizeof(double));
	    val = real(tqsps[0][i170]);
	    tppoan.write(reinterpret_cast<char *>(&val), sizeof(double));
	    val = imag(tqsps[0][i170]);
	    tppoan.write(reinterpret_cast<char *>(&val), sizeof(double));
	    tppoan.write(reinterpret_cast<char *>(&(tqse[1][i170])), sizeof(double));
	    tppoan.write(reinterpret_cast<char *>(&(tqss[1][i170])), sizeof(double));
	    val = real(tqspe[1][i170]);
	    tppoan.write(reinterpret_cast<char *>(&val), sizeof(double));
	    val = imag(tqspe[1][i170]);
	    tppoan.write(reinterpret_cast<char *>(&val), sizeof(double));
	    val = real(tqsps[1][i170]);
	    tppoan.write(reinterpret_cast<char *>(&val), sizeof(double));
	    val = imag(tqsps[1][i170]);
	    tppoan.write(reinterpret_cast<char *>(&val), sizeof(double));
	  } // End if iog[i170] >= i
	} // i170 loop
	if (nsph != 1) {
	  p_output->vec_fsat[jxi488] = tfsas;
	  double csch = 2.0 * vk * sqsfi / gcs;
	  s0 = tfsas * exri;
	  double qschu = csch * imag(s0);
	  double pschu = csch * real(s0);
	  double s0mag = cs0 * cabs(s0);
	  p_output->vec_qschut[jxi488] = qschu;
	  p_output->vec_pschut[jxi488] = pschu;
	  p_output->vec_s0magt[jxi488] = s0mag;
	}
	th = th1;
	int done_dirs = 0;
	for (int jth486 = 0; jth486 < nth; jth486++) { // OpenMP parallelizable section
	  int jth = jth486 + 1;
	  ph = ph1;
	  for (int jph484 = 0; jph484 < nph; jph484++) {
	    int jph = jph484 + 1;
	    bool goto182 = (nk == 1) && (jxi > 1);
	    if (!goto182) {
	      upvmp(th, ph, 0, cost, sint, cosp, sinp, u, upmp, unmp);
	    }
	    if (meridional_type >= 0) {
	      wmamp(0, cost, sint, cosp, sinp, in_pol, l_max, 0, nsph, argi, u, upmp, unmp, c1);
	      for (int i183 = 0; i183 < nsph; i183++) {
		double rapr = c1->sexs[i183] - gaps[i183];
		frx = rapr * u[0];
		fry = rapr * u[1];
		frz = rapr * u[2];
	      }
	      jw = 1;
	    }
	    double thsl = ths1;
	    double phsph = 0.0;
	    for (int jths482 = 0; jths482 < nths; jths482++) {
	      int jths = jths482 + 1;
	      double ths = thsl;
	      int icspnv = 0;
	      if (meridional_type > 1) ths = th + thsca;
	      if (meridional_type >= 1) {
		phsph = 0.0;
		if ((ths < 0.0) || (ths > 180.0)) phsph = 180.0;
		if (ths < 0.0) ths *= -1.0;
		if (ths > 180.0) ths = 360.0 - ths;
		if (phsph != 0.0) icspnv = 1;
	      }
	      double phs = phs1;
	      for (int jphs480 = 0; jphs480 < nphs; jphs480++) {
		int jphs = jphs480 + 1;
		if (meridional_type >= 1) {
		  phs = ph + phsph;
		  if (phs >= 360.0) phs -= 360.0;
		}
		bool goto190 = (nks == 1) && ((jxi > 1) || (jth > 1) || (jph > 1));
		if (!goto190) {
		  upvmp(ths, phs, icspnv, costs, sints, cosps, sinps, us, upsmp, unsmp);
		  if (meridional_type >= 0) {
		    wmamp(2, costs, sints, cosps, sinps, in_pol, l_max, 0, nsph, args, us, upsmp, unsmp, c1);
		  }
		}
		if (nkks != 0 || jxi == 1) {
		  upvsp(u, upmp, unmp, us, upsmp, unsmp, up, un, ups, uns, duk, isq, ibf, scan, cfmp, sfmp, cfsp, sfsp);
		  if (meridional_type < 0) {
		    wmasp(
			  cost, sint, cosp, sinp, costs, sints, cosps, sinps,
			  u, up, un, us, ups, uns, isq, ibf, in_pol,
			  l_max, 0, nsph, argi, args, c1
			  );
		  }
		  for (int i193 = 0; i193 < 3; i193++) {
		    un[i193] = unmp[i193];
		    uns[i193] = unsmp[i193];
		  }
		}
		if (meridional_type < 0) jw = 1;
		tppoan.write(reinterpret_cast<char *>(&th), sizeof(double));
		tppoan.write(reinterpret_cast<char *>(&ph), sizeof(double));
		tppoan.write(reinterpret_cast<char *>(&ths), sizeof(double));
		tppoan.write(reinterpret_cast<char *>(&phs), sizeof(double));
		tppoan.write(reinterpret_cast<char *>(&scan), sizeof(double));
		if (jw != 0) {
		  jw = 0;
		  tppoan.write(reinterpret_cast<char *>(&(u[0])), sizeof(double));
		  tppoan.write(reinterpret_cast<char *>(&(u[1])), sizeof(double));
		  tppoan.write(reinterpret_cast<char *>(&(u[2])), sizeof(double));
		}
		p_output->vec_dir_scand[done_dirs] = scan;
		p_output->vec_dir_cfmp[done_dirs] = cfmp;
		p_output->vec_dir_cfsp[done_dirs] = cfsp;
		p_output->vec_dir_sfmp[done_dirs] = sfmp;
		p_output->vec_dir_sfsp[done_dirs] = sfsp;
		if (meridional_type >= 0) {
		  p_output->vec_dir_un[3 * done_dirs] = un[0];
		  p_output->vec_dir_un[3 * done_dirs + 1] = un[1];
		  p_output->vec_dir_un[3 * done_dirs + 2] = un[2];
		  p_output->vec_dir_uns[3 * done_dirs] = uns[0];
		  p_output->vec_dir_uns[3 * done_dirs + 1] = uns[1];
		  p_output->vec_dir_uns[3 * done_dirs + 2] = uns[2];
		} else {
		  p_output->vec_dir_un[3 * done_dirs] = un[0];
		  p_output->vec_dir_un[3 * done_dirs + 1] = un[1];
		  p_output->vec_dir_un[3 * done_dirs + 2] = un[2];
		  p_output->vec_dir_uns[3 * done_dirs] = 0.0;
		  p_output->vec_dir_uns[3 * done_dirs + 1] = 0.0;
		  p_output->vec_dir_uns[3 * done_dirs + 2] = 0.0;
		}
		sscr2(nsph, l_max, vk, exri, c1);
		last_configuration = 0;
		for (int ns226 = 0; ns226 < nsph; ns226++) {
		  int ns = ns226 + 1;
		  oindex = jxi488 * nsph * ndirs + nsph * done_dirs + ns226;
		  p_output->vec_dir_sas11[oindex] = c1->sas[ns226][0][0];
		  p_output->vec_dir_sas21[oindex] = c1->sas[ns226][1][0];
		  p_output->vec_dir_sas12[oindex] = c1->sas[ns226][0][1];
		  p_output->vec_dir_sas22[oindex] = c1->sas[ns226][1][1];
		  p_output->vec_dir_fx[jxi488 * nsph * nth * nph + nsph * nph * (jth - 1) + nsph * (jph - 1) + ns226] = frx;
		  p_output->vec_dir_fy[jxi488 * nsph * nth * nph + nsph * nph * (jth - 1) + nsph * (jph - 1) + ns226] = fry;
		  p_output->vec_dir_fz[jxi488 * nsph * nth * nph + nsph * nph * (jth - 1) + nsph * (jph - 1) + ns226] = frz;
		  for (int i225 = 0; i225 < 16; i225++) c1->vint[i225] = c1->vints[ns226][i225];
		  mmulc(c1->vint, cmullr, cmul);
		  for (int imul = 0; imul < 4; imul++) {
		    int muls_index = 16 * jxi488 * nsph * ndirs + 16 * nsph * done_dirs + 4 * imul;
		    for (int jmul = 0; jmul < 4; jmul++) {
		      p_output->vec_dir_muls[muls_index + jmul] = cmul[imul][jmul];
		    }
		  }
		  for (int imul = 0; imul < 4; imul++) {
		    int muls_index = 16 * jxi488 * nsph * ndirs + 16 * nsph * done_dirs + 4 * imul;
		    for (int jmul = 0; jmul < 4; jmul++) {
		      p_output->vec_dir_mulslr[muls_index + jmul] = cmullr[imul][jmul];
		    }
		  }
		  for (int vi = 0; vi < 16; vi++) {
		    double value = real(c1->vint[vi]);
		    tppoan.write(reinterpret_cast<char *>(&value), sizeof(double));
		    value = imag(c1->vint[vi]);
		    tppoan.write(reinterpret_cast<char *>(&value), sizeof(double));
		  }
		  for (int imul = 0; imul < 4; imul++) {
		    for (int jmul = 0; jmul < 4; jmul++) {
		      tppoan.write(reinterpret_cast<char *>(&(cmul[imul][jmul])), sizeof(double));
		    }
		  }
		} // ns226 loop
		if (meridional_type < 1) phs += sc_phi_step;
		done_dirs++;
	      } // jphs480 loop
	      if (meridional_type <= 1) thsl += sc_theta_step;
	    } // jths482 loop
	    ph += in_phi_step;
	  } // jph484 loop on elevation
	  th += in_theta_step;
	} // jth486 loop on azimuth
	p_output->vec_jxi[jxi488] = jxi488 + 1;
	logger->log("INFO: finished scale iteration " + to_string(jxi) + " of " + to_string(nxi) + ".\n");
      } //jxi488 loop on scales
      tppoan.close();
    } else { // In case TPPOAN could not be opened. Should never happen.
      logger->err("ERROR: failed to open TPPOAN file.\n");
    }
    // fclose(output);
    p_output->write(output_path + "/c_OSPH.hd5", "HDF5");
    p_output->write(output_path + "/c_OSPH", "LEGACY");
    delete p_output;
    delete mpidata;
    delete c1;
    for (int zi = l_max - 1; zi > -1; zi--) {
      for (int zj = 0; zj < 3; zj++) {
	for (int zk = 0; zk < 2; zk++) {
	  delete[] zpv[zi][zj][zk];
	}
	delete[] zpv[zi][zj];
      }
      delete[] zpv[zi];
    }
    delete[] zpv;
    delete[] duk;
    delete[] u;
    delete[] us;
    delete[] un;
    delete[] uns;
    delete[] up;
    delete[] ups;
    delete[] upmp;
    delete[] upsmp;
    delete[] unmp;
    delete[] unsmp;
    delete[] argi;
    delete[] args;
    delete[] gaps;
    for (int i = 3; i > -1; i--) {
      delete[] cmul[i];
      delete[] cmullr[i];
    }
    delete[] cmul;
    delete[] cmullr;
    for (int ti = 1; ti > -1; ti--) {
      delete[] tqse[ti];
      delete[] tqss[ti];
      delete[] tqspe[ti];
      delete[] tqsps[ti];
    }
    delete[] tqse;
    delete[] tqss;
    delete[] tqspe;
    delete[] tqsps;
    logger->log("Finished: output written to " + output_path + "/c_OSPH.\n");
  } else { // NSPH mismatch between geometry and scatterer configurations.
    throw UnrecognizedConfigurationException(
					     "Inconsistent geometry and scatterer configurations."
					     );
  }
  delete sconf;
  delete gconf;
  delete logger;
}
