#ifndef CONFIGURATION_H
#define	CONFIGURATION_H

#include <map>
#include <string>
#include <vector>

#include <boost/shared_ptr.hpp>

namespace DataExporter_ns
{

class Configuration
{
public:
//------------------------------------------------------------------------------
//	[Public] Shared pointer typedef
//------------------------------------------------------------------------------
	typedef boost::shared_ptr<Configuration> SP;

//------------------------------------------------------------------------------
//	[Public] Map typedef
//------------------------------------------------------------------------------
	typedef const std::multimap<const std::string, const std::string> ExportedTablesMap;
	typedef const std::map<const std::string, const std::string> AuthorisedUsersMap;

protected:
//------------------------------------------------------------------------------
//	[Private] Constructor destructor deleter
//------------------------------------------------------------------------------
	Configuration(std::string certificateFile, std::string privateKeyFile,
        std::string dHTempFile, ExportedTablesMap exportedTablesMap,
        AuthorisedUsersMap authorisedUsersMap, std::string localHost,
        unsigned int localPort, unsigned int workerNumber,
        std::string databaseHost, unsigned int databasePort,
        std::string databaseUsername, std::string databasePassword,
        unsigned int databaseConnectionNumber) :
        m_certificateFile(certificateFile), m_privateKeyFile(privateKeyFile),
        m_dHTempFile(dHTempFile), m_exportedTablesMap(exportedTablesMap),
        m_authorisedUsersMap(authorisedUsersMap), m_localHost(localHost),
        m_localPort(localPort), m_workerNumber(workerNumber),
        m_databaseHost(databaseHost), m_databasePort(databasePort),
        m_databaseUsername(databaseUsername), m_databasePassword(databasePassword),
        m_databaseConnectionNumber(databaseConnectionNumber) {}

	virtual ~Configuration() {}

	class Deleter;
	friend class Deleter;
	class Deleter
	{
	public:
		void operator()(Configuration* c) { delete c; }
	};

public:
//------------------------------------------------------------------------------
//	[Public] Create class method
//------------------------------------------------------------------------------
	static Configuration::SP create(std::string certificateFile,
        std::string privateKeyFile, std::string dHTempFile,
        ExportedTablesMap exportedTablesMap, AuthorisedUsersMap authorisedUsersMap,
        std::string localHost, unsigned int localPort, unsigned int workerNumber,
        std::string databaseHost, unsigned int databasePort,
        std::string databaseUsername, std::string databasePassword,
        unsigned int databaseConnectionNumber)
	{
		Configuration::SP c_sp(new Configuration(certificateFile, privateKeyFile,
            dHTempFile, exportedTablesMap, authorisedUsersMap, localHost, localPort,
            workerNumber, databaseHost, databasePort, databaseUsername,
            databasePassword, databaseConnectionNumber), Configuration::Deleter());

		return c_sp;
	}

//------------------------------------------------------------------------------
//	[Public] Getter methods
//------------------------------------------------------------------------------
	std::string getCertificateFile() const { return m_certificateFile; }
	std::string getPrivateKeyFile() const { return m_privateKeyFile; }
	std::string getDHTempFile() const { return m_dHTempFile; }
    ExportedTablesMap& getExportedTablesMap() const { return m_exportedTablesMap; }
    AuthorisedUsersMap& getAuthorisedUsersMap() const { return m_authorisedUsersMap; }
	std::string getLocalHost() const { return m_localHost; }
	unsigned int getLocalPort() const { return m_localPort; }
	unsigned int getWorkerNumber() const { return m_workerNumber; }
	std::string getDatabaseHost() const { return m_databaseHost; }
	unsigned int getDatabasePort() const { return m_databasePort; }
	std::string getDatabaseUsername() const { return m_databaseUsername; }
	std::string getDatabasePassword() const { return m_databasePassword; }
    unsigned int getDatabaseConnectionNumber() const { return m_databaseConnectionNumber; }

//------------------------------------------------------------------------------
//	[Public] Utilities methods
//------------------------------------------------------------------------------
    bool isTableExported(const std::string schema, const std::string table)
    {
        std::pair<ExportedTablesMap::const_iterator, ExportedTablesMap::const_iterator > ret;

        ret = m_exportedTablesMap.equal_range(schema);

        ExportedTablesMap::const_iterator it;

        for(it=ret.first; it!=ret.second; ++it)
        {
            if(it->second.compare(table) == 0)
                return true;
        }

        return false;
    }

    bool isUserAuthorized(const std::string username, const std::string password)
    {
        AuthorisedUsersMap::const_iterator it = m_authorisedUsersMap.find(username);

        if(it!=m_authorisedUsersMap.end() &&
            it->second.compare(password)==0)
            return true;

        return false;
    }

protected:
//------------------------------------------------------------------------------
//	[Private] class variables
//------------------------------------------------------------------------------
	//Absolute path to certificate chain file
	const std::string m_certificateFile;

	//Absolute path to private key file
	const std::string m_privateKeyFile;

	//Absolute path to Diffie Hellman temporary file
	const std::string m_dHTempFile;

    //Exported tables multi map [schema table]
	ExportedTablesMap m_exportedTablesMap;

    //Authorised users map [user password]
	AuthorisedUsersMap m_authorisedUsersMap;

	//Local host address for incoming connection
	const std::string m_localHost;

	//Local port for wait incoming connection
	const unsigned int m_localPort;

	//Number of threads that call io service run methods
	const unsigned int m_workerNumber;

	//Metadata database host
	const std::string m_databaseHost;

	//Metadata database port
	const unsigned int m_databasePort;

	//Metadata database login username
	const std::string m_databaseUsername;

	//Metadata database login password
	const std::string m_databasePassword;

    //Metadata database connections number
    const unsigned int m_databaseConnectionNumber;
};

}   //End of namespace

#endif	/* CONFIGURATION_H */

