#include <ProtocolManager.h>

#include <boost/date_time.hpp>


namespace DataExporter_ns
{

//==============================================================================
//      ProtocolManager::ProtocolManager()
//==============================================================================
ProtocolManager::ProtocolManager(Tango::DeviceImpl* deviceImpl_p,
    Configuration::SP configuration_sp, DBManager::SP dBManager_sp) :
    Tango::LogAdapter(deviceImpl_p), m_deviceImpl_p(deviceImpl_p),
    m_configuration_sp(configuration_sp), m_dBManager_sp(dBManager_sp)
{
    DEBUG_STREAM << "ProtocolManager::ProtocolManager()" << endl;
}

//==============================================================================
//      ProtocolManager::~ProtocolManager()
//==============================================================================
ProtocolManager::~ProtocolManager()
{
    DEBUG_STREAM << "ProtocolManager::~ProtocolManager()" << endl;
}

//==============================================================================
//      ProtocolManager::create()
//==============================================================================
ProtocolManager::SP ProtocolManager::create(Tango::DeviceImpl* deviceImpl_p,
    Configuration::SP configuration_sp, DBManager::SP dBManager_sp)
{
    ProtocolManager::SP d_sp(new ProtocolManager(deviceImpl_p, configuration_sp,
        dBManager_sp), ProtocolManager::Deleter());

    return d_sp;
}

//==============================================================================
//      ProtocolManager::setRemoteEndpoint()
//==============================================================================
void ProtocolManager::setRemoteEndpoint(std::string remoteEndpoint)
{
    DEBUG_STREAM << "ProtocolManager::setRemoteEndpoint()" << endl;

    m_remoteEndpoint = remoteEndpoint;
}

//==============================================================================
//      ProtocolManager::prepareResponse()
//==============================================================================
ResponseSP ProtocolManager::prepareResponse(RequestSP request_sp)
        /*throw(std::runtime_error)*/
{
    DEBUG_STREAM << "ProtocolManager::prepareResponse()" << endl;

    m_fileFound = false;
    m_fileWrapper_sp.reset();

    if(!request_sp->IsInitialized())
        throw std::runtime_error("Not initialized request!");

    ResponseSP response_sp(new Response());

    std::string username = request_sp->username();
    std::string password = request_sp->password();

    if(m_configuration_sp->isUserAuthorized(username, password))
    {
        std::string schema = request_sp->schema();
        std::string table =request_sp->table();

        if(m_configuration_sp->isTableExported(schema, table))
        {
            int fileVersion = request_sp->file_version();
            std::string fileName = request_sp->file_name();

            try
            {
                DBManager::FileTuple fileTuple = m_dBManager_sp->retrieveFileInfo(
                    schema, table, fileVersion, fileName);

                if(fileTuple.get<0>() && fileTuple.get<1>())
                {
                    std::string storagePath = fileTuple.get<0>().get();
                    std::string filePath = fileTuple.get<1>().get();

                    boost::filesystem::path absolutePath = composePath(
                            storagePath, filePath, fileVersion,fileName);

                    if(boost::filesystem::is_regular_file(absolutePath))
                    {
                        boost::uint64_t fileSize =
                            boost::filesystem::file_size(absolutePath);

                        INFO_STREAM << "ProtocolManager::prepareResponse() "
                            << " transfer file " << fileName << " version "
                            << fileVersion << " size " << fileSize << endl;

                        m_fileFound = true;
                        m_fileWrapper_sp =
                            FileWrapper::create(m_deviceImpl_p, absolutePath);

                        response_sp->set_state(Response::REQUEST_ACCEPTED);
                        response_sp->set_status("Request accepted");

                        response_sp->set_file_path(filePath);
                        response_sp->set_file_version(fileVersion);
                        response_sp->set_file_name(fileName);
                        response_sp->set_file_size(fileSize);
                    }
                    else
                    {
                        WARN_STREAM << "ProtocolManager::prepareResponse() file "
                            << "not found from " << m_remoteEndpoint << endl;

                        response_sp->set_state(Response::FILE_NOT_FOUND);
                        response_sp->set_status("File not found");
                    }
                }
                else
                {
                    WARN_STREAM << "ProtocolManager::prepareResponse() file not "
                        << "downloaded from " << m_remoteEndpoint << endl;

                    response_sp->set_state(Response::FILE_NOT_DOWNLOADED);
                    response_sp->set_status("File not downloaded");
                }
            }
            catch(soci::soci_error& ex)
            {
                WARN_STREAM << "ProtocolManager::prepareResponse() "
                    << ex.what() << " from " << m_remoteEndpoint << endl;

                response_sp->set_state(Response::METADATA_NOT_FOUND);
                response_sp->set_status(ex.what());
            }
        }
        else
        {
            WARN_STREAM << "ProtocolManager::prepareResponse() "
                << "Not exported table from " << m_remoteEndpoint << endl;

            response_sp->set_state(Response::TABLE_NOT_EXPORTED);
            response_sp->set_status("Table not exported");
        }
    }
    else
    {
        WARN_STREAM << "ProtocolManager::prepareResponse() Invalid username "
            << "or password from " << m_remoteEndpoint << endl;

        response_sp->set_state(Response::ACCESS_DENY);
        response_sp->set_status("Invalid credential");
    }

    if(!response_sp->IsInitialized())
        throw std::runtime_error("Not initialized response!");

    return response_sp;
}

//==============================================================================
//      ProtocolManager::isFileFound()
//==============================================================================
bool ProtocolManager::isFileFound()
{
    DEBUG_STREAM << "ProtocolManager::isFileFound()" << endl;

    return m_fileFound;
}

//==============================================================================
//      ProtocolManager::getFileWrapper()
//==============================================================================
FileWrapper::SP ProtocolManager::getFileWrapper() /*throw(std::runtime_error)*/
{
    DEBUG_STREAM << "ProtocolManager::getFileWrapper()" << endl;

    if(!m_fileWrapper_sp)
        throw std::runtime_error("File wrapper not created");

    return m_fileWrapper_sp;
}

//==============================================================================
//      ProtocolManager::composePath()
//==============================================================================
boost::filesystem::path ProtocolManager::composePath(std::string storagePath,
     std::string filePath, int fileVersion, std::string fileName)
{
    DEBUG_STREAM << "ProtocolManager::composePath()" << endl;

    boost::filesystem::path absolutePath(storagePath);

    absolutePath /= filePath;

    std::stringstream fileStream;
    fileStream << "/" << fileVersion << "/" << fileName;

    absolutePath /= fileStream.str();

    DEBUG_STREAM << "ProtocolManager::composePath() "
        << absolutePath.string() << endl;

    return absolutePath;
}

}   //namespace
