#include <PlainClient.h>
#include <DataImporter.h>

#include <boost/lexical_cast.hpp>
#include <boost/bind.hpp>
#include <ios>
#include <vector>

namespace DataImporter_ns
{

//==============================================================================
//      PlainClient::PlainClient()
//==============================================================================
PlainClient::PlainClient(DataImporter* dataImporter_p,
    Configuration::SP configuration_sp) : Client(dataImporter_p, configuration_sp),
    m_plainSocket(m_ioService)
{
    DEBUG_STREAM << "PlainClient::PlainClient()" << endl;
}

//==============================================================================
//      PlainClient::~PlainClient()
//==============================================================================
PlainClient::~PlainClient()
{
    DEBUG_STREAM << "PlainClient::~PlainClient()" << endl;

    closeConnection();
}

//==============================================================================
//      PlainClient::create()
//==============================================================================
Client::SP PlainClient::create(DataImporter* dataImporter_p,
    Configuration::SP configuration_sp)
{
    Client::SP c_sp(new PlainClient(dataImporter_p, configuration_sp),
        PlainClient::Deleter());

    return c_sp;
}

//==============================================================================
//      PlainClient::startConnect()
//==============================================================================
void PlainClient::startConnect(boost::asio::ip::tcp::resolver::iterator endPointIterator)
{
    DEBUG_STREAM << "PlainClient::startConnect()" << endl;

    m_resetConnectionTimer.expires_from_now(
        boost::posix_time::seconds(m_configuration_sp->getTimeout()));

    if(endPointIterator != boost::asio::ip::tcp::resolver::iterator())
    {
        std::stringstream infoStream;
        infoStream << "Connecting to " << endPointIterator->endpoint();

        DEBUG_STREAM << "PlainClient::startConnect() " << infoStream.str() << endl;

        writeState(Tango::RUNNING);
        writeStatus(infoStream.str());

        m_plainSocket.async_connect(endPointIterator->endpoint(),
            boost::bind(&PlainClient::handleConnect, this,
            boost::asio::placeholders::error, endPointIterator));
    }
    else
    {
        ERROR_STREAM << "PlainClient::startConnect() no more endpoint" << endl;

        writeState(Tango::ALARM);
        writeStatus("No more endpoint");
    }
}

//==============================================================================
//      PlainClient::handleConnect()
//==============================================================================
void PlainClient::handleConnect(const boost::system::error_code& errorCode,
    boost::asio::ip::tcp::resolver::iterator endPointIterator)
{
    DEBUG_STREAM << "PlainClient::handleConnect()" << endl;

    if(!errorCode)
    {
        m_remoteEndpoint = boost::lexical_cast<std::string>(
            m_plainSocket.remote_endpoint());

        std::stringstream infoStream;
        infoStream << "Connected to " << m_remoteEndpoint;

        INFO_STREAM << "PlainClient::handleConnect() " << m_remoteEndpoint << endl;

        writeState(Tango::RUNNING);
        writeStatus(infoStream.str());

        m_protocolManager_sp->setRemoteEndpoint(m_remoteEndpoint);

        startWriteRequest();
    }
    else
    {
        ERROR_STREAM << "PlainClient::handleConnect() " << errorCode.message() << endl;

        writeState(Tango::ALARM);
        writeStatus(errorCode.message());

        if(m_plainSocket.is_open())
            m_plainSocket.close();

        startConnect(++endPointIterator);
    }
}

//==============================================================================
//      PlainClient::startRequest()
//==============================================================================
void PlainClient::startWriteRequest()
{
    DEBUG_STREAM << "PlainClient::startRequest()" << endl;

    try
    {
        RequestSP request_sp = m_protocolManager_sp->createtRequest();

        boost::uint32_t bodySize = request_sp->ByteSizeLong();

        #ifdef VERBOSE_DEBUG
            INFO_STREAM << "PlainClient::startRequest() "
                << m_remoteEndpoint << " <<<< " << bodySize << " byte" << endl;
        #endif

        std::vector<boost::uint8_t> writeBuff;
        writeBuff.resize(HEADER_SIZE + bodySize);

        encodeHeader(writeBuff, bodySize);

        request_sp->SerializeToArray(&writeBuff[HEADER_SIZE], bodySize);

        m_resetConnectionTimer.expires_from_now(
            boost::posix_time::seconds(m_configuration_sp->getTimeout()));

        boost::asio::async_write(m_plainSocket, boost::asio::buffer(writeBuff),
            boost::bind(&PlainClient::handleWriteRequest, this,
            boost::asio::placeholders::error));
    }
    catch(std::exception& ec)
    {
        ERROR_STREAM << "PlainClient::startWriteRequest() " << ec.what() << endl;

        writeState(Tango::ALARM);
        writeStatus(ec.what());
    }
    catch(...)
    {
        ERROR_STREAM << "PlainClient::startWriteRequest() unknown error" << endl;

        writeState(Tango::ALARM);
        writeStatus("Unknown error");
    }
}

//==============================================================================
//      PlainClient::startReadResponseHeader()
//==============================================================================
void PlainClient::startReadResponseHeader()
{
    DEBUG_STREAM << "PlainClient::startReadResponseHeader()" << endl;

    m_readBuff.resize(HEADER_SIZE);

    m_resetConnectionTimer.expires_from_now(
        boost::posix_time::seconds(m_configuration_sp->getTimeout()));

    boost::asio::async_read(m_plainSocket, boost::asio::buffer(m_readBuff),
        boost::bind(&PlainClient::handleReadResponseHeader, this,
            boost::asio::placeholders::error));
}

//==============================================================================
//      PlainClient::startReadResponseBody()
//==============================================================================
void PlainClient::startReadResponseBody(boost::uint32_t bodySize)
{
    DEBUG_STREAM << "PlainClient::startReadResponseBody()" << endl;

    #ifdef VERBOSE_DEBUG
        INFO_STREAM << "PlainClient::startReadResponseBody() "
            << m_remoteEndpoint << " >>>> " << bodySize << " byte" << endl;
    #endif

    m_readBuff.resize(HEADER_SIZE + bodySize);

    boost::asio::mutable_buffers_1 mutableBuffer =
        boost::asio::buffer(&m_readBuff[HEADER_SIZE], bodySize);

    boost::asio::async_read(m_plainSocket, mutableBuffer,
        boost::bind(&PlainClient::handleReadResponseBody, this,
            boost::asio::placeholders::error));
}

//==============================================================================
//      PlainClient::startReadData()
//==============================================================================
void PlainClient::startReadData(FileWrapper::SP fileWrapper_sp)
{
    boost::uint64_t leftToWrite = fileWrapper_sp->getLeftToWrite();

    boost::uint64_t bufferSize = 0;

    if(leftToWrite<BUFFER_SIZE)
        bufferSize = leftToWrite;
    else
        bufferSize = BUFFER_SIZE;

    m_fileBuff.resize(bufferSize);

    m_resetConnectionTimer.expires_from_now(
        boost::posix_time::seconds(m_configuration_sp->getTimeout()));

    boost::asio::async_read(m_plainSocket, boost::asio::buffer(m_fileBuff),
        boost::bind(&PlainClient::handleReadData, this, fileWrapper_sp,
            boost::asio::placeholders::bytes_transferred,
            boost::asio::placeholders::error));
}

//==============================================================================
//      PlainClient::closeConnection()
//==============================================================================
void PlainClient::closeConnection()
{
    DEBUG_STREAM << "PlainClient::closeConnection()" << endl;

    std::stringstream infoStream;
    infoStream << "Disconnected from " << m_remoteEndpoint;

    INFO_STREAM << "PlainClient::closeConnection() " << infoStream.str() << endl;

    m_listsUpdateTimer.expires_at(boost::posix_time::pos_infin);
    m_resetConnectionTimer.expires_at(boost::posix_time::pos_infin);

    boost::system::error_code errorCode;

    m_plainSocket.shutdown(boost::asio::ip::tcp::socket::shutdown_both, errorCode);

    m_plainSocket.close(errorCode);
}

}   //namespace
