#include <SSLClient.h>
#include <DataImporter.h>

#include <boost/lexical_cast.hpp>
#include <boost/bind.hpp>
#include <stdexcept>

namespace DataImporter_ns
{

//==============================================================================
//      SSLClient::SSLClient()
//==============================================================================
SSLClient::SSLClient(DataImporter* dataImporter_p,
    Configuration::SP configuration_sp) : Client(dataImporter_p, configuration_sp),
    m_sSLContext(boost::asio::ssl::context::sslv23),
    m_sSLSocket(m_ioService, m_sSLContext)
{
    DEBUG_STREAM << "SSLClient::SSLClient()" << endl;

    boost::system::error_code errorCode;

    std::string certificateFile = m_configuration_sp->getCertificateFile();

    INFO_STREAM << "SSLClient::SSLClient() " << certificateFile << endl;

    m_sSLContext.load_verify_file(certificateFile, errorCode);

    if(errorCode)
    {
        stringstream error_stream;
        error_stream << "SSLClient::SSLClient() " << errorCode.message() << endl;
        throw std::runtime_error(error_stream.str());
    }

    m_sSLSocket.set_verify_mode(boost::asio::ssl::verify_none);
}

//==============================================================================
//      SSLClient::~SSLClient()
//==============================================================================
SSLClient::~SSLClient()
{
    DEBUG_STREAM << "SSLClient::~SSLClient()" << endl;

    closeConnection();
}

//==============================================================================
//      SSLClient::create()
//==============================================================================
Client::SP SSLClient::create(DataImporter* dataImporter_p,
    Configuration::SP configuration_sp)

{
    Client::SP c_sp(new SSLClient(dataImporter_p, configuration_sp),
        SSLClient::Deleter());

    return c_sp;
}

//==============================================================================
//      SSLClient::startConnect()
//==============================================================================
void SSLClient::startConnect(boost::asio::ip::tcp::resolver::iterator endPointIterator)
{
    DEBUG_STREAM << "SSLClient::startConnect()" << endl;

    m_resetConnectionTimer.expires_from_now(
        boost::posix_time::seconds(m_configuration_sp->getTimeout()));

    if(endPointIterator != boost::asio::ip::tcp::resolver::iterator())
    {
        std::stringstream infoStream;
        infoStream << "Connecting to " << endPointIterator->endpoint();

        INFO_STREAM << "SSLClient::startConnect() " << infoStream.str() << endl;

        writeState(Tango::RUNNING);
        writeStatus(infoStream.str());

        m_sSLSocket.lowest_layer().async_connect(endPointIterator->endpoint(),
            boost::bind(&SSLClient::handleConnect, this,
            boost::asio::placeholders::error, endPointIterator));
    }
    else
    {
        ERROR_STREAM << "SSLClient::startConnect() No more endpoint" << endl;

        writeState(Tango::ALARM);
        writeStatus("No more endpoint");
    }
}

//==============================================================================
//      SSLClient::handleConnect()
//==============================================================================
void SSLClient::handleConnect(const boost::system::error_code& errorCode,
        boost::asio::ip::tcp::resolver::iterator endPointIterator)
{
    DEBUG_STREAM << "SSLClient::handleConnect()" << endl;

    if(!errorCode)
    {
        startHandShake();
    }
    else
    {
        ERROR_STREAM << "SSLClient::handleConnect() " << errorCode.message() << endl;

        writeState(Tango::ALARM);
        writeStatus(errorCode.message());

        if(m_sSLSocket.lowest_layer().is_open())
            m_sSLSocket.lowest_layer().close();

        startConnect(++endPointIterator);
    }
}

//==============================================================================
//      SSLClient::startHandShake()
//==============================================================================
void SSLClient::startHandShake()
{
    DEBUG_STREAM << "SSLClient::startHandShake()" << endl;

    m_resetConnectionTimer.expires_from_now(
        boost::posix_time::seconds(m_configuration_sp->getTimeout()));

    m_sSLSocket.async_handshake(boost::asio::ssl::stream_base::client,
        boost::bind(&SSLClient::handleHandShake, this,
        boost::asio::placeholders::error));
}

//==============================================================================
//      SSLClient::handleHandShake()
//==============================================================================
void SSLClient::handleHandShake(const boost::system::error_code& errorCode)
{
    DEBUG_STREAM << "SSLClient::handleHandShake()" << endl;

    if(!errorCode)
    {
        m_remoteEndpoint = boost::lexical_cast<std::string>(
            m_sSLSocket.lowest_layer().remote_endpoint());

        std::stringstream infoStream;
        infoStream << "Connected to " << m_remoteEndpoint;

        INFO_STREAM << "SSLClient::handleHandShake() " << infoStream.str() << endl;

        writeState(Tango::RUNNING);
        writeStatus(infoStream.str());

        m_protocolManager_sp->setRemoteEndpoint(m_remoteEndpoint);

        startWriteRequest();
    }
    else
    {
        ERROR_STREAM << "SSLClient::handleHandShake() " << errorCode.message() << endl;

        writeState(Tango::ALARM);
        writeStatus(errorCode.message());
    }
}

//==============================================================================
//      SSLClient::startWriteRequest()
//==============================================================================
void SSLClient::startWriteRequest()
{
    DEBUG_STREAM << "SSLClient::startWriteRequest()" << endl;

    try
    {
        RequestSP request_sp = m_protocolManager_sp->createtRequest();

        boost::uint32_t bodySize = request_sp->ByteSizeLong();

        #ifdef VERBOSE_DEBUG
            INFO_STREAM << "SSLClient::startWriteRequest() "
                << m_remoteEndpoint << " <<<< " << bodySize << " byte" << endl;
        #endif

        std::vector<boost::uint8_t> writeBuff;
        writeBuff.resize(HEADER_SIZE + bodySize);

        encodeHeader(writeBuff, bodySize);

        request_sp->SerializeToArray(&writeBuff[HEADER_SIZE], bodySize);

        m_resetConnectionTimer.expires_from_now(
            boost::posix_time::seconds(m_configuration_sp->getTimeout()));

        boost::asio::async_write(m_sSLSocket, boost::asio::buffer(writeBuff),
            boost::bind(&SSLClient::handleWriteRequest, this,
            boost::asio::placeholders::error));
    }
    catch(std::exception& ec)
    {
        ERROR_STREAM << "SSLClient::startWriteRequest() " << ec.what() << endl;

        writeState(Tango::ALARM);
        writeStatus(ec.what());
    }
    catch(...)
    {
        ERROR_STREAM << "SSLClient::startWriteRequest() unknown error" << endl;

        writeState(Tango::ALARM);
        writeStatus("Unknown error");
    }
}

//==============================================================================
//      SSLClient::startReadResponseHeader()
//==============================================================================
void SSLClient::startReadResponseHeader()
{
    DEBUG_STREAM << "SSLClient::startReadResponseHeader()" << endl;

    m_readBuff.resize(HEADER_SIZE);

    m_resetConnectionTimer.expires_from_now(
        boost::posix_time::seconds(m_configuration_sp->getTimeout()));

    boost::asio::async_read(m_sSLSocket, boost::asio::buffer(m_readBuff),
        boost::bind(&SSLClient::handleReadResponseHeader, this,
            boost::asio::placeholders::error));
}

//==============================================================================
//      SSLClient::startReadResponseBody()
//==============================================================================
void SSLClient::startReadResponseBody(boost::uint32_t bodySize)
{
    DEBUG_STREAM << "SSLClient::startReadResponseBody()" << endl;

    #ifdef VERBOSE_DEBUG
        INFO_STREAM << "SSLClient::startReadResponseBody() "
            << m_remoteEndpoint << " >>>> " << bodySize << " byte" << endl;
    #endif

    m_readBuff.resize(HEADER_SIZE + bodySize);

    boost::asio::mutable_buffers_1 mutableBuffer =
        boost::asio::buffer(&m_readBuff[HEADER_SIZE], bodySize);

    boost::asio::async_read(m_sSLSocket, mutableBuffer,
        boost::bind(&SSLClient::handleReadResponseBody, this,
            boost::asio::placeholders::error));
}

//==============================================================================
//      SSLClient::startReadData()
//==============================================================================
void SSLClient::startReadData(FileWrapper::SP fileWrapper_sp)
{
    boost::uint64_t leftToWrite = fileWrapper_sp->getLeftToWrite();

    boost::uint64_t bufferSize = 0;

    if(leftToWrite<BUFFER_SIZE)
        bufferSize = leftToWrite;
    else
        bufferSize = BUFFER_SIZE;

    m_fileBuff.resize(bufferSize);

    m_resetConnectionTimer.expires_from_now(
        boost::posix_time::seconds(m_configuration_sp->getTimeout()));

    boost::asio::async_read(m_sSLSocket, boost::asio::buffer(m_fileBuff),
        boost::bind(&SSLClient::handleReadData, this, fileWrapper_sp,
            boost::asio::placeholders::bytes_transferred,
            boost::asio::placeholders::error));
}

//==============================================================================
//      SSLClient::closeConnection()
//==============================================================================
void SSLClient::closeConnection()
{
    DEBUG_STREAM << "SSLClient::closeConnection()" << endl;

    std::stringstream infoStream;
    infoStream << "Disconnected from " << m_remoteEndpoint;

    INFO_STREAM << "SSLClient::closeConnection() " << infoStream.str() << endl;

    m_listsUpdateTimer.expires_at(boost::posix_time::pos_infin);
    m_resetConnectionTimer.expires_at(boost::posix_time::pos_infin);

    boost::system::error_code errorCode;

    m_sSLSocket.lowest_layer().shutdown(
        boost::asio::ip::tcp::socket::shutdown_both, errorCode);

    m_sSLSocket.lowest_layer().close(errorCode);
}

}   //namespace
