#ifndef PROTOCOLMANAGER_H
#define	PROTOCOLMANAGER_H

#include <Response.pb.h>
#include <Request.pb.h>
#include <Configuration.h>
#include <DBManager.h>
#include <FileWrapper.h>

#include <tango.h>

#include <boost/date_time.hpp>

namespace DataImporter_ns
{

//Protocol buffer request class shared pointer
typedef boost::shared_ptr<Request> RequestSP;

//Protocol buffer response class shared pointer
typedef boost::shared_ptr<Response> ResponseSP;

class ProtocolManager : public Tango::LogAdapter
{
public:
//------------------------------------------------------------------------------
//  [Public] Shared pointer typedef
//------------------------------------------------------------------------------
    typedef boost::shared_ptr<ProtocolManager> SP;

protected:
//------------------------------------------------------------------------------
//  [Protected] Constructor destructor deleter
//------------------------------------------------------------------------------
    ProtocolManager(Tango::DeviceImpl*, Configuration::SP, DBManager::SP);

    virtual ~ProtocolManager();

    class Deleter;
    friend class Deleter;
    class Deleter
    {
        public:
            void operator()(ProtocolManager* d) { delete d; }
    };

public:
//------------------------------------------------------------------------------
//	[Public] Class creation method
//------------------------------------------------------------------------------
    static ProtocolManager::SP create(Tango::DeviceImpl*, Configuration::SP,
        DBManager::SP);

//------------------------------------------------------------------------------
//	[Public] Remote endpoint setter method
//------------------------------------------------------------------------------
    virtual void setRemoteEndpoint(std::string);

//------------------------------------------------------------------------------
//	[Public] Files lists update methods
//------------------------------------------------------------------------------
    virtual void updateNewList() throw(std::runtime_error);

    virtual void updateFailedList() throw(std::runtime_error);

//------------------------------------------------------------------------------
//	[Public] Files lists handling methods
//------------------------------------------------------------------------------
    virtual bool hasNextNewList();

    virtual bool hasNextFailedList();

    virtual bool isRecoveryTimeElapsed();

//------------------------------------------------------------------------------
//	[Public] Recovery mode getter and setter methods
//------------------------------------------------------------------------------
    virtual bool getRecoveryMode();

    virtual void setRecoveryMode(bool);

//------------------------------------------------------------------------------
//	[Public] Request response methods
//------------------------------------------------------------------------------
    RequestSP createNewListRequest()
        throw(std::logic_error, std::runtime_error);

    RequestSP createFailedListRequest()
        throw(std::logic_error, std::runtime_error);

    FileWrapper::SP processResponse(ResponseSP)
        throw(std::logic_error, std::runtime_error);

//------------------------------------------------------------------------------
//	[Public] Files status methods
//------------------------------------------------------------------------------
    virtual void setNewFileTransfered(FileWrapper::SP)
            throw(std::logic_error, std::runtime_error);

    virtual void setFailedFileTransfered(FileWrapper::SP)
            throw(std::logic_error, std::runtime_error);

    virtual void setNewFileFailed()
        throw(std::logic_error, std::runtime_error);

    virtual void setFailedFileFailed()
        throw(std::logic_error, std::runtime_error);

protected:
//------------------------------------------------------------------------------
//  [Protected] Class variables
//------------------------------------------------------------------------------
    //Tango server class pointer
    Tango::DeviceImpl* m_deviceImpl_p;

    //Configuration parameters shared pointer
    Configuration::SP m_configuration_sp;

    //Database manger shared pointer
    DBManager::SP m_dBManager_sp;

    //Address and port of remote endpoint
    std::string m_remoteEndpoint;

    //Processing file from failed list
    bool m_recoveryMode;

    //Processing file from failed list last timestamp
    boost::posix_time::ptime m_recoveryModeTime;

    //New file list shared pointer
    DBManager::NewFileRowsetSP m_newFileRowset_sp;

    //New file list iterator
    DBManager::NewFileRowset::const_iterator m_newFileRowsetIt;

    //Failed file list shared pointer
    DBManager::FailedFileRowsetSP m_failedFileRowset_sp;

    //Failed file list iterator
    DBManager::FailedFileRowset::const_iterator m_failedFileRowsetIt;
};

}   //End of namespace

#endif	/* PROTOCOLMANAGER_H */

