/*
 * File:   DBManager.h
 * Author: mdm
 *
 * Created on October 24, 2013, 2:57 PM
 */

#ifndef DBMANAGER_H
#define	DBMANAGER_H

#include <Configuration.h>

#include <tango.h>

#include <ctime>

#include <boost/tuple/tuple.hpp>
#include <boost/optional/optional.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/thread/mutex.hpp>

#include <soci/soci.h>
#include <soci/error.h>
#include <soci/row.h>
#include <soci/rowset.h>
#include <soci/boost-tuple.h>
#include <soci/boost-optional.h>
#include <soci/session.h>

namespace DataImporter_ns
{

class DBManager : public Tango::LogAdapter
{
public:
//------------------------------------------------------------------------------
//  [Public] Shared pointer typedef
//------------------------------------------------------------------------------
    typedef boost::shared_ptr<DBManager> SP;

protected:
//------------------------------------------------------------------------------
//  [Protected] Constructor destructor deleter
//------------------------------------------------------------------------------
    DBManager(Tango::DeviceImpl*, Configuration::SP);

    virtual ~DBManager();

    class Deleter;
    friend class Deleter;
    class Deleter
    {
        public:
            void operator()(DBManager* d) { delete d; }
    };

public:
//------------------------------------------------------------------------------
//  [Public] Class creation method
//------------------------------------------------------------------------------
    static DBManager::SP create(Tango::DeviceImpl*, Configuration::SP);

//------------------------------------------------------------------------------
//  [Public] Connections handling methods
//------------------------------------------------------------------------------
    virtual void connectAll() /*throw(soci::soci_error)*/;

    virtual void disconnectAll();

//------------------------------------------------------------------------------
//  [Public] Transaction retriever methods
//------------------------------------------------------------------------------
    typedef boost::shared_ptr<soci::transaction> TransactionSP;

    TransactionSP getMainTransaction();

    TransactionSP getAuxTransaction();

//------------------------------------------------------------------------------
//  [Public] Timestamp methods
//------------------------------------------------------------------------------
    virtual boost::posix_time::ptime retrieveLastTimestamp()
        /*throw(soci::soci_error)*/;

    virtual void persistLastTimestamp(boost::posix_time::ptime)
        /*throw(soci::soci_error)*/;

//------------------------------------------------------------------------------
//  [Public] File row set definition
//------------------------------------------------------------------------------
    typedef boost::tuple< boost::optional<std::string>,
        boost::optional<std::string>, boost::optional<int>,
        boost::optional<std::string>, boost::optional<std::tm> > FileRow;

    typedef soci::rowset< FileRow > FileRowset;

    typedef boost::shared_ptr< FileRowset > FileRowsetSP;

//------------------------------------------------------------------------------
//  [Public] New file method
//------------------------------------------------------------------------------
    virtual FileRowsetSP retrieveNewFiles(boost::posix_time::ptime)
        /*throw(soci::soci_error)*/;

    virtual void updateNewFilePath(std::string, std::string, int, std::string)
        /*throw(soci::soci_error)*/;

//------------------------------------------------------------------------------
//  [Public] Failed file methods
//------------------------------------------------------------------------------
    virtual void addFailedFile(int, std::string)
        /*throw(soci::soci_error)*/;

    virtual void removeFailedFile(int, std::string)
        /*throw(soci::soci_error)*/;

    virtual FileRowsetSP retrieveFailedFiles()
        /*throw(soci::soci_error)*/;

protected:
//------------------------------------------------------------------------------
//  [Protected] Class variables
//------------------------------------------------------------------------------
    //Device name
    std::string m_deviceName;

    //Configuration shared pointer
    Configuration::SP m_configuration_sp;

    //Metadata database connection mutex
    boost::mutex m_sessionMutex;

    //Main database connection scoped pointer
    boost::scoped_ptr<soci::session> m_mainSession_sp;

    //Auxiliary database connection scoped pointer
    boost::scoped_ptr<soci::session> m_auxSession_sp;
};

}   //End of namespace

#endif	/* DBMANAGER_H */
