package it.inaf.ia2.gms.persistence;

import it.inaf.ia2.gms.DataSourceConfig;
import it.inaf.ia2.gms.model.GroupBreadcrumb;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import javax.sql.DataSource;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.internal.util.collections.Sets;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringRunner;

@RunWith(SpringRunner.class)
@ContextConfiguration(classes = DataSourceConfig.class)
public class GroupsDAOTest {

    @Autowired
    private DataSource dataSource;

    private GroupsDAO dao;

    @Before
    public void setUp() {
        dao = new GroupsDAO(dataSource);
    }

    @Test
    //@Sql("/sql/init.sql")
    public void testAll() {

        // Create groups
        GroupEntity root = new GroupEntity();
        root.setId("ROOT");
        root.setName("ROOT");
        root.setPath("");
        dao.createGroup(root);

        GroupEntity lbt = new GroupEntity();
        lbt.setId(getNewGroupId());
        lbt.setName("LBT");
        lbt.setPath(lbt.getId());
        dao.createGroup(lbt);

        GroupEntity tng = new GroupEntity();
        tng.setId(getNewGroupId());
        tng.setName("TNG");
        tng.setPath(tng.getId());
        dao.createGroup(tng);

        GroupEntity lbtInaf = new GroupEntity();
        lbtInaf.setId(getNewGroupId());
        lbtInaf.setName("INAF");
        lbtInaf.setPath(lbt.getId() + "." + lbtInaf.getId());
        dao.createGroup(lbtInaf);

        // Check count
        assertEquals(4, dao.count());

        // Find by id
        Optional<GroupEntity> group = dao.findGroupById(lbtInaf.getId());
        assertTrue(group.isPresent());
        assertEquals(lbtInaf, group.get());

        group = dao.findGroupByPath(lbtInaf.getPath());
        assertTrue(group.isPresent());
        assertEquals(lbtInaf, group.get());

        // Find unexisting group
        group = dao.findGroupById("not-found");
        assertFalse(group.isPresent());

        // Sub list
        List<GroupEntity> groups = dao.listSubGroups("");
        assertEquals(2, groups.size());
        assertEquals("LBT", groups.get(0).getName());
        assertEquals("TNG", groups.get(1).getName());

        groups = dao.listSubGroups(lbt.getId());
        assertEquals(1, groups.size());
        assertEquals("INAF", groups.get(0).getName());

        // Group by parent and name
        Optional<GroupEntity> optGroup = dao.findGroupByParentAndName(root.getPath(), lbt.getName());
        assertTrue(optGroup.isPresent());
        assertEquals(lbt.getId(), optGroup.get().getId());

        optGroup = dao.findGroupByParentAndName(lbt.getPath(), lbtInaf.getName());
        assertTrue(optGroup.isPresent());
        assertEquals(lbtInaf.getId(), optGroup.get().getId());

        // Children map
        Map<String, Boolean> childrenMap = dao.getHasChildrenMap(Sets.newSet(root.getId()));
        assertEquals(1, childrenMap.size());
        assertTrue(childrenMap.get(root.getId()));

        childrenMap = dao.getHasChildrenMap(Sets.newSet(lbt.getId(), tng.getId()));
        assertEquals(2, childrenMap.size());
        assertTrue(childrenMap.get(lbt.getId()));
        assertFalse(childrenMap.get(tng.getId()));

        // Rename
        String newName = "renamed";
        tng.setName(newName);
        dao.updateGroup(tng);
        tng = dao.findGroupById(tng.getId()).get();
        assertEquals(newName, tng.getName());

        // Breadcrumbs
        List<GroupBreadcrumb> breadcrumbs = dao.getBreadcrumbs(lbt.getId() + "." + lbtInaf.getId());
        assertEquals(3, breadcrumbs.size());
        assertEquals(root.getName(), breadcrumbs.get(0).getGroupName());
        assertEquals(lbt.getName(), breadcrumbs.get(1).getGroupName());
        assertEquals(lbtInaf.getName(), breadcrumbs.get(2).getGroupName());

        // Delete
        dao.deleteGroupById(lbtInaf.getId());
        groups = dao.listSubGroups(lbt.getId());
        assertTrue(groups.isEmpty());
    }

    private String getNewGroupId() {
        return UUID.randomUUID().toString().replaceAll("-", "");
    }
}
