package it.inaf.ia2.gms.controller;

import it.inaf.ia2.gms.authn.SessionData;
import it.inaf.ia2.gms.exception.UnauthorizedException;
import it.inaf.ia2.gms.model.request.AddMemberRequest;
import it.inaf.ia2.gms.model.request.MemberRequest;
import it.inaf.ia2.gms.model.response.PaginatedData;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.RapUser;
import it.inaf.ia2.gms.model.request.RemoveMemberRequest;
import it.inaf.ia2.gms.model.request.TabRequest;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.gms.service.MembersService;
import it.inaf.ia2.gms.service.PermissionsService;
import java.util.List;
import javax.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class MembersController {

    @Autowired
    private SessionData session;

    @Autowired
    private GroupsService groupsService;

    @Autowired
    private PermissionsService permissionsService;

    @Autowired
    private MembersService membersService;

    @GetMapping(value = "/members", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PaginatedData<RapUser>> getMembersTab(TabRequest request) {

        GroupEntity group = groupsService.getGroupById(request.getGroupId());
        Permission currentNodePermissions = permissionsService.getUserPermissionForGroup(group, session.getUserId());

        if (currentNodePermissions == Permission.TRAVERSE) {
            throw new UnauthorizedException("You have not the permission to view members");
        }

        List<RapUser> members = membersService.getMembers(group.getId());
        PaginatedData<RapUser> membersPanel = new PaginatedData<>(members, request.getPaginatorPage(), request.getPaginatorPageSize());

        return ResponseEntity.ok(membersPanel);
    }

    @PostMapping(value = "/member", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PaginatedData<RapUser>> addMember(@Valid @RequestBody AddMemberRequest request) {

        GroupEntity group = groupsService.getGroupById(request.getGroupId());
        Permission currentUserPermission = verifyCurrentUserCanManageMembers(group);

        membersService.addMember(request.getGroupId(), request.getUserId());
        if (currentUserPermission == Permission.MANAGE_MEMBERS) {
            // Automatically assign the VIEW_MEMBERS permission ("Add collaborator" feature)
            permissionsService.addPermission(group, request.getUserId(), Permission.VIEW_MEMBERS);
        } else if (request.getPermission() != null) {
            // Admin users can specify a permission
            permissionsService.addPermission(group, request.getUserId(), request.getPermission());
        }

        return new ResponseEntity<>(getMembersPanel(request), HttpStatus.CREATED);
    }

    @DeleteMapping(value = "/member", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PaginatedData<RapUser>> removeMember(@Valid RemoveMemberRequest request) {

        GroupEntity group = groupsService.getGroupById(request.getGroupId());
        Permission currentUserPermission = verifyCurrentUserCanManageMembers(group);

        membersService.removeMember(group.getId(), request.getUserId());

        // For users having the MANAGE_MEMBERS permission, the VIEW_MEMBERS permission
        // is automatically assigned when they add a member ("Add collaborator" feature).
        // We want to keep also the reverse behavior.
        // If the member permission is not VIEW_MEMBERS that means that it has been
        // changed by an ADMIN user, so we don't remove it.
        boolean removeCollaborator = currentUserPermission == Permission.MANAGE_MEMBERS
                && permissionsService.getUserPermissionForGroup(group, request.getUserId()) == Permission.VIEW_MEMBERS;

        // ADMIN users can choose if delete also the permission or not.
        boolean adminRemovePermission = currentUserPermission == Permission.ADMIN && request.isRemoveAlsoPermission();

        if (removeCollaborator || adminRemovePermission) {
            permissionsService.removePermission(group, request.getUserId());
        }

        return ResponseEntity.ok(getMembersPanel(request));
    }

    private Permission verifyCurrentUserCanManageMembers(GroupEntity group) {
        Permission currentNodePermission = permissionsService.getUserPermissionForGroup(group, session.getUserId());
        if (currentNodePermission != Permission.ADMIN && currentNodePermission != Permission.MANAGE_MEMBERS) {
            throw new UnauthorizedException("Missing admin or manage members permissions");
        }
        return currentNodePermission;
    }

    private PaginatedData<RapUser> getMembersPanel(MemberRequest request) {
        List<RapUser> members = membersService.getMembers(request.getGroupId());
        return new PaginatedData<>(members, request.getPaginatorPage(), request.getPaginatorPageSize());
    }
}
