package it.inaf.ia2.gms.controller;

import com.fasterxml.jackson.databind.ObjectMapper;
import it.inaf.ia2.gms.model.request.AddMemberWsRequest;
import it.inaf.ia2.gms.model.request.AddPermissionWsRequest;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.PrepareToJoinRequest;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.MembershipEntity;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.gms.service.MembersService;
import it.inaf.ia2.gms.service.PermissionsService;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import static org.hamcrest.CoreMatchers.is;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import static org.mockito.ArgumentMatchers.eq;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.mockito.junit.MockitoJUnitRunner;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.delete;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

@RunWith(MockitoJUnitRunner.class)
public class WebServiceControllerTest {

    @Mock
    private GroupsService groupsService;

    @Mock
    private MembersService membersService;

    @Mock
    private PermissionsService permissionsService;

    @InjectMocks
    private WebServiceController controller;

    private MockMvc mockMvc;

    private final ObjectMapper mapper = new ObjectMapper();

    @Before
    public void init() {
        mockMvc = MockMvcBuilders.standaloneSetup(controller).build();
    }

    @Test
    public void testCreateGroup() throws Exception {

        GroupEntity root = new GroupEntity();
        root.setId(GroupsService.ROOT);
        root.setName(GroupsService.ROOT);
        root.setPath("");

        GroupEntity lbt = new GroupEntity();
        lbt.setId("lbt_id");
        lbt.setName("LBT");
        lbt.setPath("lbt_id");

        GroupEntity inaf = getInafGroup();

        when(groupsService.getRoot()).thenReturn(root);
        when(groupsService.findGroupByParentAndName(eq(root), eq("LBT"))).thenReturn(Optional.of(lbt));
        when(groupsService.addGroup(eq(lbt), eq("INAF"))).thenReturn(inaf);

        List<String> names = Arrays.asList("LBT", "INAF");

        mockMvc.perform(post("/ws/group")
                .content(mapper.writeValueAsString(names))
                .contentType(MediaType.APPLICATION_JSON_UTF8))
                .andExpect(status().isCreated())
                .andExpect(jsonPath("$.id", is("inaf_id")))
                .andExpect(jsonPath("$.name", is("INAF")))
                .andExpect(jsonPath("$.path", is("lbt_id.inaf_id")))
                .andExpect(jsonPath("$.parentPath", is("lbt_id")));
    }

    @Test
    public void testDeleteGroupByPath() throws Exception {

        List<String> names = Arrays.asList("LBT", "INAF");

        GroupEntity inaf = getInafGroup();

        when(groupsService.findGroupByNames(names)).thenReturn(Optional.of(inaf));

        mockMvc.perform(delete("/ws/group?names=LBT&names=INAF"))
                .andExpect(status().isNoContent());

        verify(groupsService, times(1)).deleteGroup(eq(inaf));
    }

    @Test
    public void testAddMember() throws Exception {

        List<String> names = Arrays.asList("LBT", "INAF");

        GroupEntity inaf = getInafGroup();

        when(groupsService.findGroupByNames(names)).thenReturn(Optional.of(inaf));

        AddMemberWsRequest request = new AddMemberWsRequest();
        request.setNames(names);
        request.setUserId("user_id");

        MembershipEntity membership = new MembershipEntity();
        membership.setGroupId(inaf.getId());
        membership.setUserId(request.getUserId());

        when(membersService.addMember(eq(inaf.getId()), eq(request.getUserId())))
                .thenReturn(membership);

        mockMvc.perform(post("/ws/member")
                .content(mapper.writeValueAsString(request))
                .contentType(MediaType.APPLICATION_JSON_UTF8))
                .andExpect(status().isCreated())
                .andExpect(jsonPath("$.groupId", is(inaf.getId())))
                .andExpect(jsonPath("$.userId", is(request.getUserId())));
    }

    @Test
    public void testRemoveMember() throws Exception {

        List<String> names = Arrays.asList("LBT", "INAF");

        GroupEntity inaf = getInafGroup();

        when(groupsService.findGroupByNames(names)).thenReturn(Optional.of(inaf));

        mockMvc.perform(delete("/ws/member?names=LBT&names=INAF&userId=user_id"))
                .andExpect(status().isNoContent());

        verify(membersService, times(1)).removeMember(eq(inaf.getId()), eq("user_id"));
    }

    @Test
    public void testAddPermission() throws Exception {

        List<String> names = Arrays.asList("LBT", "INAF");

        AddPermissionWsRequest request = new AddPermissionWsRequest();
        request.setNames(names);
        request.setUserId("user_id");
        request.setPermission(Permission.ADMIN);

        GroupEntity inaf = getInafGroup();
        when(groupsService.findGroupByNames(names)).thenReturn(Optional.of(inaf));

        PermissionEntity permissionEntity = new PermissionEntity();
        permissionEntity.setGroupId(inaf.getId());
        permissionEntity.setUserId(request.getUserId());
        permissionEntity.setPermission(request.getPermission());
        permissionEntity.setGroupPath(inaf.getPath());

        when(permissionsService.addPermission(eq(inaf), eq(request.getUserId()),
                eq(request.getPermission()))).thenReturn(permissionEntity);

        mockMvc.perform(post("/ws/permission")
                .content(mapper.writeValueAsString(request))
                .contentType(MediaType.APPLICATION_JSON_UTF8))
                .andExpect(status().isCreated())
                .andExpect(jsonPath("$.groupId", is(inaf.getId())))
                .andExpect(jsonPath("$.userId", is(request.getUserId())))
                .andExpect(jsonPath("$.permission", is("ADMIN")));

        verify(permissionsService, times(1))
                .addPermission(eq(inaf), eq(request.getUserId()), eq(request.getPermission()));
    }

    @Test
    public void testRemovePermission() throws Exception {

        List<String> names = Arrays.asList("LBT", "INAF");
        GroupEntity inaf = getInafGroup();
        when(groupsService.findGroupByNames(names)).thenReturn(Optional.of(inaf));

        mockMvc.perform(delete("/ws/permission?names=LBT&names=INAF&userId=user_id&permission=ADMIN"))
                .andExpect(status().isNoContent());

        verify(permissionsService, times(1)).removePermission(eq(inaf), eq("user_id"));
    }

    @Test
    public void testPrepareToJoin() throws Exception {

        PrepareToJoinRequest request = new PrepareToJoinRequest();
        request.setFromUserId("from_user");
        request.setToUserId("to_user");

        mockMvc.perform(post("/ws/prepare-join")
                .content(mapper.writeValueAsString(request))
                .contentType(MediaType.APPLICATION_JSON_UTF8))
                .andExpect(status().isOk());

        verify(permissionsService, times(1)).movePermissions(request.getFromUserId(), request.getToUserId());
        verify(membersService, times(1)).moveMemberships(request.getFromUserId(), request.getToUserId());
    }

    private GroupEntity getInafGroup() {
        GroupEntity inaf = new GroupEntity();
        inaf.setId("inaf_id");
        inaf.setName("INAF");
        inaf.setPath("lbt_id.inaf_id");
        return inaf;
    }
}
