package it.inaf.ia2.gms.manager;

import it.inaf.ia2.gms.exception.UnauthorizedException;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.persistence.GroupsDAO;
import it.inaf.ia2.gms.persistence.MembershipsDAO;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.MembershipEntity;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.rap.client.RapClient;
import it.inaf.ia2.rap.data.RapUser;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class GroupStatusManager extends UserAwareComponent {

    private static final Logger LOG = LoggerFactory.getLogger(GroupStatusManager.class);

    @Autowired
    private PermissionsManager permissionsManager;

    @Autowired
    private GroupsService groupsService;

    @Autowired
    private GroupsDAO groupsDAO;

    @Autowired
    private MembershipsDAO membershipsDAO;

    @Autowired
    private RapClient rapClient;

    public List<String[]> generateStatus(String groupId) {

        GroupEntity parentGroup = groupsService.getGroupById(groupId);

        Permission groupPermission = permissionsManager.getCurrentUserPermission(parentGroup);

        if (!Permission.includes(groupPermission, Permission.VIEW_MEMBERS)) {
            throw new UnauthorizedException("VIEW_MEMBERS permission is needed for performing this action");
        }

        List<GroupEntity> groups = groupsDAO.getAllChildren(parentGroup.getPath());
        groups.add(parentGroup);

        Map<String, String> names = groupsDAO.getGroupCompleteNamesFromId(groups.stream()
                .map(g -> g.getId()).collect(Collectors.toSet()));

        List<MembershipEntity> memberships = membershipsDAO.findByGroups(groups.stream()
                .map(g -> g.getId()).collect(Collectors.toList()));

        Map<String, List<String>> membersMap = new HashMap<>();

        for (MembershipEntity membership : memberships) {

            List<String> users = membersMap.get(membership.getGroupId());
            if (users == null) {
                users = new ArrayList<>();
                membersMap.put(membership.getGroupId(), users);
            }
            users.add(membership.getUserId());
        }

        Map<String, String> usersMap = new HashMap<>();
        for (RapUser user : rapClient.getUsers(memberships.stream()
                .map(u -> u.getUserId()).collect(Collectors.toSet()))) {
            usersMap.put(user.getId(), user.getPrimaryEmailAddress());
        }

        List<String[]> rows = new ArrayList<>();

        for (int i = 0; i < groups.size(); i++) {
            GroupEntity group = groups.get(i);
            String groupName = names.get(group.getId());
            if (groupName != null) {
                List<String> users = membersMap.get(group.getId());
                if (users != null) {
                    for (String userId : users) {
                        String email = usersMap.get(userId);
                        if (email == null) {
                            LOG.warn("Unable to retrieve information about user " + userId);
                            continue;
                        }
                        String[] row = new String[]{groupName, email};
                        rows.add(row);
                    }
                }
            }
        }

        Collections.sort(rows, (r1, r2) -> (r1[0]).compareTo(r2[0]));

        return rows;
    }
}
