package it.inaf.ia2.gms.service;

import it.inaf.ia2.gms.exception.BadRequestException;
import it.inaf.ia2.gms.model.GroupNode;
import it.inaf.ia2.gms.model.response.PaginatedData;
import it.inaf.ia2.gms.model.request.PaginatedModelRequest;
import it.inaf.ia2.gms.persistence.GroupsDAO;
import it.inaf.ia2.gms.persistence.PermissionsDAO;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class GroupsTreeBuilder {

    private final GroupsDAO groupsDAO;
    private final PermissionsDAO permissionsDAO;

    @Autowired
    public GroupsTreeBuilder(GroupsDAO groupsDAO, PermissionsDAO permissionsDAO) {
        this.groupsDAO = groupsDAO;
        this.permissionsDAO = permissionsDAO;
    }

    public PaginatedData<GroupNode> listSubGroups(String parentGroupId, String userId, PaginatedModelRequest paginatedModelRequest) {

        GroupEntity parent = groupsDAO.findGroupById(parentGroupId)
                .orElseThrow(() -> new BadRequestException("Group " + parentGroupId + " not found"));

        // All the sub groups of given parent
        List<GroupEntity> allGroups = groupsDAO.getDirectSubGroups(parent.getPath());

        // Select only the groups visible to the user
        List<PermissionEntity> permissions = permissionsDAO.findUserPermissions(userId);
        List<GroupNode> userGroups = filterOnPermissions(allGroups, permissions);

        PaginatedData<GroupNode> paginatedGroups = new PaginatedData<>(userGroups,
                paginatedModelRequest.getPaginatorPage(), paginatedModelRequest.getPaginatorPageSize());

        fillHasChildrenFlags(paginatedGroups.getItems());

        return paginatedGroups;
    }

    private List<GroupNode> filterOnPermissions(List<GroupEntity> allGroups, List<PermissionEntity> permissions) {

        List<GroupNode> nodes = new ArrayList<>();

        for (GroupEntity group : allGroups) {

            PermissionUtils.getGroupPermission(group, permissions).ifPresent(permission -> {

                GroupNode node = new GroupNode();
                node.setGroupId(group.getId());
                node.setGroupName(group.getName());
                node.setPermission(permission);
                nodes.add(node);
            });
        }

        // Sort by group name
        nodes.sort((n1, n2) -> n1.getGroupName().compareTo(n2.getGroupName()));
        return nodes;
    }

    private void fillHasChildrenFlags(List<GroupNode> groups) {

        Set<String> groupIds = groups.stream().map(g -> g.getGroupId()).collect(Collectors.toSet());

        Map<String, Boolean> hasChildrenMap = groupsDAO.getHasChildrenMap(groupIds);

        for (GroupNode group : groups) {
            group.setHasChildren(hasChildrenMap.get(group.getGroupId()));
        }
    }
}
