package it.inaf.ia2.gms.controller;

import it.inaf.ia2.gms.authn.SessionData;
import it.inaf.ia2.gms.exception.UnauthorizedException;
import it.inaf.ia2.gms.model.MemberRequest;
import it.inaf.ia2.gms.model.PaginatedData;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.RapUser;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.gms.service.MembersService;
import it.inaf.ia2.gms.service.PermissionsService;
import java.util.List;
import javax.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class MembersController {

    @Autowired
    private SessionData session;

    @Autowired
    private GroupsService groupsService;

    @Autowired
    private PermissionsService permissionsService;

    @Autowired
    private MembersService membersService;

    @PostMapping(value = "/member", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PaginatedData<RapUser>> addMember(@Valid @RequestBody MemberRequest request) {

        GroupEntity group = groupsService.getGroupById(request.getGroupId());
        verifyCurrentUserCanManageMembers(group);

        membersService.addMember(request.getGroupId(), request.getUserId());

        return new ResponseEntity<>(getMembersPanel(request), HttpStatus.CREATED);
    }

    @DeleteMapping(value = "/member", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PaginatedData<RapUser>> removeMember(@Valid MemberRequest request) {

        GroupEntity group = groupsService.getGroupById(request.getGroupId());
        verifyCurrentUserCanManageMembers(group);

        membersService.removeMember(group.getId(), request.getUserId());

        return ResponseEntity.ok(getMembersPanel(request));
    }

    private void verifyCurrentUserCanManageMembers(GroupEntity group) {
        Permission currentNodePermission = permissionsService.getUserPermissionForGroup(group, session.getUserId());
        if (currentNodePermission != Permission.ADMIN && currentNodePermission != Permission.MANAGE_MEMBERS) {
            throw new UnauthorizedException("Missing admin or manage members permissions");
        }
    }

    private PaginatedData<RapUser> getMembersPanel(MemberRequest request) {
        List<RapUser> members = membersService.getMembers(request.getGroupId());
        return new PaginatedData<>(members, request.getPaginatorPage(), request.getPaginatorPageSize());
    }
}
