package it.inaf.ia2.gms.controller;

import com.fasterxml.jackson.databind.ObjectMapper;
import it.inaf.ia2.gms.authn.SessionData;
import it.inaf.ia2.gms.model.MemberRequest;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.gms.service.MembersService;
import it.inaf.ia2.gms.service.PermissionsService;
import static org.hamcrest.CoreMatchers.is;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import static org.mockito.ArgumentMatchers.eq;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.mockito.junit.MockitoJUnitRunner;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.delete;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

@RunWith(MockitoJUnitRunner.class)
public class MembersControllerTest {

    @Mock
    private SessionData session;

    @Mock
    private GroupsService groupsService;

    @Mock
    private PermissionsService permissionsService;

    @Mock
    private MembersService membersService;

    @InjectMocks
    private MembersController controller;

    private MockMvc mockMvc;

    private final ObjectMapper mapper = new ObjectMapper();

    @Before
    public void init() {
        mockMvc = MockMvcBuilders.standaloneSetup(controller).build();

        when(session.getUserId()).thenReturn("admin_id");

        GroupEntity group = new GroupEntity();
        group.setId("group_id");
        group.setName("GroupName");
        group.setPath("parent_id.group_id");

        when(groupsService.getGroupById(eq("group_id"))).thenReturn(group);
        when(permissionsService.getUserPermissionForGroup(eq(group), eq("admin_id"))).thenReturn(Permission.ADMIN);
    }

    @Test
    public void testAddMember() throws Exception {

        MemberRequest request = new MemberRequest();
        request.setGroupId("group_id");
        request.setUserId("user_id");
        request.setPaginatorPage(1);
        request.setPaginatorPageSize(10);

        mockMvc.perform(post("/member")
                .content(mapper.writeValueAsString(request))
                .contentType(MediaType.APPLICATION_JSON_UTF8))
                .andExpect(status().isCreated())
                .andExpect(jsonPath("$.currentPage", is(1)));

        verify(membersService, times(1)).addMember(eq("group_id"), eq("user_id"));
    }

    @Test
    public void testRemoveMember() throws Exception {

        mockMvc.perform(delete("/member?groupId=group_id&userId=user_id&paginatorPage=1&paginatorPageSize=10"))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.currentPage", is(1)));

        verify(membersService, times(1)).removeMember(eq("group_id"), eq("user_id"));
    }
}
