package it.inaf.ia2.gms.controller;

import com.fasterxml.jackson.databind.ObjectMapper;
import it.inaf.ia2.gms.authn.SessionData;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.PermissionRequest;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.gms.service.PermissionsService;
import static org.hamcrest.CoreMatchers.is;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import static org.mockito.ArgumentMatchers.eq;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.mockito.junit.MockitoJUnitRunner;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.delete;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

@RunWith(MockitoJUnitRunner.class)
public class PermissionsControllerTest {

    @Mock
    private SessionData session;

    @Mock
    private GroupsService groupsService;

    @Mock
    private PermissionsService permissionsService;

    @InjectMocks
    private PermissionsController controller;

    private MockMvc mockMvc;

    private final ObjectMapper mapper = new ObjectMapper();

    private GroupEntity group;

    @Before
    public void init() {
        mockMvc = MockMvcBuilders.standaloneSetup(controller).build();

        when(session.getUserId()).thenReturn("admin_id");

        group = new GroupEntity();
        group.setId("group_id");
        group.setName("GroupName");
        group.setPath("parent_id.group_id");

        when(groupsService.getGroupById(eq("group_id"))).thenReturn(group);
        when(permissionsService.getUserPermissionForGroup(eq(group), eq("admin_id"))).thenReturn(Permission.ADMIN);
    }

    @Test
    public void testAddPermission() throws Exception {

        PermissionRequest request = new PermissionRequest();
        request.setGroupId("group_id");
        request.setUserId("user_id");
        request.setPermission(Permission.ADMIN);
        request.setPaginatorPage(1);
        request.setPaginatorPageSize(10);

        mockMvc.perform(post("/permission")
                .content(mapper.writeValueAsString(request))
                .contentType(MediaType.APPLICATION_JSON_UTF8))
                .andExpect(status().isCreated())
                .andExpect(jsonPath("$.currentPage", is(1)));

        verify(permissionsService, times(1)).addPermission(eq(group), eq("user_id"), eq(Permission.ADMIN));
    }

    @Test
    public void testDeletePermission() throws Exception {

        mockMvc.perform(delete("/permission?groupId=group_id&userId=user_id&permission=ADMIN&paginatorPage=1&paginatorPageSize=10"))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.currentPage", is(1)));

        verify(permissionsService, times(1)).deletePermission(eq(group), eq("user_id"), eq(Permission.ADMIN));
    }
}
