package it.inaf.ia2.gms.controller;

import it.inaf.ia2.gms.manager.MembershipManager;
import it.inaf.ia2.gms.manager.PermissionsManager;
import it.inaf.ia2.gms.model.request.AddMemberRequest;
import it.inaf.ia2.gms.model.response.PaginatedData;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.RapUser;
import it.inaf.ia2.gms.model.request.PaginatedModelRequest;
import it.inaf.ia2.gms.model.request.RemoveMemberRequest;
import it.inaf.ia2.gms.model.request.TabRequest;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.service.GroupsService;
import java.util.Collections;
import java.util.List;
import javax.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class MembersController {

    @Autowired
    private GroupsService groupsService;

    @Autowired
    private MembershipManager membershipManager;

    @Autowired
    private PermissionsManager permissionsManager;

    @GetMapping(value = "/members", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<PaginatedData<RapUser>> getMembersTab(TabRequest request) {

        GroupEntity group = groupsService.getGroupById(request.getGroupId());

        PaginatedData<RapUser> membersPanel = getMembersPanel(group, request);

        return ResponseEntity.ok(membersPanel);
    }

    @PostMapping(value = "/member", consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<PaginatedData<RapUser>> addMember(@Valid @RequestBody AddMemberRequest request) {

        GroupEntity group = groupsService.getGroupById(request.getGroupId());

        membershipManager.addMember(group, request.getUserId());

        Permission currentUserPermission = permissionsManager.getCurrentUserPermission(group);

        if (currentUserPermission == Permission.MANAGE_MEMBERS) {
            // Automatically assign the VIEW_MEMBERS permission ("Add collaborator" feature)
            permissionsManager.addPermission(group, request.getUserId(), Permission.VIEW_MEMBERS);
        } else if (request.getPermission() != null) {
            // Admin users can specify a permission
            permissionsManager.addPermission(group, request.getUserId(), request.getPermission());
        }

        return new ResponseEntity<>(getMembersPanel(group, request), HttpStatus.CREATED);
    }

    @DeleteMapping(value = "/member", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<PaginatedData<RapUser>> removeMember(@Valid RemoveMemberRequest request) {

        GroupEntity group = groupsService.getGroupById(request.getGroupId());

        membershipManager.removeMember(group, request.getUserId());

        Permission currentUserPermission = permissionsManager.getCurrentUserPermission(group);

        // For users having the MANAGE_MEMBERS permission, the VIEW_MEMBERS permission
        // is automatically assigned when they add a member ("Add collaborator" feature).
        // We want to keep also the reverse behavior.
        // If the member permission is not VIEW_MEMBERS that means that it has been
        // changed by an ADMIN user, so we don't remove it.
        boolean removeCollaborator = currentUserPermission == Permission.MANAGE_MEMBERS
                && permissionsManager.getUserPermission(group, request.getUserId()) == Permission.VIEW_MEMBERS;

        // ADMIN users can choose if delete also the permission or not.
        boolean adminRemovePermission = currentUserPermission == Permission.ADMIN && request.isRemoveAlsoPermission();

        if (removeCollaborator || adminRemovePermission) {
            permissionsManager.removePermission(group, request.getUserId());
        }

        return ResponseEntity.ok(getMembersPanel(group, request));
    }

    private PaginatedData<RapUser> getMembersPanel(GroupEntity group, PaginatedModelRequest request) {
        List<RapUser> members = membershipManager.getMembers(group);
        Collections.sort(members, (m1, m2) -> {
            return m1.getDisplayName().compareTo(m2.getDisplayName());
        });
        return new PaginatedData<>(members, request.getPaginatorPage(), request.getPaginatorPageSize());
    }
}
