package it.inaf.ia2.gms.controller;

import it.inaf.ia2.gms.authn.SessionData;
import it.inaf.ia2.gms.exception.UnauthorizedException;
import it.inaf.ia2.gms.manager.InvitedRegistrationManager;
import it.inaf.ia2.gms.model.GroupBreadcrumb;
import it.inaf.ia2.gms.model.GroupNode;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.request.GroupsRequest;
import it.inaf.ia2.gms.model.response.GroupsTabResponse;
import it.inaf.ia2.gms.model.response.HomePageResponse;
import it.inaf.ia2.gms.model.response.PaginatedData;
import it.inaf.ia2.gms.persistence.model.InvitedRegistration;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.support.ServletUriComponentsBuilder;

@Controller
public class HomePageController {

    @Autowired
    private SessionData session;

    @Autowired
    private GroupsTabResponseBuilder groupsTabResponseBuilder;

    @Autowired
    private InvitedRegistrationManager invitedRegistrationManager;

    @ResponseBody
    @GetMapping(value = "/home", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<HomePageResponse> getMainPage(@Valid GroupsRequest request) {

        HomePageResponse response = new HomePageResponse();

        response.setUser(session.getUserName());

        try {
            GroupsTabResponse groupsTabResponse = groupsTabResponseBuilder.getGroupsTab(request);
            response.setBreadcrumbs(groupsTabResponse.getBreadcrumbs());
            response.setGroupsPanel(groupsTabResponse.getGroupsPanel());
            response.setPermission(groupsTabResponse.getPermission());
        } catch (UnauthorizedException ex) {
            if ("ROOT".equals(request.getGroupId())) {
                response.setBreadcrumbs(getRootBreadcrumbs());
                response.setGroupsPanel(getEmptyGroupsPanel(request));
                response.setPermission(Permission.TRAVERSE);
            } else {
                throw ex;
            }
        }

        return ResponseEntity.ok(response);
    }

    private List<GroupBreadcrumb> getRootBreadcrumbs() {
        List<GroupBreadcrumb> breadcrumbs = new ArrayList<>();
        GroupBreadcrumb breadcrumb = new GroupBreadcrumb();
        breadcrumb.setGroupId("ROOT");
        breadcrumb.setGroupName("ROOT");
        breadcrumbs.add(breadcrumb);
        return breadcrumbs;
    }

    private PaginatedData<GroupNode> getEmptyGroupsPanel(GroupsRequest request) {
        return new PaginatedData<>(new ArrayList<>(), 1, request.getPaginatorPageSize());
    }

    @GetMapping(value = "/", produces = MediaType.TEXT_HTML_VALUE)
    public String index(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {

        // This page MUST NOT be cached to avoid losing the login redirect
        response.setHeader("Cache-Control", "no-store, must-revalidate");
        response.setHeader("Expires", "0");

        Optional<List<InvitedRegistration>> optReg = invitedRegistrationManager.completeInvitedRegistrationIfNecessary();
        if (optReg.isPresent()) {
            request.setAttribute("invited-registrations", optReg.get());
            return "/registration-completed";
        }

        return "index.html";
    }

    @GetMapping(value = "/logout", produces = MediaType.TEXT_HTML_VALUE)
    public void logout(HttpSession httpSession, HttpServletResponse response) throws IOException {
        httpSession.invalidate();
        String baseUrl = ServletUriComponentsBuilder.fromCurrentContextPath().build().toUriString();
        response.sendRedirect(baseUrl + "/");
    }
}
