package it.inaf.ia2.gms.persistence;

import it.inaf.ia2.gms.DataSourceConfig;
import it.inaf.ia2.gms.HooksConfig;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.InvitedRegistration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.sql.DataSource;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringRunner;

@RunWith(SpringRunner.class)
@ContextConfiguration(classes = {DataSourceConfig.class, HooksConfig.class})
public class InvitedRegistrationDAOTest {

    @Autowired
    private DataSource dataSource;

    private GroupsDAO groupsDAO;

    private InvitedRegistrationDAO dao;

    @Before
    public void setUp() {
        groupsDAO = new GroupsDAO(dataSource);
        dao = new InvitedRegistrationDAO(dataSource);
    }

    @Test
    public void test() {

        GroupEntity root = new GroupEntity();
        root.setId("ROOT");
        root.setName("ROOT");
        root.setPath("");
        groupsDAO.createGroup(root);

        GroupEntity group1 = new GroupEntity();
        group1.setId("group1");
        group1.setName("Group1");
        group1.setPath(group1.getId());
        groupsDAO.createGroup(group1);

        GroupEntity group2 = new GroupEntity();
        group2.setId("group2");
        group2.setName("Group2");
        group2.setPath(group2.getId());
        groupsDAO.createGroup(group2);

        Map<String, Permission> groupsPermissions = new HashMap<>();
        groupsPermissions.put("group1", Permission.VIEW_MEMBERS);
        groupsPermissions.put("group2", Permission.MANAGE_MEMBERS);

        InvitedRegistration reg = new InvitedRegistration()
                .setId("id1")
                .setEmail("test@inaf.it")
                .setTokenHash("token_hash")
                .setGroupsPermissions(groupsPermissions);

        dao.addInvitedRegistration(reg);

        InvitedRegistration regFromToken = dao.getInvitedRegistrationFromToken("token_hash").get();
        assertEquals(reg.getId(), regFromToken.getId());
        assertEquals(reg.getEmail(), regFromToken.getEmail());
        assertNotNull(regFromToken.getCreationTime());

        InvitedRegistration regFromGroup = dao.getPendingInvitedRegistrationsForGroup("group1").get(0);
        assertEquals(reg.getId(), regFromGroup.getId());
        assertEquals(reg.getEmail(), regFromGroup.getEmail());
        assertNotNull(regFromGroup.getCreationTime());

        List<String> addresses = new ArrayList<>();
        addresses.add("test@inaf.it");
        addresses.add("test2@inaf.it");
        assertEquals(1, dao.getInvitedRegistrationsFromEmailAddresses(addresses).size());

        dao.setRegistrationDone(regFromGroup);

        assertTrue(dao.getPendingInvitedRegistrationsForGroup("group1").isEmpty());

        List<String> groupsIds = new ArrayList<>();
        groupsIds.add(group1.getId());
        groupsIds.add(group2.getId());
        dao.deleteAllGroupsInvitedRegistrations(groupsIds);

        assertTrue(dao.getInvitedRegistrationFromToken("token_hash").isEmpty());

        dao.addInvitedRegistration(reg);
        assertEquals(1, dao.getPendingInvitedRegistrationsForGroup("group1").size());
        assertEquals(1, dao.getPendingInvitedRegistrationsForGroup("group2").size());

        dao.deleteInvitedRegistrationRequest(reg.getId(), "group1");
        assertTrue(dao.getPendingInvitedRegistrationsForGroup("group1").isEmpty());
        assertEquals(1, dao.getPendingInvitedRegistrationsForGroup("group2").size());

        dao.deleteInvitedRegistrationRequest(reg.getId(), "group2");

        groupsDAO.deleteGroup(group1);
        groupsDAO.deleteGroup(group2);
    }

    @Test
    public void testDeleteAllGroupsInvitedRegistrationsEmptyInput() {
        dao.deleteAllGroupsInvitedRegistrations(new ArrayList<>());
    }

    @Test
    public void testgetInvitedRegistrationsFromEmailAddressesEmptyInput() {
        boolean exception = false;
        try {
            dao.getInvitedRegistrationsFromEmailAddresses(new ArrayList<>());
        } catch (IllegalArgumentException ex) {
            exception = true;
        }
        assertTrue(exception);
    }
}
