package it.inaf.ia2.gms.controller;

import it.inaf.ia2.gms.exception.BadRequestException;
import it.inaf.ia2.gms.model.request.AddMemberWsRequest;
import it.inaf.ia2.gms.model.request.AddPermissionWsRequest;
import it.inaf.ia2.gms.model.PrepareToJoinRequest;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.MembershipEntity;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.gms.service.MembersService;
import it.inaf.ia2.gms.service.PermissionsService;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import javax.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

@RestController
@RequestMapping("/ws")
public class WebServiceController {

    @Autowired
    private GroupsService groupsService;

    @Autowired
    private MembersService membersService;

    @Autowired
    private PermissionsService permissionsService;

    /**
     * Creates a group and its ancestors if they are missing. It doesn't fail if
     * the last group already exists.
     */
    @PostMapping(value = "/group", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<GroupEntity> createGroup(@RequestBody List<String> names) {

        GroupEntity group = groupsService.getRoot();

        for (String name : names) {
            Optional<GroupEntity> optGroup = groupsService.findGroupByParentAndName(group, name);
            if (optGroup.isPresent()) {
                group = optGroup.get();
            } else {
                group = groupsService.addGroup(group, name);
            }
        }

        return new ResponseEntity<>(group, HttpStatus.CREATED);
    }

    @DeleteMapping("/group")
    public ResponseEntity<?> deleteGroupByPath(@RequestParam("names") String[] names) {

        GroupEntity groupToDelete = getGroupByNames(Arrays.asList(names));

        groupsService.deleteGroup(groupToDelete);

        return ResponseEntity.noContent().build();
    }

    @PostMapping(value = "/member", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<MembershipEntity> addMember(@RequestBody AddMemberWsRequest request) {

        GroupEntity group = getGroupByNames(request.getNames());

        MembershipEntity membership = membersService.addMember(group.getId(), request.getUserId());

        return new ResponseEntity<>(membership, HttpStatus.CREATED);
    }

    @DeleteMapping("/member")
    public ResponseEntity<?> removeMember(@RequestParam("names") String[] names, @RequestParam("userId") String userId) {

        GroupEntity group = getGroupByNames(Arrays.asList(names));

        membersService.removeMember(group.getId(), userId);

        return ResponseEntity.noContent().build();
    }

    @PostMapping(value = "/permission", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PermissionEntity> addPermission(@Valid @RequestBody AddPermissionWsRequest request) {

        GroupEntity group = getGroupByNames(request.getNames());

        PermissionEntity newPermission = permissionsService.addPermission(group, request.getUserId(), request.getPermission());

        return new ResponseEntity<>(newPermission, HttpStatus.CREATED);
    }

    @DeleteMapping("/permission")
    public ResponseEntity<?> deletePermission(@RequestParam("names") String[] names, @RequestParam("userId") String userId) {

        GroupEntity group = getGroupByNames(Arrays.asList(names));

        permissionsService.removePermission(group, userId);

        return ResponseEntity.noContent().build();
    }

    @PostMapping(value = "/prepare-join", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<?> prepareToJoin(@Valid @RequestBody PrepareToJoinRequest request) {

        permissionsService.movePermissions(request.getFromUserId(), request.getToUserId());
        membersService.moveMemberships(request.getFromUserId(), request.getToUserId());

        return ResponseEntity.ok().build();
    }

    private GroupEntity getGroupByNames(List<String> names) {
        return groupsService.findGroupByNames(names)
                .orElseThrow(() -> new BadRequestException("Unable to find requested group"));
    }
}
