package it.inaf.ia2.gms.service;

import it.inaf.ia2.gms.exception.BadRequestException;
import it.inaf.ia2.gms.exception.UnauthorizedException;
import it.inaf.ia2.gms.model.GroupBreadcrumb;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.persistence.GroupsDAO;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import java.util.List;
import java.util.UUID;

@Service
public class GroupsService {

    public static final String ROOT = "ROOT";

    private final GroupsDAO groupsDAO;
    private final PermissionsService permissionsService;

    @Autowired
    public GroupsService(GroupsDAO groupsDAO,
            PermissionsService permissionsService) {
        this.groupsDAO = groupsDAO;
        this.permissionsService = permissionsService;
        createRootIfNecessary();
    }

    private void createRootIfNecessary() {
        if (groupsDAO.count() == 0) {
            GroupEntity root = new GroupEntity();
            root.setId(ROOT);
            root.setName(ROOT);
            root.setPath("");
            groupsDAO.createGroup(root);
        }
    }

    public GroupEntity addGroup(String parentId, String groupName, String userId) {

        GroupEntity parent = getGroupById(parentId);

        if (permissionsService.getGroupPermission(parent, userId) != Permission.ADMIN) {
            throw new UnauthorizedException("Missing admin privileges");
        }

        if (groupsDAO.listSubGroups(parent.getPath()).stream()
                .anyMatch(g -> g.getName().equals(groupName))) {
            throw new BadRequestException("There is already a group named " + groupName);
        }

        String newGroupId = UUID.randomUUID().toString().replaceAll("-", "");

        String path = parent.getPath();
        if (!path.isEmpty()) {
            path += ".";
        }
        path += newGroupId;

        GroupEntity group = new GroupEntity();
        group.setId(newGroupId);
        group.setName(groupName);
        group.setPath(path);

        groupsDAO.createGroup(group);

        return group;
    }

    public GroupEntity renameGroup(String groupId, String newGroupName, String userId) {

        GroupEntity group = getGroupById(groupId);

        if (permissionsService.getGroupPermission(group, userId) != Permission.ADMIN) {
            throw new UnauthorizedException("Missing admin privileges");
        }

        if (groupsDAO.listSubGroups(group.getPath()).stream()
                .anyMatch(g -> g.getName().equals(newGroupName))) {
            throw new BadRequestException("There is already a group named " + newGroupName);
        }

        group.setName(newGroupName);

        return groupsDAO.updateGroup(group);
    }

    public GroupEntity deleteGroup(String groupId, String userId) {

        GroupEntity group = getGroupById(groupId);

        if (permissionsService.getGroupPermission(group, userId) != Permission.ADMIN) {
            throw new UnauthorizedException("Missing admin privileges");
        }

        if (ROOT.equals(group.getId())) {
            throw new UnauthorizedException("It is not possible to remove the ROOT");
        }

        String parentPath = group.getParentPath();
        GroupEntity parent = groupsDAO.findGroupByPath(parentPath)
                .orElseThrow(() -> new BadRequestException("No group found at path " + parentPath));

        groupsDAO.deleteGroupById(groupId);

        return parent;
    }

    public GroupEntity getGroupById(String groupId) {
        return groupsDAO.findGroupById(groupId)
                .orElseThrow(() -> new BadRequestException("Group " + groupId + " not found"));
    }

    public GroupEntity getGroupByPath(String path) {
        return groupsDAO.findGroupByPath(path)
                .orElseThrow(() -> new BadRequestException("Group not found at path " + path));
    }

    public List<GroupBreadcrumb> getBreadcrumbs(String path) {
        return groupsDAO.getBreadcrumbs(path);
    }
}
