package it.inaf.ia2.gms.controller;

import it.inaf.ia2.gms.authn.SessionData;
import it.inaf.ia2.gms.exception.UnauthorizedException;
import it.inaf.ia2.gms.model.PermissionRequest;
import it.inaf.ia2.gms.model.PaginatedData;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.UserPermission;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.gms.service.PermissionsService;
import java.util.List;
import javax.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class PermissionsController {

    @Autowired
    private SessionData session;

    @Autowired
    private GroupsService groupsService;

    @Autowired
    private PermissionsService permissionsService;

    @PostMapping(value = "/permission", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PaginatedData<UserPermission>> addPermission(@Valid @RequestBody PermissionRequest request) {

        GroupEntity group = groupsService.getGroupById(request.getGroupId());
        verifyAdminSession(group);

        permissionsService.addPermission(group, request.getUserId(), request.getPermission());

        return new ResponseEntity<>(getPermissionsPanel(group, request), HttpStatus.CREATED);
    }

    @DeleteMapping(value = "/permission", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PaginatedData<UserPermission>> deletePermission(@Valid PermissionRequest request) {

        GroupEntity group = groupsService.getGroupById(request.getGroupId());
        verifyAdminSession(group);

        permissionsService.deletePermission(group, request.getUserId(), request.getPermission());

        return ResponseEntity.ok(getPermissionsPanel(group, request));
    }

    private void verifyAdminSession(GroupEntity group) {
        Permission currentNodePermissions = permissionsService.getUserPermissionForGroup(group, session.getUserId());
        if (currentNodePermissions != Permission.ADMIN) {
            throw new UnauthorizedException("Only admin users can handle permissions");
        }
    }

    private PaginatedData<UserPermission> getPermissionsPanel(GroupEntity group, PermissionRequest request) {
        List<UserPermission> permissions = permissionsService.getAllPermissions(group);
        return new PaginatedData<>(permissions, request.getPaginatorPage(), request.getPaginatorPageSize());
    }
}
