package it.inaf.ia2.gms.controller;

import it.inaf.ia2.gms.authn.SessionData;
import it.inaf.ia2.gms.exception.UnauthorizedException;
import it.inaf.ia2.gms.model.request.AddGroupRequest;
import it.inaf.ia2.gms.model.GroupNode;
import it.inaf.ia2.gms.model.response.PaginatedData;
import it.inaf.ia2.gms.model.request.PaginatedModelRequest;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.request.DeleteGroupRequest;
import it.inaf.ia2.gms.model.request.GroupsRequest;
import it.inaf.ia2.gms.model.request.RenameGroupRequest;
import it.inaf.ia2.gms.model.request.SearchFilterRequest;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.gms.service.GroupsTreeBuilder;
import it.inaf.ia2.gms.service.PermissionsService;
import javax.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class GroupsController {

    @Autowired
    private SessionData session;

    @Autowired
    private PermissionsService permissionsService;

    @Autowired
    private GroupsService groupsService;

    @Autowired
    private GroupsTreeBuilder groupsTreeBuilder;

    @Autowired
    private GroupsTabResponseBuilder groupsTabResponseBuilder;

    @GetMapping(value = "/groups", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<?> getGroupsTab(@Valid GroupsRequest request) {
        if (request.isOnlyPanel()) {
            // Only groupsPanel
            GroupEntity group = groupsService.getGroupById(request.getGroupId());
            return ResponseEntity.ok(getGroupsPanel(group, request));
        } else {
            // Complete GroupsTabResponse
            return ResponseEntity.ok(groupsTabResponseBuilder.getGroupsTab(request));
        }
    }

    @PostMapping(value = "/group", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PaginatedData<GroupNode>> createGroup(@Valid @RequestBody AddGroupRequest request) {

        GroupEntity parent = groupsService.getGroupById(request.getParentGroupId());

        if (permissionsService.getUserPermissionForGroup(parent, session.getUserId()) != Permission.ADMIN) {
            throw new UnauthorizedException("Missing admin permission");
        }

        groupsService.addGroup(parent, request.getNewGroupName());

        PaginatedData<GroupNode> groupsPanel = getGroupsPanel(parent, request);

        return new ResponseEntity<>(groupsPanel, HttpStatus.CREATED);
    }

    @PutMapping(value = "/group/{groupId}", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PaginatedData<GroupNode>> renameGroup(@PathVariable("groupId") String groupId, @Valid @RequestBody RenameGroupRequest request) {

        GroupEntity group = groupsService.getGroupById(groupId);

        if (permissionsService.getUserPermissionForGroup(group, session.getUserId()) != Permission.ADMIN) {
            throw new UnauthorizedException("Missing admin permission");
        }

        GroupEntity renamedGroup = groupsService.renameGroup(group, request.getNewGroupName());

        GroupEntity parent = groupsService.getGroupByPath(renamedGroup.getParentPath());

        PaginatedData<GroupNode> groupsPanel = getGroupsPanel(parent, request);

        return ResponseEntity.ok(groupsPanel);
    }

    @DeleteMapping(value = "/group/{groupId}", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<?> deleteGroup(@PathVariable("groupId") String groupId, DeleteGroupRequest request) {

        GroupEntity group = groupsService.getGroupById(groupId);

        if (permissionsService.getUserPermissionForGroup(group, session.getUserId()) != Permission.ADMIN) {
            throw new UnauthorizedException("Missing admin permission");
        }

        GroupEntity parent = groupsService.deleteGroup(group);

        PaginatedData<GroupNode> groupsPanel = getGroupsPanel(parent, request);

        return ResponseEntity.ok(groupsPanel);
    }

    private <T extends PaginatedModelRequest & SearchFilterRequest> PaginatedData<GroupNode> getGroupsPanel(GroupEntity parentGroup, T request) {
        return groupsTreeBuilder.listSubGroups(parentGroup, request, session.getUserId());
    }
}
