package it.inaf.ia2.gms.service;

import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.response.PaginatedData;
import it.inaf.ia2.gms.model.response.SearchResponseItem;
import it.inaf.ia2.gms.model.response.SearchResponseType;
import it.inaf.ia2.gms.model.response.UserSearchResponse;
import it.inaf.ia2.gms.persistence.GroupsDAO;
import it.inaf.ia2.gms.persistence.MembershipsDAO;
import it.inaf.ia2.gms.persistence.PermissionsDAO;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import it.inaf.ia2.gms.rap.RapClient;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class SearchService {

    @Autowired
    private RapClient rapClient;

    @Autowired
    private GroupsService groupsService;

    @Autowired
    private GroupsDAO groupsDAO;

    @Autowired
    private PermissionsDAO permissionsDAO;

    @Autowired
    private MembershipsDAO membershipsDAO;

    /**
     * Generic search (both groups and users).
     */
    public PaginatedData<SearchResponseItem> search(String query, String userId, int page, int pageSize) {

        List<SearchResponseItem> items = searchUsers(query);
        items.addAll(searchGroups(query, userId));

        // sort by label
        items.sort((i1, i2) -> i1.getLabel().compareTo(i2.getLabel()));

        return new PaginatedData<>(items, page, pageSize);
    }

    private List<SearchResponseItem> searchUsers(String query) {
        return rapClient.searchUsers(query).stream()
                .map(u -> {
                    SearchResponseItem item = new SearchResponseItem();
                    item.setType(SearchResponseType.USER);
                    item.setId(u.getId());
                    item.setLabel(u.getDisplayName());
                    return item;
                })
                .collect(Collectors.toList());
    }

    private List<SearchResponseItem> searchGroups(String query, String userId) {

        List<GroupEntity> allGroups = groupsDAO.searchGroups(query);

        // Select only the groups visible to the user
        List<PermissionEntity> permissions = permissionsDAO.findUserPermissions(userId);

        List<SearchResponseItem> items = new ArrayList<>();
        for (GroupEntity group : allGroups) {

            PermissionUtils.getGroupPermission(group, permissions).ifPresent(permission -> {
                SearchResponseItem item = new SearchResponseItem();
                item.setType(SearchResponseType.GROUP);
                item.setId(group.getId());
                item.setLabel(group.getName());
                items.add(item);
            });
        }

        return items;
    }

    /**
     * Retrieve additional data about an user displayed into the generic search.
     *
     * @param actorUserId the user who performed the search
     * @param targetUserId the user displayed into the search results
     */
    public UserSearchResponse getUserSearchResult(String actorUserId, String targetUserId) {

        List<GroupEntity> allGroups = membershipsDAO.getUserMemberships(targetUserId);

        // Select only the groups visible to the user
        List<PermissionEntity> permissions = permissionsDAO.findUserPermissions(actorUserId);

        List<GroupEntity> visibleGroups = new ArrayList<>();
        for (GroupEntity group : allGroups) {

            PermissionUtils.getGroupPermission(group, permissions).ifPresent(permission -> {
                visibleGroups.add(group);
            });
        }

        UserSearchResponse response = new UserSearchResponse();
        response.setGroups(visibleGroups);

        // Super-admin user is able to see also other user permissions
        PermissionUtils.getGroupPermission(groupsService.getRoot(), permissions).ifPresent(permission -> {
            if (permission.equals(Permission.ADMIN)) {
                List<PermissionEntity> targetUserPermissions = permissionsDAO.findUserPermissions(targetUserId);
                response.setPermissions(targetUserPermissions);
            }
        });

        return response;
    }
}
