package it.inaf.ia2.gms.service;

import it.inaf.ia2.gms.model.Identity;
import it.inaf.ia2.gms.model.IdentityType;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.RapUser;
import it.inaf.ia2.gms.model.response.PaginatedData;
import it.inaf.ia2.gms.model.response.SearchResponseItem;
import it.inaf.ia2.gms.model.response.SearchResponseType;
import it.inaf.ia2.gms.model.response.UserSearchResponse;
import it.inaf.ia2.gms.persistence.GroupsDAO;
import it.inaf.ia2.gms.persistence.MembershipsDAO;
import it.inaf.ia2.gms.persistence.PermissionsDAO;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import it.inaf.ia2.gms.rap.RapClient;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import org.junit.Test;
import org.junit.runner.RunWith;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import static org.mockito.Mockito.when;
import org.mockito.junit.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class SearchServiceTest {

    @Mock
    private RapClient rapClient;

    @Mock
    private GroupsService groupsService;

    @Mock
    private GroupsDAO groupsDAO;

    @Mock
    private PermissionsDAO permissionsDAO;

    @Mock
    private MembershipsDAO membershipsDAO;

    @InjectMocks
    private SearchService searchService;

    @Test
    public void testGenericSearch() {

        RapUser user = new RapUser();
        user.setId("user_id");
        Identity identity = new Identity();
        identity.setPrimary(true);
        identity.setType(IdentityType.EDU_GAIN);
        identity.setEmail("user@inaf.it");
        user.setIdentities(Collections.singletonList(identity));

        when(rapClient.searchUsers(any())).thenReturn(Collections.singletonList(user));

        GroupEntity group1 = new GroupEntity();
        group1.setId("group1_id");
        group1.setName("Group 1");
        group1.setPath("group1_id");
        GroupEntity group2 = new GroupEntity();
        group2.setId("group2_id");
        group2.setName("Group 2");
        group2.setPath("group2_id");
        List<GroupEntity> allGroups = new ArrayList<>();
        allGroups.add(group1);
        allGroups.add(group2);

        when(groupsDAO.searchGroups(any())).thenReturn(allGroups);

        PermissionEntity permission = new PermissionEntity();
        permission.setGroupId("group1_id");
        permission.setUserId("manager_id");
        permission.setGroupPath("group1_id");
        permission.setPermission(Permission.MANAGE_MEMBERS);

        when(permissionsDAO.findUserPermissions(any())).thenReturn(
                Collections.singletonList(permission));

        PaginatedData<SearchResponseItem> response = searchService.search("foo", "manager_id", 1, 10);

        assertEquals(2, response.getTotalItems());

        SearchResponseItem item0 = response.getItems().get(0);
        assertEquals(SearchResponseType.GROUP, item0.getType());
        assertEquals("group1_id", item0.getId());
        assertEquals("Group 1", item0.getLabel());

        SearchResponseItem item1 = response.getItems().get(1);
        assertEquals(SearchResponseType.USER, item1.getType());
        assertEquals("user_id", item1.getId());
        assertEquals("user@inaf.it (EduGAIN)", item1.getLabel());
    }

    @Test
    public void testGetUserSearchResult() {

        GroupEntity group1 = new GroupEntity();
        group1.setId("group1_id");
        group1.setName("Group 1");
        group1.setPath("group1_id");

        when(membershipsDAO.getUserMemberships(any()))
                .thenReturn(Collections.singletonList(group1));

        PermissionEntity adminPermission = new PermissionEntity();
        adminPermission.setGroupId("ROOT");
        adminPermission.setUserId("admin_id");
        adminPermission.setGroupPath("");
        adminPermission.setPermission(Permission.ADMIN);

        when(permissionsDAO.findUserPermissions(eq("admin_id")))
                .thenReturn(Collections.singletonList(adminPermission));

        GroupEntity root = new GroupEntity();
        root.setId("ROOT");
        root.setName("Root");
        root.setPath("");
        when(groupsService.getRoot()).thenReturn(root);

        UserSearchResponse response = searchService.getUserSearchResult("admin_id", "target_id");
        assertEquals(1, response.getGroups().size());
        assertNotNull(response.getPermissions());
    }
}
