package it.inaf.ia2.gms.controller;

import com.fasterxml.jackson.databind.ObjectMapper;
import it.inaf.ia2.gms.manager.GroupsManager;
import it.inaf.ia2.gms.manager.PermissionsManager;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.request.AddPermissionRequest;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import static org.hamcrest.CoreMatchers.is;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import static org.mockito.ArgumentMatchers.eq;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.mockito.junit.MockitoJUnitRunner;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.delete;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

@RunWith(MockitoJUnitRunner.class)
public class PermissionsControllerTest {

    @Mock
    private GroupsManager groupsManager;

    @Mock
    private PermissionsManager permissionsManager;

    @InjectMocks
    private PermissionsController controller;

    private MockMvc mockMvc;

    private final ObjectMapper mapper = new ObjectMapper();

    private GroupEntity group;

    @Before
    public void init() {
        mockMvc = MockMvcBuilders.standaloneSetup(controller).build();

        group = new GroupEntity();
        group.setId("group_id");
        group.setName("GroupName");
        group.setPath("parent_id.group_id");

        when(groupsManager.getGroupById(eq("group_id"))).thenReturn(group);
    }

    @Test
    public void testGetPermission() throws Exception {

        when(permissionsManager.getUserPermission(eq(group), eq("user_id"))).thenReturn(Permission.MANAGE_MEMBERS);

        mockMvc.perform(get("/permission?groupId=group_id&userId=user_id")
                .contentType(MediaType.APPLICATION_JSON_UTF8))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.permission", is("MANAGE_MEMBERS")));
    }

    @Test
    public void testGetInexistentPermission() throws Exception {

        mockMvc.perform(get("/permission?groupId=group_id&userId=user_id")
                .contentType(MediaType.APPLICATION_JSON_UTF8))
                .andExpect(status().isNotFound());
    }

    @Test
    public void testAddPermission() throws Exception {

        AddPermissionRequest request = new AddPermissionRequest();
        request.setGroupId("group_id");
        request.setUserId("user_id");
        request.setPermission(Permission.ADMIN);
        request.setPaginatorPage(1);
        request.setPaginatorPageSize(10);

        mockMvc.perform(post("/permission")
                .content(mapper.writeValueAsString(request))
                .contentType(MediaType.APPLICATION_JSON_UTF8))
                .andExpect(status().isCreated())
                .andExpect(jsonPath("$.currentPage", is(1)));

        verify(permissionsManager, times(1)).addPermission(eq(group), eq("user_id"), eq(Permission.ADMIN));
    }

    @Test
    public void testRemovePermission() throws Exception {

        mockMvc.perform(delete("/permission?groupId=group_id&userId=user_id&paginatorPage=1&paginatorPageSize=10"))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.currentPage", is(1)));

        verify(permissionsManager, times(1)).removePermission(eq(group), eq("user_id"));
    }
}
