package it.inaf.ia2.gms.persistence;

import it.inaf.ia2.gms.DataSourceConfig;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.MembershipEntity;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import java.util.HashSet;
import java.util.Set;
import javax.sql.DataSource;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringRunner;

@RunWith(SpringRunner.class)
@ContextConfiguration(classes = DataSourceConfig.class)
public class JoinDAOTest {

    private static final String USER_1 = "user-1";
    private static final String USER_2 = "user-2";

    @Autowired
    private DataSource dataSource;

    private GroupsDAO groupsDAO;
    private MembershipsDAO membershipsDAO;
    private PermissionsDAO permissionsDAO;
    private JoinDAO joinDAO;

    @Before
    public void setUp() {
        groupsDAO = new GroupsDAO(dataSource);
        membershipsDAO = new MembershipsDAO(dataSource);
        permissionsDAO = new PermissionsDAO(dataSource);
        joinDAO = new JoinDAO(dataSource);
    }

    @Test
    public void testJoin() {

        groupsDAO.createGroup(groupEntity("A"));
        groupsDAO.createGroup(groupEntity("B"));

        permissionsDAO.createOrUpdatePermission(permissionEntity(USER_1, "A", Permission.VIEW_MEMBERS));
        permissionsDAO.createOrUpdatePermission(permissionEntity(USER_1, "B", Permission.VIEW_MEMBERS));
        permissionsDAO.createOrUpdatePermission(permissionEntity(USER_2, "A", Permission.ADMIN));

        membershipsDAO.addMember(membershipEntity(USER_2, "A"));

        Set<MembershipEntity> membershipsToAdd = new HashSet<>();
        membershipsToAdd.add(membershipEntity(USER_1, "A"));

        Set<PermissionEntity> permissionsToAdd = new HashSet<>();
        permissionsToAdd.add(permissionEntity(USER_1, "A", Permission.ADMIN));

        assertEquals(1, membershipsDAO.getUserMemberships(USER_2).size());
        assertEquals(2, permissionsDAO.findUserPermissions(USER_1).size());
        assertEquals(1, permissionsDAO.findUserPermissions(USER_2).size());

        joinDAO.join(membershipsToAdd, permissionsToAdd, USER_2);

        assertTrue(membershipsDAO.getUserMemberships(USER_2).isEmpty());
        assertTrue(permissionsDAO.findUserPermissions(USER_2).isEmpty());
        assertEquals(2, permissionsDAO.findUserPermissions(USER_1).size());
        assertTrue(permissionsDAO.findUserPermissions(USER_1)
                .contains(permissionEntity(USER_1, "A", Permission.ADMIN)));
        assertEquals(1, membershipsDAO.getUserMemberships(USER_1).size());
    }

    private GroupEntity groupEntity(String groupId) {
        GroupEntity groupEntity = new GroupEntity();
        groupEntity.setId(groupId);
        groupEntity.setName(groupId);
        groupEntity.setPath(groupId);
        groupEntity.setLeaf(false);
        return groupEntity;
    }

    private MembershipEntity membershipEntity(String userId, String groupId) {
        MembershipEntity membershipEntity = new MembershipEntity();
        membershipEntity.setUserId(userId);
        membershipEntity.setGroupId(groupId);
        return membershipEntity;
    }

    private PermissionEntity permissionEntity(String userId, String groupId, Permission permission) {
        PermissionEntity permissionEntity = new PermissionEntity();
        permissionEntity.setGroupId(groupId);
        permissionEntity.setUserId(userId);
        permissionEntity.setPermission(permission);
        permissionEntity.setGroupPath(groupId);
        return permissionEntity;
    }
}
