package it.inaf.ia2.gms.persistence;

import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import java.sql.PreparedStatement;
import java.sql.Types;
import java.util.ArrayList;
import java.util.List;
import javax.sql.DataSource;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Component;

@Component
public class PermissionsDAO {

    private final JdbcTemplate jdbcTemplate;

    @Autowired
    public PermissionsDAO(DataSource dataSource) {
        jdbcTemplate = new JdbcTemplate(dataSource);
    }

    public PermissionEntity createPermission(PermissionEntity userPermission) {

        String sql = "INSERT INTO gms_permission(group_id, user_id, permission, group_path) VALUES(?, ?, ?, ?)";

        jdbcTemplate.update(conn -> {
            PreparedStatement ps = conn.prepareStatement(sql);
            ps.setString(1, userPermission.getGroupId());
            ps.setString(2, userPermission.getUserId());
            ps.setObject(3, userPermission.getPermission().toString(), Types.OTHER);
            ps.setObject(4, userPermission.getGroupPath(), Types.OTHER);
            return ps;
        });

        return userPermission;
    }

    public List<PermissionEntity> findUserPermissions(String userId) {

        String sql = "SELECT group_id, permission, group_path FROM gms_permission WHERE user_id = ?";

        return jdbcTemplate.query(conn -> {
            PreparedStatement ps = conn.prepareStatement(sql);
            ps.setString(1, userId);
            return ps;
        }, resultSet -> {
            List<PermissionEntity> permissions = new ArrayList<>();
            while (resultSet.next()) {
                PermissionEntity permission = new PermissionEntity();
                permission.setGroupId(resultSet.getString("group_id"));
                permission.setUserId(userId);
                permission.setPermission(Permission.valueOf(resultSet.getString("permission")));
                permission.setGroupPath(resultSet.getString("group_path"));
                permissions.add(permission);
            }
            return permissions;
        });
    }

    public List<PermissionEntity> findUserPermissions(String userId, String path) {

        String sql = "SELECT group_id, permission, group_path FROM gms_permission WHERE user_id = ?\n"
                + "AND (group_path <@ ? OR group_path @> ?)";

        return jdbcTemplate.query(conn -> {
            PreparedStatement ps = conn.prepareStatement(sql);
            ps.setString(1, userId);
            ps.setObject(2, path, Types.OTHER);
            ps.setObject(3, path, Types.OTHER);
            return ps;
        }, resultSet -> {
            List<PermissionEntity> permissions = new ArrayList<>();
            while (resultSet.next()) {
                PermissionEntity permission = new PermissionEntity();
                permission.setGroupId(resultSet.getString("group_id"));
                permission.setUserId(userId);
                permission.setPermission(Permission.valueOf(resultSet.getString("permission")));
                permission.setGroupPath(resultSet.getString("group_path"));
                permissions.add(permission);
            }
            return permissions;
        });
    }

    public List<PermissionEntity> getGroupsPermissions(String groupId) {

        String sql = "SELECT user_id, permission, group_path FROM gms_permission WHERE group_id = ?";

        return jdbcTemplate.query(conn -> {
            PreparedStatement ps = conn.prepareStatement(sql);
            ps.setString(1, groupId);
            return ps;
        }, resultSet -> {
            List<PermissionEntity> permissions = new ArrayList<>();
            while (resultSet.next()) {
                PermissionEntity permission = new PermissionEntity();
                permission.setGroupId(groupId);
                permission.setUserId(resultSet.getString("user_id"));
                permission.setPermission(Permission.valueOf(resultSet.getString("permission")));
                permission.setGroupPath(resultSet.getString("group_path"));
                permissions.add(permission);
            }
            return permissions;
        });
    }

    public void deletePermission(String groupId, String userId, Permission permission) {

        String sql = "DELETE FROM gms_permission WHERE group_id = ? AND user_id = ? AND permission = ?";

        jdbcTemplate.update(conn -> {
            PreparedStatement ps = conn.prepareStatement(sql);
            ps.setString(1, groupId);
            ps.setString(2, userId);
            ps.setObject(3, permission.toString(), Types.OTHER);
            return ps;
        });
    }
}
