package it.inaf.ia2.gms.persistence;

import it.inaf.ia2.gms.DataSourceConfig;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import javax.sql.DataSource;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringRunner;

@RunWith(SpringRunner.class)
@ContextConfiguration(classes = DataSourceConfig.class)
public class PermissionsDAOTest {

    private static final String USER_ID = "user_id";

    @Autowired
    private DataSource dataSource;

    private GroupsDAO groupsDAO;
    private PermissionsDAO permissionsDAO;

    @Before
    public void setUp() {
        groupsDAO = new GroupsDAO(dataSource);
        permissionsDAO = new PermissionsDAO(dataSource);
    }

    @Test
    public void testAll() {

        GroupEntity group = groupEntity("A");

        PermissionEntity permission = permissionEntity(group, USER_ID, Permission.ADMIN);

        permissionsDAO.createOrUpdatePermission(permission);

        List<PermissionEntity> permissions = permissionsDAO.findUserPermissions(USER_ID);

        assertEquals(1, permissions.size());
        assertEquals(Permission.ADMIN, permissions.get(0).getPermission());
        assertEquals(USER_ID, permissions.get(0).getUserId());
        assertEquals("admin1", permissions.get(0).getSetBy());
        assertNotNull(permissions.get(0).getUpdateTime());

        permissions = permissionsDAO.findUserPermissions(USER_ID, group.getPath());
        assertEquals(1, permissions.size());

        permissions = permissionsDAO.getGroupsPermissions(group.getId());
        assertEquals(1, permissions.size());

        // test upsert
        permission.setPermission(Permission.MANAGE_MEMBERS);
        permission.setSetBy("admin2");
        permissionsDAO.createOrUpdatePermission(permission);

        permissions = permissionsDAO.findUserPermissions(USER_ID);
        assertEquals(1, permissions.size());
        assertEquals(Permission.MANAGE_MEMBERS, permissions.get(0).getPermission());
        assertEquals("admin2", permissions.get(0).getSetBy());
        assertNotNull(permissions.get(0).getUpdateTime());

        // test update
        permission.setPermission(Permission.VIEW_MEMBERS);
        permission.setSetBy("admin3");
        permissionsDAO.updatePermission(permission);

        permissions = permissionsDAO.findUserPermissions(USER_ID);
        assertEquals(1, permissions.size());
        assertEquals(Permission.VIEW_MEMBERS, permissions.get(0).getPermission());
        assertEquals("admin3", permissions.get(0).getSetBy());
        assertNotNull(permissions.get(0).getUpdateTime());

        assertTrue(permissionsDAO.findPermissionEntity(group.getId(), USER_ID).isPresent());

        permissionsDAO.deletePermission(permission.getGroupId(), permission.getUserId());

        permissions = permissionsDAO.findUserPermissions(USER_ID);
        assertTrue(permissions.isEmpty());
    }

    @Test
    public void testDeleteAllGroupsPermissions() {

        GroupEntity groupB = groupEntity("B");
        GroupEntity groupC = groupEntity("C");

        PermissionEntity permissionB = permissionEntity(groupB, USER_ID, Permission.ADMIN);
        PermissionEntity permissionC = permissionEntity(groupC, USER_ID, Permission.VIEW_MEMBERS);

        permissionsDAO.createOrUpdatePermission(permissionB);
        permissionsDAO.createOrUpdatePermission(permissionC);

        assertTrue(permissionsDAO.findPermissionEntity(groupB.getId(), USER_ID).isPresent());
        assertTrue(permissionsDAO.findPermissionEntity(groupC.getId(), USER_ID).isPresent());

        permissionsDAO.deleteAllGroupsPermissions(Arrays.asList(groupB.getId(), groupC.getId()));

        assertFalse(permissionsDAO.findPermissionEntity(groupB.getId(), USER_ID).isPresent());
        assertFalse(permissionsDAO.findPermissionEntity(groupC.getId(), USER_ID).isPresent());
    }

    @Test
    public void testDeleteAllGroupsPermissionsEmptyList() {
        permissionsDAO.deleteAllGroupsPermissions(new ArrayList<>());
    }

    private GroupEntity groupEntity(String groupId) {
        GroupEntity groupEntity = new GroupEntity();
        groupEntity.setId(groupId);
        groupEntity.setName(groupId);
        groupEntity.setPath(groupId);
        groupEntity.setLeaf(false);
        groupsDAO.createGroup(groupEntity);
        return groupEntity;
    }

    private PermissionEntity permissionEntity(GroupEntity group, String userId, Permission permission) {
        PermissionEntity permissionEntity = new PermissionEntity();
        permissionEntity.setGroupId(group.getId());
        permissionEntity.setUserId(userId);
        permissionEntity.setPermission(permission);
        permissionEntity.setSetBy("admin1");
        permissionEntity.setGroupPath(group.getPath());
        return permissionEntity;
    }
}
