package it.inaf.ia2.gms.service;

import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.persistence.JoinDAO;
import it.inaf.ia2.gms.persistence.MembershipsDAO;
import it.inaf.ia2.gms.persistence.PermissionsDAO;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.MembershipEntity;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import java.util.Arrays;
import java.util.Set;
import static org.junit.Assert.assertEquals;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentMatcher;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.ArgumentMatchers.eq;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.mockito.junit.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class JoinServiceTest {

    private static final String USER_1 = "user-1";
    private static final String USER_2 = "user-2";

    @Mock
    private MembershipsDAO membershipsDAO;

    @Mock
    private PermissionsDAO permissionsDAO;

    @Mock
    private JoinDAO joinDAO;

    @InjectMocks
    private JoinService joinService;

    @Test
    public void testJoin() {

        when(membershipsDAO.getUserMemberships(eq(USER_1)))
                .thenReturn(Arrays.asList(groupEntity("A"), groupEntity("C")));

        when(membershipsDAO.getUserMemberships(eq(USER_2)))
                .thenReturn(Arrays.asList(groupEntity("A"), groupEntity("B")));

        when(permissionsDAO.findUserPermissions(eq(USER_1)))
                .thenReturn(Arrays.asList(
                        permissionEntity("A", USER_1, Permission.ADMIN),
                        permissionEntity("B", USER_1, Permission.VIEW_MEMBERS),
                        permissionEntity("C", USER_1, Permission.MANAGE_MEMBERS)
                ));

        when(permissionsDAO.findUserPermissions(eq(USER_2)))
                .thenReturn(Arrays.asList(
                        permissionEntity("A", USER_2, Permission.VIEW_MEMBERS),
                        permissionEntity("B", USER_2, Permission.ADMIN),
                        permissionEntity("C", USER_2, Permission.MANAGE_MEMBERS),
                        permissionEntity("D", USER_2, Permission.VIEW_MEMBERS)
                ));

        String remainingUserId = joinService.join(USER_1, USER_2);

        verify(joinDAO, times(1)).join(argThat(memberships -> {
            assertEquals(1, memberships.size());
            MembershipEntity entity = memberships.iterator().next();
            assertEquals("B", entity.getGroupId());
            assertEquals(USER_1, entity.getUserId());
            return true;
        }),
                argThat(verifyPermissionsToAdd1()), eq(USER_2));

        assertEquals(USER_1, remainingUserId);
    }

    @Test
    public void testJoinReverseOrder() {

        when(membershipsDAO.getUserMemberships(eq(USER_1)))
                .thenReturn(Arrays.asList(groupEntity("C")));

        when(membershipsDAO.getUserMemberships(eq(USER_2)))
                .thenReturn(Arrays.asList(groupEntity("A"), groupEntity("B")));

        when(permissionsDAO.findUserPermissions(eq(USER_1)))
                .thenReturn(Arrays.asList(
                        permissionEntity("C", USER_1, Permission.ADMIN),
                        permissionEntity("D", USER_1, Permission.VIEW_MEMBERS)
                ));

        when(permissionsDAO.findUserPermissions(eq(USER_2)))
                .thenReturn(Arrays.asList(
                        permissionEntity("A", USER_2, Permission.ADMIN),
                        permissionEntity("B", USER_2, Permission.VIEW_MEMBERS),
                        permissionEntity("C", USER_2, Permission.MANAGE_MEMBERS)
                ));

        String remainingUserId = joinService.join(USER_1, USER_2);

        verify(joinDAO, times(1)).join(argThat(verifyMembershipsToAdd2()),
                argThat(verifyPermissionsToAdd2()), eq(USER_1));

        assertEquals(USER_2, remainingUserId);
    }

    private GroupEntity groupEntity(String groupId) {
        GroupEntity group = new GroupEntity();
        group.setId(groupId);
        return group;
    }

    private PermissionEntity permissionEntity(String groupId, String userId, Permission permission) {
        PermissionEntity permissionEntity = new PermissionEntity();
        permissionEntity.setGroupId(groupId);
        permissionEntity.setUserId(userId);
        permissionEntity.setPermission(permission);
        return permissionEntity;
    }

    private ArgumentMatcher<Set<MembershipEntity>> verifyMembershipsToAdd1() {
        return memberships -> {
            assertEquals(1, memberships.size());
            MembershipEntity entity = memberships.iterator().next();
            assertEquals("B", entity.getGroupId());
            assertEquals(USER_1, entity.getUserId());
            return true;
        };
    }

    private ArgumentMatcher<Set<PermissionEntity>> verifyPermissionsToAdd1() {
        return permissions -> {
            assertEquals(2, permissions.size());
            return permissions.contains(permissionEntity("B", USER_1, Permission.ADMIN))
                    && permissions.contains(permissionEntity("D", USER_1, Permission.VIEW_MEMBERS));
        };
    }

    private ArgumentMatcher<Set<MembershipEntity>> verifyMembershipsToAdd2() {
        return memberships -> {
            assertEquals(1, memberships.size());
            MembershipEntity entity = memberships.iterator().next();
            assertEquals("C", entity.getGroupId());
            assertEquals(USER_2, entity.getUserId());
            return true;
        };
    }

    private ArgumentMatcher<Set<PermissionEntity>> verifyPermissionsToAdd2() {
        return permissions -> {
            assertEquals(2, permissions.size());
            return permissions.contains(permissionEntity("C", USER_2, Permission.ADMIN))
                    && permissions.contains(permissionEntity("D", USER_2, Permission.VIEW_MEMBERS));
        };
    }
}
