package it.inaf.ia2.gms.controller;

import it.inaf.ia2.gms.manager.GroupsManager;
import it.inaf.ia2.gms.manager.MembershipManager;
import it.inaf.ia2.gms.manager.PermissionsManager;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.UserPermission;
import it.inaf.ia2.gms.persistence.GroupsDAO;
import it.inaf.ia2.gms.persistence.PermissionsDAO;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.MembershipEntity;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.gms.service.JoinService;
import it.inaf.ia2.rap.data.RapUser;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.ArgumentMatchers.eq;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.mockito.junit.MockitoJUnitRunner;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.delete;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.content;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

@RunWith(MockitoJUnitRunner.class)
public class JWTWebServiceControllerTest {

    @Mock
    private JoinService joinService;

    @Mock
    private GroupsDAO groupsDAO;

    @Mock
    private GroupsManager groupsManager;

    @Mock
    private GroupsService groupsService;

    @Mock
    private MembershipManager membershipManager;

    @Mock
    private PermissionsManager permissionsManager;

    @Mock
    private PermissionsDAO permissionsDAO;

    @InjectMocks
    private JWTWebServiceController controller;

    private final Principal principal = new Principal() {
        @Override
        public String getName() {
            return "TEST_PRINCIPAL";
        }
    };

    private MockMvc mockMvc;

    private GroupEntity root;
    private GroupEntity lbt;
    private GroupEntity inaf;

    @Before
    public void init() {
        mockMvc = MockMvcBuilders.standaloneSetup(controller).build();
        root = getRoot();
        lbt = getLbtGroup();
        inaf = getInafGroup();
    }

    @Test
    public void testIsMemberOf() throws Exception {

        GroupEntity group1 = new GroupEntity();
        group1.setId("1");
        group1.setPath("1");
        group1.setName("group.1");

        GroupEntity group2 = new GroupEntity();
        group2.setId("2");
        group2.setPath("1.2");
        group2.setName("subgroup");

        GroupEntity group3 = new GroupEntity();
        group3.setId("3");
        group3.setPath("1.2.3");
        group3.setName("subsubgroup");

        when(groupsDAO.findGroupByParentAndName(eq(""), eq("group.1")))
                .thenReturn(Optional.of(group1));
        when(groupsDAO.findGroupByParentAndName(eq("1"), eq("subgroup")))
                .thenReturn(Optional.of(group2));
        when(groupsDAO.findGroupByParentAndName(eq("1.2"), eq("subsubgroup")))
                .thenReturn(Optional.of(group3));

        when(membershipManager.isCurrentUserMemberOf(eq(group3.getId()))).thenReturn(true);

        String group = "group\\.1.subgroup.subsubgroup";

        mockMvc.perform(get("/ws/jwt/search/" + group).principal(principal))
                .andExpect(status().isOk())
                .andExpect(content().string(group + "\n"));
    }

    @Test
    public void testCreateGroup() throws Exception {

        when(groupsManager.getRoot()).thenReturn(root);
        when(groupsManager.createGroup(any(GroupEntity.class), eq("LBT"), eq(false))).thenReturn(lbt);
        when(groupsManager.createGroup(any(GroupEntity.class), eq("INAF"), eq(true))).thenReturn(inaf);

        mockMvc.perform(post("/ws/jwt/LBT.INAF")
                .param("leaf", "true")
                .contentType(MediaType.APPLICATION_FORM_URLENCODED))
                .andExpect(status().isCreated());

        verify(groupsService, times(2)).findGroupByParentAndName(any(GroupEntity.class), any());
        verify(groupsManager, times(1)).createGroup(argGroupIdEq(GroupsService.ROOT), eq("LBT"), eq(false));
        verify(groupsManager, times(1)).createGroup(argGroupIdEq("lbt_id"), eq("INAF"), eq(true));
    }

    private GroupEntity argGroupIdEq(String groupId) {
        return argThat(g -> g.getId().equals(groupId));
    }

    @Test
    public void testDeleteGroupByPath() throws Exception {

        when(groupsDAO.findGroupByParentAndName("", "LBT")).thenReturn(Optional.of(lbt));
        when(groupsDAO.findGroupByParentAndName("lbt_id", "INAF")).thenReturn(Optional.of(inaf));

        mockMvc.perform(delete("/ws/jwt/LBT.INAF"))
                .andExpect(status().isNoContent());

        verify(groupsDAO, times(1)).deleteGroup(eq(inaf));
    }

    @Test
    public void testAddMember() throws Exception {

        when(groupsDAO.findGroupByParentAndName("", "LBT")).thenReturn(Optional.of(lbt));
        when(groupsDAO.findGroupByParentAndName("lbt_id", "INAF")).thenReturn(Optional.of(inaf));

        String userId = "target_user";

        MembershipEntity membership = new MembershipEntity();
        membership.setGroupId(inaf.getId());
        membership.setUserId(userId);

        when(membershipManager.addMember(eq(inaf), eq(userId)))
                .thenReturn(membership);

        mockMvc.perform(post("/ws/jwt/membership/LBT.INAF")
                .param("user_id", userId)
                .contentType(MediaType.APPLICATION_FORM_URLENCODED))
                .andExpect(status().isOk());
    }

    @Test
    public void testRemoveMember() throws Exception {

        when(groupsDAO.findGroupByParentAndName("", "LBT")).thenReturn(Optional.of(lbt));
        when(groupsDAO.findGroupByParentAndName("lbt_id", "INAF")).thenReturn(Optional.of(inaf));

        mockMvc.perform(delete("/ws/jwt/membership/LBT.INAF?user_id=userId"))
                .andExpect(status().isNoContent());

        verify(membershipManager, times(1)).removeMember(eq(inaf), eq("userId"));
    }

    @Test
    public void testGetGroupPermissions() throws Exception {

        when(groupsDAO.findGroupByParentAndName("", "LBT")).thenReturn(Optional.of(lbt));
        when(groupsDAO.findGroupByParentAndName("lbt_id", "INAF")).thenReturn(Optional.of(inaf));

        List<UserPermission> permissions = new ArrayList<>();
        UserPermission up = new UserPermission();
        up.setUser(getRapUser());
        up.setPermission(Permission.ADMIN);
        permissions.add(up);
        when(permissionsManager.getAllPermissions(any())).thenReturn(permissions);

        mockMvc.perform(get("/ws/jwt/permission/LBT.INAF").principal(principal))
                .andExpect(status().isOk())
                .andExpect(content().string("rap_user ADMIN\n"));
    }

    @Test
    public void testAddPermission() throws Exception {

        String userId = "target_user";
        Permission permission = Permission.ADMIN;

        when(groupsDAO.findGroupByParentAndName("", "LBT")).thenReturn(Optional.of(lbt));
        when(groupsDAO.findGroupByParentAndName("lbt_id", "INAF")).thenReturn(Optional.of(inaf));

        PermissionEntity permissionEntity = new PermissionEntity();
        permissionEntity.setGroupId(inaf.getId());
        permissionEntity.setUserId(userId);
        permissionEntity.setPermission(permission);
        permissionEntity.setGroupPath(inaf.getPath());

        when(permissionsManager.addPermission(eq(inaf), eq(userId),
                eq(permission))).thenReturn(permissionEntity);

        mockMvc.perform(post("/ws/jwt/permission/LBT.INAF")
                .param("user_id", userId)
                .param("permission", permission.toString())
                .contentType(MediaType.APPLICATION_FORM_URLENCODED))
                .andExpect(status().isOk());

        verify(permissionsManager, times(1))
                .addPermission(eq(inaf), eq(userId), eq(permission));
    }

    @Test
    public void testRemovePermission() throws Exception {

        List<String> names = Arrays.asList("LBT", "INAF");

        //when(groupsService.findGroupByNames(names)).thenReturn(Optional.of(inaf));
        when(groupsDAO.findGroupByParentAndName("", "LBT")).thenReturn(Optional.of(lbt));
        when(groupsDAO.findGroupByParentAndName("lbt_id", "INAF")).thenReturn(Optional.of(inaf));

        mockMvc.perform(delete("/ws/jwt/permission/LBT.INAF?user_id=userId&permission=ADMIN"))
                .andExpect(status().isNoContent());

        verify(permissionsManager, times(1)).removePermission(eq(inaf), eq("userId"));
    }

    private GroupEntity getRoot() {
        GroupEntity root = new GroupEntity();
        root.setId(GroupsService.ROOT);
        root.setName(GroupsService.ROOT);
        root.setPath("");
        return root;
    }

    private GroupEntity getLbtGroup() {
        GroupEntity lbt = new GroupEntity();
        lbt.setId("lbt_id");
        lbt.setName("LBT");
        lbt.setPath("lbt_id");
        return lbt;
    }

    private GroupEntity getInafGroup() {
        GroupEntity inaf = new GroupEntity();
        inaf.setId("inaf_id");
        inaf.setName("INAF");
        inaf.setPath("lbt_id.inaf_id");
        return inaf;
    }

    private RapUser getRapUser() {
        RapUser user = new RapUser();
        user.setId("rap_user");
        return user;
    }
}
