const BASE_API_URL = process.env.VUE_APP_API_BASE_URL;

function apiRequest(url, options, showLoading = true) {
  if (showLoading) {
    loading(true);
  }
  return new Promise((resolve) => {
    fetch(url, options)
      .then(response => {
        loading(false);
        if ([200, 201, 204, 400].includes(response.status)) { // valid status codes
          if (response.status === 204) {
            resolve({});
          } else {
            resolve(response.json());
          }
        } else {
          response.json().then(jsonValue => dispatchApiErrorEvent(jsonValue));
        }
      })
      .catch(error => {
        loading(false);
        dispatchApiErrorEvent(error);
      });
  });
}

function dispatchApiErrorEvent(error) {
  let message;
  if (error.message) {
    message = error.message;
  } else {
    message = 'Generic error';
  }

  let event = new CustomEvent('apiError');
  event.message = message;
  document.dispatchEvent(event);
}

/* For loading animation */
function loading(value) {
  let event = new CustomEvent('loading');
  event.value = value;
  document.dispatchEvent(event);
}

export default {
  fetchHomePageModel(input) {
    let url = BASE_API_URL +
      'home?groupId=' + input.selectedGroupId +
      '&paginatorPageSize=' + input.paginatorPageSize +
      '&paginatorPage=' + input.paginatorPage;
    return apiRequest(url, {
      method: 'GET',
      cache: 'no-cache',
      credentials: 'include',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      }
    });
  },
  fetchGroupsTab(input) {
    let url = BASE_API_URL +
      'groups?groupId=' + input.selectedGroupId +
      '&paginatorPageSize=' + input.paginatorPageSize +
      '&paginatorPage=' + input.paginatorPage +
      '&onlyPanel=false';
    if (input.searchFilter !== null) {
      url += '&searchFilter=' + input.searchFilter;
    }
    return apiRequest(url, {
      method: 'GET',
      cache: 'no-cache',
      credentials: 'include',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      }
    });
  },
  fetchGroupsPanel(input) {
    let url = BASE_API_URL +
      'groups?groupId=' + input.selectedGroupId +
      '&paginatorPageSize=' + input.paginatorPageSize +
      '&paginatorPage=' + input.paginatorPage +
      '&onlyPanel=true';
    if (input.searchFilter !== null) {
      url += '&searchFilter=' + input.searchFilter;
    }
    return apiRequest(url, {
      method: 'GET',
      cache: 'no-cache',
      credentials: 'include',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      }
    });
  },
  fetchMembersPanel(input) {
    let url = BASE_API_URL +
      'members?groupId=' + input.selectedGroupId +
      '&paginatorPageSize=' + input.paginatorPageSize +
      '&paginatorPage=' + input.paginatorPage;
    return apiRequest(url, {
      method: 'GET',
      cache: 'no-cache',
      credentials: 'include',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      }
    });
  },
  fetchPermissionsPanel(input) {
    let url = BASE_API_URL +
      'permissions?groupId=' + input.selectedGroupId +
      '&paginatorPageSize=' + input.paginatorPageSize +
      '&paginatorPage=' + input.paginatorPage;
    return apiRequest(url, {
      method: 'GET',
      cache: 'no-cache',
      credentials: 'include',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      }
    });
  },
  addGroup(newGroupName, leaf, input) {
    let url = BASE_API_URL + 'group';
    return apiRequest(url, {
      method: 'POST',
      cache: 'no-cache',
      credentials: 'include',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      },
      body: JSON.stringify({
        newGroupName: newGroupName,
        parentGroupId: input.selectedGroupId,
        paginatorPageSize: input.paginatorPageSize,
        paginatorPage: input.paginatorPage,
        searchFilter: input.searchFilter,
        leaf: leaf
      })
    });
  },
  renameGroup(groupId, newGroupName, leaf, input) {
    let url = BASE_API_URL + 'group/' + groupId;
    return apiRequest(url, {
      method: 'PUT',
      cache: 'no-cache',
      credentials: 'include',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      },
      body: JSON.stringify({
        newGroupName: newGroupName,
        leaf: leaf,
        paginatorPageSize: input.paginatorPageSize,
        paginatorPage: input.paginatorPage,
        searchFilter: input.searchFilter
      })
    });
  },
  removeGroup(groupId, input) {
    let url = BASE_API_URL + 'group/' + groupId +
      '?paginatorPageSize=' + input.paginatorPageSize +
      '&paginatorPage=' + input.paginatorPage;
    if (input.searchFilter !== null) {
      url += '&searchFilter=' + input.searchFilter;
    }
    return apiRequest(url, {
      method: 'DELETE',
      cache: 'no-cache',
      credentials: 'include',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      }
    });
  },
  searchUser(searchInput) {
    let url = BASE_API_URL + 'users?search=' + searchInput;

    return apiRequest(url, {
      method: 'GET',
      cache: 'no-cache',
      credentials: 'include',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      }
    });
  },
  addPermission(userId, permission, input) {
    let url = BASE_API_URL + 'permission';

    return apiRequest(url, {
      method: 'POST',
      cache: 'no-cache',
      credentials: 'include',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      },
      body: JSON.stringify({
        groupId: input.selectedGroupId,
        userId: userId,
        permission: permission,
        paginatorPageSize: input.paginatorPageSize,
        paginatorPage: input.paginatorPage
      })
    });
  },
  getPermission(groupId, userId) {
    let url = BASE_API_URL + 'permission?groupId=' + groupId + '&userId=' + userId;

    return apiRequest(url, {
      method: 'GET',
      cache: 'no-cache',
      credentials: 'include',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      }
    });
  },
  addMember(userId, permission, input) {
    let url = BASE_API_URL + 'member';

    return apiRequest(url, {
      method: 'POST',
      cache: 'no-cache',
      credentials: 'include',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      },
      body: JSON.stringify({
        groupId: input.selectedGroupId,
        userId: userId,
        permission: permission,
        paginatorPageSize: input.paginatorPageSize,
        paginatorPage: input.paginatorPage
      })
    });
  },
  removeMember(userId, removeAlsoPermission, input) {
    let url = BASE_API_URL + 'member' +
      '?groupId=' + input.selectedGroupId +
      '&userId=' + userId +
      '&removeAlsoPermission=' + removeAlsoPermission +
      '&paginatorPageSize=' + input.paginatorPageSize +
      '&paginatorPage=' + input.paginatorPage;
    return apiRequest(url, {
      method: 'DELETE',
      cache: 'no-cache',
      credentials: 'include',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      }
    });
  },
  removePermission(userId, input) {
    let url = BASE_API_URL + 'permission' +
      '?groupId=' + input.selectedGroupId +
      '&userId=' + userId +
      '&paginatorPageSize=' + input.paginatorPageSize +
      '&paginatorPage=' + input.paginatorPage;
    return apiRequest(url, {
      method: 'DELETE',
      cache: 'no-cache',
      credentials: 'include',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      }
    });
  },
  search(input) {
    let url = BASE_API_URL + 'search?query=' + input.genericSearch.filter +
      '&page=' + input.genericSearch.paginatorPage + '&pageSize=' + input.genericSearch.paginatorPageSize;

    return apiRequest(url, {
      method: 'GET',
      cache: 'no-cache',
      credentials: 'include',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      }
    });
  },
  openUserSearchResult(userId) {
    let url = BASE_API_URL + 'search/user/' + userId;

    return apiRequest(url, {
      method: 'GET',
      cache: 'no-cache',
      credentials: 'include',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      }
    });
  },
  keepAlive() {
    let url = BASE_API_URL + 'keepAlive';

    return apiRequest(url, {
      method: 'GET',
      cache: 'no-cache',
      credentials: 'include'
    }, false);
  }
};
