package it.inaf.ia2.gms.service;

import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.RapUser;
import it.inaf.ia2.gms.model.UserPermission;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import it.inaf.ia2.gms.persistence.PermissionsDAO;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.rap.RapClient;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class PermissionsService {

    private final PermissionsDAO permissionsDAO;
    private final RapClient rapClient;

    @Autowired
    public PermissionsService(PermissionsDAO permissionsDAO, RapClient rapClient) {
        this.permissionsDAO = permissionsDAO;
        this.rapClient = rapClient;
    }

    public List<UserPermission> getAllPermissions(GroupEntity group) {

        List<PermissionEntity> permissions = permissionsDAO.getGroupsPermissions(group.getId());

        Set<String> userIdentifiers = permissions.stream()
                .map(p -> p.getUserId())
                .collect(Collectors.toSet());

        Map<String, RapUser> users = rapClient.getUsers(userIdentifiers).stream()
                .collect(Collectors.toMap(RapUser::getId, Function.identity()));

        List<UserPermission> result = new ArrayList<>();

        for (PermissionEntity p : permissions) {
            RapUser rapUser = users.get(p.getUserId());
            if (rapUser != null) {
                UserPermission permission = new UserPermission();
                permission.setPermission(p.getPermission());
                permission.setUser(rapUser);
                result.add(permission);
            }
        }

        return result;
    }

    public Permission getUserPermissionForGroup(GroupEntity group, String userId) {
        List<PermissionEntity> permissions = permissionsDAO.findUserPermissions(userId, group.getPath());
        return PermissionUtils.getGroupPermission(group, permissions).orElse(null);
    }

    public void removePermission(GroupEntity group, String userId) {
        permissionsDAO.deletePermission(group.getId(), userId);
    }

    public PermissionEntity addPermission(GroupEntity group, String userId, Permission permission) {

        Optional<PermissionEntity> existingPermissionEntity = permissionsDAO.findPermissionEntity(group.getId(), userId);

        PermissionEntity permissionEntity;
        if (existingPermissionEntity.isPresent()) {
            permissionEntity = existingPermissionEntity.get();
            Permission resultingPermission = Permission.addPermission(permissionEntity.getPermission(), permission);
            if (resultingPermission == permissionEntity.getPermission()) {
                return permissionEntity;
            } else {
                permissionEntity.setPermission(resultingPermission);
            }
        } else {
            permissionEntity = new PermissionEntity();
            permissionEntity.setGroupId(group.getId());
            permissionEntity.setUserId(userId);
            permissionEntity.setPermission(permission);
            permissionEntity.setGroupPath(group.getPath());
        }

        return permissionsDAO.createOrUpdatePermission(permissionEntity);
    }
}
