#include <ProtocolManager.h>

#include <boost/date_time.hpp>

namespace MetadataExporter_ns
{

//==============================================================================
//      ProtocolManager::ProtocolManager()
//==============================================================================
ProtocolManager::ProtocolManager(Tango::DeviceImpl* deviceImpl_p,
    Configuration::SP configuration_sp, DBManager::SP dBManager_sp) :
    Tango::LogAdapter(deviceImpl_p), m_configuration_sp(configuration_sp),
    m_dBManager_sp(dBManager_sp)
{
    DEBUG_STREAM << "ProtocolManager::ProtocolManager()" << endl;

    m_isAuthorised = false;
    m_isValidated = false;
}

//==============================================================================
//      ProtocolManager::~ProtocolManager()
//==============================================================================
ProtocolManager::~ProtocolManager()
{
    DEBUG_STREAM << "ProtocolManager::~ProtocolManager()" << endl;
}

//==============================================================================
//      ProtocolManager::create()
//==============================================================================
ProtocolManager::SP ProtocolManager::create(Tango::DeviceImpl* deviceImpl_p,
    Configuration::SP configuration_sp, DBManager::SP dBManager_sp)
{
    ProtocolManager::SP d_sp(new ProtocolManager(deviceImpl_p, configuration_sp,
        dBManager_sp), ProtocolManager::Deleter());

    return d_sp;
}

//==============================================================================
//      ProtocolManager::setRemoteEndpoint()
//==============================================================================
void ProtocolManager::setRemoteEndpoint(std::string remoteEndpoint)
{
    DEBUG_STREAM << "ProtocolManager::setRemoteEndpoint()" << endl;

    m_remoteEndpoint = remoteEndpoint;
}

//==============================================================================
//      ProtocolManager::prepareResponse()
//==============================================================================
ResponseSP ProtocolManager::prepareResponse(RequestSP request_sp)
    throw(std::runtime_error)
{
    DEBUG_STREAM << "ProtocolManager::prepareResponse()" << endl;

    if(!request_sp->IsInitialized())
        throw std::runtime_error("Not initialized request!");

    ResponseSP response_sp;

    switch(request_sp->type())
    {
        case Request::AUTHORIZATION:
        {
            response_sp = prepareAuthroisation(request_sp);
            break;
        }
        case Request::VALIDATION:
        {
            response_sp = prepareValidation(request_sp);
            break;
        }
        case Request::METADATA:
        {
            response_sp = prepareMetadata(request_sp);
            break;
        }
        default:
            throw std::runtime_error("Unknown request type!");
    }

    if(!response_sp->IsInitialized())
        throw std::runtime_error("Not initialized response!");

    return response_sp;
}

//==============================================================================
//      ProtocolManager::prepareAuthroisation()
//==============================================================================
ResponseSP ProtocolManager::prepareAuthroisation(RequestSP request_sp)
{
    DEBUG_STREAM << "ProtocolManager::prepareAuthroisation()" << endl;

    ResponseSP response_sp(new Response());

    response_sp->set_type(Response::AUTHORIZATION);

    Response::Authorization* authResp = response_sp->mutable_authorization();

    if(!m_isAuthorised)
    {
        const Request::Authorization& authReq = request_sp->authorization();
        std::string username =  authReq.username();
        std::string password = authReq.password();

        if(m_configuration_sp->isUserAuthorized(username, password))
        {
            INFO_STREAM << "ProtocolManager::prepareAuthroisation() "
                << "Authorization accepted from " << m_remoteEndpoint << endl;

            m_isAuthorised = true;

            authResp->set_state(Response::Authorization::ACCEPTED);
            authResp->set_status("Authorization accepted");
        }
        else
        {
            WARN_STREAM << "ProtocolManager::prepareAuthroisation() "
                << "Invalid username or password from " << m_remoteEndpoint << endl;

            m_isAuthorised = false;

            authResp->set_state(Response::Authorization::REJECTED);
            authResp->set_status("Invalid username or password");
        }
    }
    else
    {
        WARN_STREAM << "ProtocolManager::prepareAuthroisation() "
            << "Already authorized from " << m_remoteEndpoint << endl;

        authResp->set_state(Response::Authorization::REJECTED);
        authResp->set_status("Already authorized");
    }

    return response_sp;
}

//==============================================================================
//      ProtocolManager::prepareValidation()
//==============================================================================
ResponseSP ProtocolManager::prepareValidation(RequestSP request_sp)
{
    DEBUG_STREAM << "ProtocolManager::prepareValidation()" << endl;

    ResponseSP response_sp(new Response());

    response_sp->set_type(Response::VALIDATION);

    Response::Validation* validationRes = response_sp->mutable_validation();

    if(m_isAuthorised)
    {
        if(!m_isValidated)
        {
            const Request::Validation& validationReq = request_sp->validation();

            m_validatedSchema = validationReq.schema();
            m_validatedTable =  validationReq.table();

            if(m_configuration_sp->isTableExported(m_validatedSchema, m_validatedTable))
            {
                try
                {
                    DBManager::InformationList informationList =
                        m_dBManager_sp->retrieveInformation(m_validatedSchema, m_validatedTable);

                    if(informationList.empty())
                    {
                        std::stringstream errorStream;
                        errorStream << "Table " << m_validatedSchema << "."
                            << m_validatedTable << " not exists";
                        throw std::runtime_error(errorStream.str());
                    }

                    if(validationReq.columns_size() != (int)informationList.size())
                    {
                        std::stringstream errorStream;
                        errorStream << "Table " << m_validatedSchema << "."
                            << m_validatedTable << " has different columns size";
                        throw std::runtime_error(errorStream.str());
                    }

                    for(int i=0; i<validationReq.columns_size(); ++i)
                        validateColumn(validationReq.columns(i), informationList);

                    INFO_STREAM << "ProtocolManager::prepareValidation() "
                        << "Validation accepted for " << m_validatedSchema << "."
                        << m_validatedTable << " from " << m_remoteEndpoint << endl;

                    m_isValidated = true;

                    validationRes->set_state(Response::Validation::ACCEPTED);
                    validationRes->set_status("Table validated");
                }
                catch(std::runtime_error& ex)
                {
                    WARN_STREAM << "ProtocolManager::prepareValidation() "
                        << ex.what() << " from " << m_remoteEndpoint << endl;

                    validationRes->set_state(Response::Validation::REJECTED);
                    validationRes->set_status(ex.what());
                }
            }
            else
            {
                WARN_STREAM << "ProtocolManager::prepareValidation() "
                    << "Not exported table from " << m_remoteEndpoint << endl;

                validationRes->set_state(Response::Validation::REJECTED);
                validationRes->set_status("Not exported table");
            }
        }
        else
        {
            WARN_STREAM << "ProtocolManager::prepareValidation() "
                << "Already validated from " << m_remoteEndpoint << endl;

            validationRes->set_state(Response::Validation::REJECTED);
            validationRes->set_status("Already validated");
        }
    }
    else
    {
        WARN_STREAM << "ProtocolManager::prepareValidation() "
            << "Not authorised from " << m_remoteEndpoint << endl;

        validationRes->set_state(Response::Validation::REJECTED);
        validationRes->set_status("Not authorised");
    }

    return response_sp;
}

//==============================================================================
//      ProtocolManager::prepareMetadata()
//==============================================================================
ResponseSP ProtocolManager::prepareMetadata(RequestSP request_sp)
{
    DEBUG_STREAM << "ProtocolManager::prepareMetadata()" << endl;

    ResponseSP response_sp(new Response());

    response_sp->set_type(Response::METADATA);

    Response::Metadata* metadataRes = response_sp->mutable_metadata();

    if(m_isAuthorised)
    {
        if(m_isValidated)
        {
            const Request::Metadata& metadataReq = request_sp->metadata();

            int64_t rawTimestamp = metadataReq.timestamp();
            std::tm tmTimestamp = *localtime(&rawTimestamp);

            try
            {
                boost::posix_time::ptime ptTimestamp =
                    boost::posix_time::ptime_from_tm(tmTimestamp);

                INFO_STREAM << "ProtocolManager::prepareMetadata() Searching in "
                    << m_validatedSchema << "." << m_validatedTable << " timestamp "
                    << boost::posix_time::to_simple_string(ptTimestamp)
                    << " from " << m_remoteEndpoint << endl;

                if(!m_rowSet_sp)
                {
                    m_rowSet_sp = m_dBManager_sp->retrieveNewTuples(
                        m_validatedSchema, m_validatedTable, tmTimestamp);

                    m_it = m_rowSet_sp->begin();
                }

                if(m_rowSet_sp && m_it != m_rowSet_sp->end())
                {
                    metadataRes->set_state(Response::Metadata::ACCEPTED);
                    metadataRes->set_status("Data ready");

                    fillMetadata(metadataRes);
                }
                else
                {
                    metadataRes->set_state(Response::Metadata::ACCEPTED);
                    metadataRes->set_status("No more data");

                    m_rowSet_sp.reset();
                }
            }
            catch(std::exception& ex)
            {
                WARN_STREAM << "ProtocolManager::prepareMetadata() "
                    << ex.what() << " from " << m_remoteEndpoint << endl;

                metadataRes->set_state(Response::Metadata::REJECTED);
                metadataRes->set_status(ex.what());

                m_rowSet_sp.reset();
            }
        }
        else
        {
            WARN_STREAM << "ProtocolManager::prepareMetadata() "
                << "Not validated from " << m_remoteEndpoint << endl;

            metadataRes->set_state(Response::Metadata::REJECTED);
            metadataRes->set_status("Not validated");
        }
    }
    else
    {
        WARN_STREAM << "ProtocolManager::prepareMetadata() "
            << "Not authorised from " << m_remoteEndpoint << endl;

        metadataRes->set_state(Response::Metadata::REJECTED);
        metadataRes->set_status("Not authorised");
    }

    return response_sp;
}

//==============================================================================
//      ProtocolManager::validateColumn()
//==============================================================================
void ProtocolManager::validateColumn(const Request::Validation::Column& column,
    DBManager::InformationList& informationList) throw(std::runtime_error)
{
    DEBUG_STREAM << "ProtocolManager::validateColumn()" << endl;

    bool found = false;

    DBManager::InformationList::const_iterator it;
    for(it=informationList.begin(); it!=informationList.end(); ++it)
    {
        if(!it->get<0>())
            throw std::runtime_error("Empty column name in information schema");
        std::string columnName = it->get<0>().get();

        if(column.name().compare(columnName)==0)
        {
            found = true;

            if(!it->get<1>())
                throw std::runtime_error("Empty column type in information schema");
            std::string columnType = it->get<1>().get();

            if(column.type().compare(columnType)!=0)
            {
                std::stringstream errorStream;
                errorStream << "Column " << column.name() << " type error "
                    << "server " << columnType << " client " << column.type();
                throw std::runtime_error(errorStream.str());
            }

            if(!it->get<2>())
                throw std::runtime_error("Empty column nullable in information schema");
            std::string isNullable = it->get<2>().get();

            if(column.nullable().compare(isNullable)!=0)
            {
                std::stringstream errorStream;
                errorStream << "Column " << column.name() << " nullable error "
                    << "server " << isNullable << " client " << column.nullable();
                throw std::runtime_error(errorStream.str());
            }

            #ifdef VERBOSE_DEBUG
                INFO_STREAM << "ProtocolManager::validateColumn() "
                    << column.name() << "<->" << columnName << " "
                    << column.type() << "<->" << columnType << " "
                    << column.nullable() << "<->" << isNullable << endl;
            #endif
        }
    }

    if(!found)
    {
        std::stringstream errorStream;
        errorStream << "Column " << column.name() << " not found";
        throw std::runtime_error(errorStream.str());
    }
}

//==============================================================================
//      ProtocolManager::fillMetadata()
//==============================================================================
void ProtocolManager::fillMetadata(Response::Metadata* metadataRes)
    throw(std::runtime_error, std::out_of_range)
{
    DEBUG_STREAM << "ProtocolManager::fillMetadata()" << endl;

    boost::posix_time::ptime ptCurrent;

    while(m_it != m_rowSet_sp->end())
    {
        int id = m_it->get<int>("id");
        int file_version = m_it->get<int>("file_version");
        std::string file_name = m_it->get<std::string>("file_name");
        std::tm tmNew = m_it->get<std::tm>("update_time");

        boost::posix_time::ptime ptNew = boost::posix_time::ptime_from_tm(tmNew);

        if(ptCurrent.is_not_a_date_time())
            ptCurrent = ptNew;

        INFO_STREAM << "ProtocolManager::fillMetadata() sending "
            << id << " " << file_version << " " << file_name << " "
            << boost::posix_time::to_simple_string(ptNew) << endl;

        if(ptNew > ptCurrent)
        {
            INFO_STREAM << "ProtocolManager::fillMetadata() stop ["
                << boost::posix_time::to_simple_string(ptNew) << " > "
                << boost::posix_time::to_simple_string(ptCurrent) << "]" << endl;
            break;
        }

        Response::Metadata::Row* row = metadataRes->add_rows();
        fillRow(row);

        m_it++;
    }

    if(m_it == m_rowSet_sp->end())
    {
        INFO_STREAM << "ProtocolManager::fillMetadata() all data sent" << endl;
        m_rowSet_sp.reset();
    }
}

//==============================================================================
//      ProtocolManager::fillRow()
//==============================================================================
void ProtocolManager::fillRow(Response::Metadata::Row* row)
    throw(std::runtime_error, std::out_of_range)
{
    DEBUG_STREAM << "ProtocolManager::fillRow()" << endl;

    for(std::size_t i=0; i!=m_it->size(); ++i)
    {
        const soci::column_properties& props = m_it->get_properties(i);

        std::string name = props.get_name();

        if(name.compare("id")==0 || 
        	name.compare("storage_path")==0 || 
		name.compare("file_path")==0)
        {
            #ifdef VERBOSE_DEBUG
                INFO_STREAM << "ProtocolManager::fillRow() skipping " << name << endl;
            #endif
            continue;
        }

        if(m_it->get_indicator(i) == soci::i_null)
        {
            #ifdef VERBOSE_DEBUG
                INFO_STREAM << "ProtocolManager::fillRow() " << name
                    << " null" << endl;
            #endif
            continue;
        }

        switch(props.get_data_type())
        {
            case soci::dt_string:
            {
                std::string value = m_it->get<std::string>(i);

                #ifdef VERBOSE_DEBUG
                    INFO_STREAM << "ProtocolManager::fillRow() " << name
                        << " " << value << endl;
                #endif

                Response::Metadata::Row::DtString* dtString = row->add_strings_list();
                dtString->set_key(name);
                dtString->set_value(value);

                break;
            }
            case soci::dt_double:
            {
                double value = m_it->get<double>(i);

                #ifdef VERBOSE_DEBUG
                    INFO_STREAM << "ProtocolManager::fillRow() " << name
                        << " " << value << endl;
                #endif

                Response::Metadata::Row::DtDouble* dtDouble = row->add_double_list();
                dtDouble->set_key(name);
                dtDouble->set_value(value);

                break;
            }
            case soci::dt_integer:
            {
                int value = m_it->get<int>(i);

                #ifdef VERBOSE_DEBUG
                    INFO_STREAM << "ProtocolManager::fillRow() " << name
                        << " " << value << endl;
                #endif

                Response::Metadata::Row::DtInteger* dtInteger = row->add_integer_list();
                dtInteger->set_key(name);
                dtInteger->set_value(value);

                break;
            }
            case soci::dt_long_long:
            {
                long long value = m_it->get<long long>(i);

                #ifdef VERBOSE_DEBUG
                    INFO_STREAM << "ProtocolManager::fillRow() " << name
                        << " " << value << endl;
                #endif

                Response::Metadata::Row::DtLongLong* dtLongLong = row->add_long_long_list();
                dtLongLong->set_key(name);
                dtLongLong->set_value(value);

                break;
            }
            case soci::dt_unsigned_long_long:
            {
                unsigned long value = m_it->get<unsigned long>(i);

                #ifdef VERBOSE_DEBUG
                    INFO_STREAM << "ProtocolManager::fillRow() " << name
                        << " " << value << endl;
                #endif

                Response::Metadata::Row::DtUnsignedLong* dtUnsignedLong = row->add_unsinged_long_list();
                dtUnsignedLong->set_key(name);
                dtUnsignedLong->set_value(value);

                break;
            }
            case soci::dt_date:
            {
                std::tm tmValue = m_it->get<std::tm>(i);
                boost::posix_time::ptime ptValue = boost::posix_time::ptime_from_tm(tmValue);

                #ifdef VERBOSE_DEBUG
                    INFO_STREAM << "ProtocolManager::fillRow() " << name << " "
                        << boost::posix_time::to_simple_string(ptValue) << endl;
                #endif

                Response::Metadata::Row::DtDate* dtDate = row->add_date_list();
                dtDate->set_key(name);
                dtDate->set_value(mktime(&tmValue));

                break;
            }
            default:
                throw std::runtime_error("Unknown data type");
        }
    }
}

}   //namespace
