#include <ProtocolManager.h>

namespace MetadataExporter_ns
{

//==============================================================================
//      ProtocolManager::ProtocolManager()
//==============================================================================
ProtocolManager::ProtocolManager(Tango::DeviceImpl* deviceImpl_p,
    Configuration::SP configuration_sp, DBManager::SP dBManager_sp) :
    Tango::LogAdapter(deviceImpl_p), m_configuration_sp(configuration_sp),
    m_dBManager_sp(dBManager_sp)
{
    DEBUG_STREAM << "ProtocolManager::ProtocolManager()" << endl;

    m_isAuthenticated = false;
    m_isValidated = false;
}

//==============================================================================
//      ProtocolManager::ProtocolManager()
//==============================================================================
ProtocolManager::~ProtocolManager()
{
    DEBUG_STREAM << "ProtocolManager::~ProtocolManager()" << endl;
}

//==============================================================================
//      ProtocolManager::ProtocolManager()
//==============================================================================
ProtocolManager::SP ProtocolManager::create(Tango::DeviceImpl* deviceImpl_p,
    Configuration::SP configuration_sp, DBManager::SP dBManager_sp)
{
    ProtocolManager::SP d_sp(new ProtocolManager(deviceImpl_p, configuration_sp,
        dBManager_sp), ProtocolManager::Deleter());

    return d_sp;
}

//==============================================================================
//      ProtocolManager::prepareResponse()
//==============================================================================
ResponseSP ProtocolManager::prepareResponse(RequestSP request_sp)
    throw(std::runtime_error)
{
    DEBUG_STREAM << "ProtocolManager::prepareResponse()" << endl;

    ResponseSP response_sp;

    switch(request_sp->type())
    {
        case Request::AUTHORIZATION:
        {
            response_sp = prepareAuthroisation(request_sp);
            break;
        }
        case Request::VALIDATION:
        {
            response_sp = prepareValidation(request_sp);
            break;
        }
        case Request::METADATA:
        {
            response_sp = prepareMetadata(request_sp);
            break;
        }
        default:
            throw std::runtime_error("Unknown request type!");
    }

    if(!response_sp->IsInitialized())
        throw std::runtime_error("Not initialized response!");

    return response_sp;
}

//==============================================================================
//      ProtocolManager::prepareAuthroisation()
//==============================================================================
ResponseSP ProtocolManager::prepareAuthroisation(RequestSP request_sp)
    throw(std::runtime_error)
{
    DEBUG_STREAM << "ProtocolManager::prepareAuthroisation()" << endl;

    ResponseSP response_sp(new Response());

    response_sp->set_type(Response::AUTHORIZATION);

    Response::Authorization* auth_resp = response_sp->mutable_authorization();

    if(!m_isAuthenticated)
    {
        const Request::Authorization& auth_req = request_sp->authorization();
        std::string username =  auth_req.username();
        std::string password = auth_req.password();

        if(m_configuration_sp->isUserAuthorized(username, password))
        {
            INFO_STREAM << "ProtocolManager::prepareAuthroisation() "
                << "Authorization accepted" << endl;

            m_isAuthenticated = true;

            auth_resp->set_state(Response::Authorization::ACCEPTED);
            auth_resp->set_status("Authorization accepted");
        }
        else
        {
            WARN_STREAM << "ProtocolManager::prepareAuthroisation() "
                << "Invalid username or password" << endl;

            m_isAuthenticated = false;

            auth_resp->set_state(Response::Authorization::REJECTED);
            auth_resp->set_status("Invalid username or password");
        }
    }
    else
    {
        WARN_STREAM << "ProtocolManager::prepareAuthroisation() "
            << "Already authorized" << endl;

        auth_resp->set_state(Response::Authorization::REJECTED);
        auth_resp->set_status("Already authorized");
    }

    return response_sp;
}

//==============================================================================
//      ProtocolManager::prepareValidation()
//==============================================================================
ResponseSP ProtocolManager::prepareValidation(RequestSP request_sp)
    throw(std::runtime_error)
{
    DEBUG_STREAM << "ProtocolManager::prepareValidation()" << endl;

    ResponseSP response_sp(new Response());

    response_sp->set_type(Response::VALIDATION);

    Response::Validation* validation = response_sp->mutable_validation();

    if(!m_isValidated)
    {
        INFO_STREAM << "ProtocolManager::prepareValidation() "
            << "Validation accepted" << endl;

        m_isValidated = true;

        validation->set_state(Response::Validation::ACCEPTED);
        validation->set_status("Validation accepted");
    }
    else
    {
        WARN_STREAM << "ProtocolManager::prepareValidation() "
            << "Already validated" << endl;

        validation->set_state(Response::Validation::REJECTED);
        validation->set_status("Already validated");
    }

    return response_sp;
}

//==============================================================================
//      ProtocolManager::prepareMetadata()
//==============================================================================
ResponseSP ProtocolManager::prepareMetadata(RequestSP request_sp)
    throw(std::runtime_error)
{
    DEBUG_STREAM << "ProtocolManager::prepareMetadata()" << endl;

    ResponseSP response_sp(new Response());

    response_sp->set_type(Response::METADATA);

    Response::Metadata* metadata = response_sp->mutable_metadata();

    metadata->set_state(Response::Metadata::ACCEPTED);
    metadata->set_status("Metadata ready");
    metadata->set_partial(1);
    metadata->set_total(1);

    return response_sp;
}

}   //namespace
